T.namespace('Foremka');
/**
 * Base Class for all objects 
 * (generates unique ids for all objects - static cntr)
 */

Foremka.Base = (function() {

    var ctr = 0;

    /**
     * Creates class internal constructor
     */
    var clsFactory = function() {
        var fn = function() {
            if (arguments[0] !== self)
                this._constr.apply(this, arguments);
        };
        return fn;
    };

    /**
     * Create Base class
     */
    var self = clsFactory();
   
    /**
     * Base Class Constructor
     */
    self.prototype._constr = function(name) {
        this.uniqeId = self.getNextUniqeId();

        this._name = name;
        this._annotateInstance();
    };
    
    /**
     * Generates next uniqe id
     */
    self.getNextUniqeId = function() {
        return ++ctr;
    };
    
    /**
     * Base class uniqeId
     */
    self.uniqeId = self.getNextUniqeId();

    /**
     * Creates Class that extends base class 
     * @param {Function} base
     * @param {Array} annotations
     * @param {Function} constructor
     */
    self.cls = function(base, annotations, constr) {
        if (!constr && annotations) { constr = annotations; annotations = null; }
        //assert(typeof constr == "function", "constructor must be a function");
        //assert(self.isClass(base), "base class must be an existing class");
        
        // make class stub
        var cls = clsFactory();
        
        // make class extending base class
        self.extend(cls, base);
        
        // add annotations to class
        if (annotations) self.annotate(cls, "__", annotations);
        
        // attach constructor
        if (constr) self.method(cls, "_constr", constr);
        
        return cls;
    };
    
    /**
     * Inits prototype of cls class according to base class
     * @param {Function} cls Class constructor
     * @param {Function} base Base Class constructor
     */
    self.extend = function(cls, base) {
        base = base || self;
        //assert(self.isClass(base), "base must be a class definition");
        //assert(!(new cls(self) instanceof base), "class can inherit only from one superclass");
        
        cls.prototype = new base(self);
        cls.prototype.constructor = cls;
        cls.prototype.base = base;
        cls.uniqeId = self.getNextUniqeId();
        cls.base = base;

        // code below is used for calling overriden methods
        // (it must be factorized like this to keep depth and proto in closure)
        
        var depth = {}, 
            proto = cls.prototype;
        
        cls.prototype.sup = function(name) {
            depth[name] = depth[name] || 0;

            var func, t = depth[name],
                baseProto = base.prototype;

            // find prototype of expected level
            while (t--) 
                if (baseProto.base)
                    baseProto = baseProto.base.prototype;
                else return;
                
            depth[name]++;
            var ret = baseProto[name].apply(this, Array.prototype.slice.call(arguments, 1));
            depth[name]--;
            
            return ret;
        };
        
        return cls;
    };
 
    /**
     * Adds annotations to class
     * @param {Function} cls Class that is annotated
     * @param {String} name Name of element that annotation describes
     * @param {Array} annotations List of annotations
     */
    self.annotate = function(cls, name, annots) {
        assert(self.isClass(cls), "cls must be a class definition to annotate");
        assert(annots && annots.length > 0, "invalid annotations"); 
        
        cls.annotations = cls.annotations || {};
        cls.annotations[name] = [];
        
        for (var i = 0; i < annots.length; i++) {
            
            // if param of annotation is an array then treat it as length annotations, each with i-th param
            if (annots[i][1] instanceof Array) {
                for (var j = 0; j < annots[i][1].length; j++)
                    cls.annotations[name].push([annots[i][0], annots[i][1][j]]);
                    
            } else {
                cls.annotations[name].push(annots[i]);
            }
        }
    };
 
    /**
     * Returns true if cls is class constructor
     * @param {Function} cls
     */
    self.isClass = function(cls) {
        return typeof cls == "function" && "uniqeId" in cls; 
    };
    
    /**
     * Defines method
     * @param {Function} cls
     * @param {String} prop
     * @param {Array} annotations
     * @param {Function} func
     */
    self.method = function(cls, prop, annots, func) {
        if (!func) { func = annots; annots = null; }
        
        //assert(self.isClass(cls), "cls must be a valid class");
        //assert(typeof func == "function", "method must be a function");
        
        cls.prototype[prop] = func;
        func.isMethod = true;
        
        if (annots) self.annotate(cls, prop, annots);
    };

    /**
     * Defines property
     * if getter is not set then property is creating getter and setter that works on _[property name] member
     * @param {Function} cls
     * @param {String} prop
     * @param {Function} setter
     * @param {Function} getter
     */
    self.property = function(cls, prop, annots, setter, getter) {
        // annotations is optional argument
        if (typeof annots == "function") { 
            getter = setter; setter = annots; annots = null;
        }
        var camel = prop.charAt(0).toUpperCase() + prop.substr(1),
            set = "_set" + camel,
            get = "_get" + camel;
            
        cls.prototype[set] = getter && setter || function(val, force) {
            var old = this["_" + prop];
            if (old != val || force) {
                this["_" + prop] = val;
                
                if (setter) setter.call(this, old, val);
            }
        };
        cls.prototype[get] = getter || function() {
            return this["_" + prop];
        };
        
        var fn = cls.prototype[prop] = function() {
            if (arguments.length > 0) {
                this[set].apply(this, arguments);
                return this;
            }
            return this[get].call(this);
        };
        fn.isProperty = true;
        
        if (annots) self.annotate(cls, prop, annots);
    };
    
    /**
     * Return all annotations (including inherited ones)
     */
    self.prototype._getAnnotations = function() {
        
        var cls = this.constructor, 
            ret = {};
            
        // go down the protoype chain and collect annotations
        while (cls) {
            if (cls.annotations)
                for (var name in cls.annotations)
                    ret[name] = (ret[name] || [])
                        .concat(cls.annotations[name]);
            cls = cls.base;
        }
        
        // reverse it so that top class annotations are first
        for (var name in ret)
            ret[name].reverse();
        
        return ret;
    };
    
    /**
     * Adds annotations to object instance
     * @param {Object} obj
     * @param {Object} name
     */
    self.prototype._annotateInstance = function() {
        
        var annots = this._getAnnotations(),
            names = [];
        
        // first annotate class
        if ("__" in annots)
            names.push("__");

        // then class members
        for (var name in annots)
            if (name != "__")
                names.push(name);
        
        // annotate it all
        for (var i = 0; i < names.length; i++) {
            var name = names[i],
                list = annots[name];
            for (var j = 0; j < list.length; j++)
                new list[j][0]().annotate(this, name, list[j][1]);
        }
    };
    
    self.prototype.toString = function() {
        return '[Object ' + this._name + ']';
    };

    return self;
})();