
var TAG = "AclWebApp";
var DEBUG = 0;
var DEBUG_TOUCH = 0;
var DEBUG_CONTACT = 0;

// these variables are disabled automatically by apk-to-webapp converter
// to reflect the application permissions
var ENABLE_LOCATION_SYNC=0;
var ENABLE_CONTACT_SYNC=0;
var ENABLE_MEDIA_SYNC=0;

// Message type
var WINDOW         = 1;
var NOTIFICATION   = 2;
var WEB_ACTIVITY   = 3;
var POWER          = 4;
var SETTINGS       = 5;
var USERALERT      = 6;
var RADIO          = 7;
var CONTACTS       = 8;
var MEDIA          = 9;
var LOCATION       = 10;

var USERALERT_LEVEL_WARNING = 0;
var USERALERT_LEVEL_ERROR = 1;
var USERALERT_LEVEL_FATAL = 2;

var RADIO_SMS_RECEIVED      = 1;
var RADIO_SMS_STATUS_REPORT = 2;
var RADIO_CONNSTATUS        = 3;

var MEDIA_SUBTYPE_SYNC      = 1;
var MEDIA_SUBTYPE_SYNCALL   = 2;

var MESSAGING_ACTIVITY   = 102;
var ACTIVITY_EMAIL       = 103;
var ACTIVITY_BLUETOOTH   = 104;

// Commands sent to ACL
var ACTION_LAUNCH      = 'LCH';
var ACTION_EXIT        = 'EXT';
var ACTION_ACTIVATE    = 'ACT';
var ACTION_DEACTIVATE  = 'DCT';
var ACTION_FOCUS       = 'FCS';
var ACTION_BLUR        = 'BLR';

// Power requests
var ACQUIRE_WAKELOCK   = 1;
var RELEASE_WAKELOCK   = 0;

// Window Commands received from ACL
var CMD_SHOW_DESKTOP  = 7;

var STORAGE_PREF = "/storage";
var SETTINGS_MEDIA_SYNC = "media-sync";
var MEDIA_SYNC_INSERT = 1;
var MEDIA_SYNC_DELETE = 2;

var PORT = 4564;

// Socket status
var UNDEFINED  = 0;
var CONNECTING = 1;
var CONNECTED  = 2;

var sock_status = UNDEFINED;
var tcpsock;

var appInfo;
var appName;
var pkgName;
var pkgShortname = null;
var app;
var wakelock = null;
var aclPlayer = null;
var l10n;
var connection;
var dataConnectionState = null;
var voiceConnectionState = null;
var aclUpgraded = false;
var aclVersion = null;
var aclInstallState = null;
var msgBuffer = '';
var pictures;

var positionWatchID = null;

// a dictionary of contacts, indexed by their ID's
var gContacts = {};
var gContactsDirection = {};
var gLastUpdate = 0;
var ANDROID2FFOS = 'a2f';
var FFOS2ANDROID = 'f2a';

function createPlayer() {
    //console.log(TAG + "-- Win w=" + window.screen.width + " h=" + window.screen.height);
    if (aclPlayer === null) {
        if (DEBUG) console.log(TAG + "****** Creating ACL player ******");
        var startCmd = {
                command : "start"
        };
        if (DEBUG) console.log(TAG + "createPlayer: posting message " + JSON.stringify(startCmd));
        aclPlayer = document.getElementById("AclPlayer");
        aclPlayer.postMessage(JSON.stringify(startCmd));
    }
    else {
        console.log(TAG + "****** Player is still alive !!! ****");
    }
}

function deletePlayer() {
    if (DEBUG) console.log(TAG + "****** deletePlayer aclPlayer=" + aclPlayer);
    if (aclPlayer) {
        var stopCmd = {
            command : "stop"
        };
        if (DEBUG) console.log(TAG + "deletePlayer: posting message " + JSON.stringify(stopCmd));
        aclPlayer.postMessage(JSON.stringify(stopCmd));
        delete aclPlayer;
        aclPlayer = null;
    }
}

function getAppReference(cb) {
    var request = navigator.mozApps.getSelf();
    request.onsuccess = function onApp(evt) {
        cb(evt.target.result);
    };
}

function handleNotification(action, title, text, icon, tag, pkg) {
    if (DEBUG)console.log(TAG + "handleNotification: action=" + action +
            " title="+ title + " text='" + text + "'" + " pkg=" + pkg +
            " icon='" + icon + "'" + " tag='" + tag + "'");
    if (action === "remove") {
        // Retrieve the notification and close it
        var n = Notification.get({tag: tag});
        n.then(
            function(nt) {
                if (nt.length === 1) {
                    nt[0].close();
                }
            },
            function(nt) {
            });

        return;
    }

    // We are using package name as icon name
    icon = pkg;
    var iconPath = "app://com.omww." + pkg + "/style/icons/" + icon + ".png";

    //Get app reference
    getAppReference(function onsuccess(a) {
            app = a;
    });

    if (DEBUG) console.log(TAG + "handleNotification: Icon path=" + iconPath);
    var notification = new Notification(title,
            { body: text, tag: tag, icon: iconPath});

    notification.onclick = function onclick() {
        notification.close();
        app.launch();
    };

    notification.onclose = function onclose() {
        if (DEBUG)console.log(TAG + 'Notification closed');
    };
}

function get_l10n(s)
{
    if (l10n !== undefined) {
        return l10n.get(s);
    }
    else {
        return s;
    }
}


function handleUserAlert(severity, message, pkg) {
    var alertMessage;
    var restartMessage = get_l10n('alert_restart');
    if (severity === USERALERT_LEVEL_WARNING) {
        alertMessage = "ACL WARNING: " + message;
    }
    else if (severity === USERALERT_LEVEL_ERROR) {
        alertMessage = "ACL ERROR: " + message + " -- " + restartMessage;
    }
    else if (severity === USERALERT_LEVEL_FATAL) {
        alertMessage = "ACL FATAL ERROR: " + message + " -- " + restartMessage;
    }
    if (DEBUG) console.log(TAG + "User alert received: " + alertMessage);
    alert(alertMessage);

    if (severity === USERALERT_LEVEL_FATAL) {
        if (DEBUG) console.log(TAG + "FATAL alert received, closing app");
        deletePlayer();
        var msg = { type:    WINDOW,
                    action:  ACTION_EXIT,
                    package: pkg };

        send_message(msg);
        alert("Closing window...");
        window.close();
    }
}

function saveCapturedImage(mozObj, blob) {
    if (DEBUG) console.log(TAG + "-- saveCapturedImage ");
    var sdcard  = navigator.getDeviceStorage("sdcard");
    var request = sdcard.add(blob);
    request.origObj = mozObj;

    request.onsuccess = function() {
        if (DEBUG) console.log(TAG + "-- Picture saved to sdcard");
        this.origObj.status = 0;  // Success
        this.origObj.data = STORAGE_PREF + this.result;
        send_message(this.origObj);
    }

    request.onerror = function() {
        console.error("-- Failed to save picture to sdcard: " + this.error);
        this.origObj.status = -1;  // Error
        send_message(this.origObj);
    }
}

function initMozActivity(mozActivity, id, activity) {
    mozActivity.id        = id;
    mozActivity.type      = WEB_ACTIVITY;
    mozActivity.activity  = activity;
    mozActivity.onsuccess = onMozActivitySuccess;
    mozActivity.onerror   = onMozActivityError;
}

function createMozActivity(data) {
    if (DEBUG)console.log(TAG + "createMozActivity() data: " + data);
    if (data.name === 'open') {
        var fileName = data.data.path;
        var filePath = fileName.slice(15);
        if (DEBUG)console.log(TAG + "=============fileName:" + fileName + "filePath:" + filePath);
        var availableStorages;
        var currentStorage;
        if (navigator.getDeviceStorages) {
            availableStorages = navigator.getDeviceStorages('sdcard');
            currentStorage = availableStorages[0];
        } else {
            currentStorage = navigator.getDeviceStorage('sdcard');
            availableStorages = [currentStorage];
        }
        if (DEBUG)console.log(TAG + 'Will try to open %s', filePath);
        var request = currentStorage.get(filePath);

        request.onsuccess = function() {
            var file = this.result;

            newActivity = new MozActivity({
                name: 'open',
                data: {
                    type: file.type,
                    blob: file,
                    filename: file.name
                }
            });
            initMozActivity(newActivity, data.id, data.activity);
        };
        request.onerror = function() {
            console.error(TAG + 'Unable to access device storage');
        };
    } else if ((data.activity == MESSAGING_ACTIVITY && data.name === 'share')
            || (data.activity == ACTIVITY_EMAIL && data.numofattachments > 0)
            || (data.activity == ACTIVITY_BLUETOOTH  && data.name === 'share')) {
        handleAttachments(data);
    } else {
        newActivity = new MozActivity(data);
        initMozActivity(newActivity, data.id, data.activity);
    }
}

function onMozActivitySuccess() {
    if (DEBUG) console.log(TAG + "onMozActivitySuccess() result=" + this.result);
    if (this.result !== null && this.result.blob !== undefined) {
        if (this.result.blob.name !== undefined) {
            this.status = 0;  // Success
            this.data = STORAGE_PREF + this.result.blob.name;
            send_message(this);
        } else {
            saveCapturedImage(this, this.result.blob);
        }
    } else {
        this.status = 0;  // Success
        this.data = '';
        send_message(this);
    }
}

function onMozActivityError() {
    console.error("onMozActivityError()  " + this.error);
    this.status = -1;  // Error
    send_message(this);
}

function handleAttachments(data) {
    var blobs = [];
    var isReady = 0;
    var sdcardArray = navigator.getDeviceStorages('sdcard');
    var sdcard = sdcardArray[0];
    var filepaths = data.data.filepaths;

    for (i = 0; i < filepaths.length; i++) {
        path = filepaths[i];
        var pathStart = path.search('/sdcard');
        if(pathStart > 0) {
            path = path.substr(pathStart);
        }

        if(DEBUG) console.log(TAG + ' path: ' + path);

        request = sdcard.get(path);

        request.onsuccess = function() {
            blobs.push(this.result);
            isReady++;
            if (isReady === data.data.filepaths.length) {
                data.data["blobs"] = blobs;
                newActivity = new MozActivity(data);
                initMozActivity(newActivity, data.id, data.activity);
            }
        }
        request.onerror = function() {
            console.error(TAG + "Unable to access device storage: " + this.error.name);
        }
    }
}

function smsReceived(evt) {
    var message = evt.settingValue;
    if (message !== 'listening') {
        if(DEBUG) console.log(TAG + 'Received SMS message: ' + JSON.stringify(message));
        var msg = { type: RADIO,
                    radio_type: RADIO_SMS_RECEIVED,
                    content: message };
        send_message(msg);
        // We got the msg, set status back to 'listening'
        // navigator.mozSettings.createLock().set( { 'acl.sms.received': 'listening' });``
    }
}

function initSMSListener() {
    var settings = navigator.mozSettings;
    if(settings !== null) {
        if(DEBUG) console.log(TAG + 'Adding SMS listener');
        settings.addObserver('acl.sms.received', smsReceived);
    }
    else {
        if(DEBUG) console.log(TAG + 'initSMSListener: cannot access mozSettings');
    }
}

function removeSMSListener() {
    var settings = navigator.mozSettings;
    if(settings !== null) {
        if(DEBUG) console.log(TAG + 'Removing SMS listener');
        settings.removeObserver('acl.sms.received', smsReceived);
    }
    else {
        if(DEBUG) console.log(TAG + 'initSMSListener: cannot access mozSettings');
    }
}

function initConnectionMonitor() {
    connection = navigator.mozMobileConnections[0];
    if (!connection) {
        if(DEBUG) console.log(TAG + "Cannot access mozMobileConnections");

        // If we can't access this then we just won't do this function
        return;
    }

    function sendStateToACL() {
        if (!connection) {
            return;
        }

        // TODO: Most likely this will only happen if there's no SIM card.
        // In cases where this does go away during operation we may want
        // to inform ACL of this
        if (!connection.voice.network) {
            return;
        }

        if ((voiceConnectionState !== connection.voice.connected) ||
            (dataConnectionState !== connection.data.connected)) {
            if (DEBUG) console.log(TAG + "Connection status changed, updating");
            var settings = navigator.mozSettings;
            var req = settings.createLock().get('ril.data.apnSettings');
            req.onsuccess = function() {
                var apnSettings = req.result['ril.data.apnSettings'];
                if (DEBUG) console.log(TAG + "APN settings retrieved: " + JSON.stringify(apnSettings));
                var msg = {
                    type: RADIO,
                    radio_type: RADIO_CONNSTATUS,
                    package: pkgName,
                    selectionMode: connection.networkSelectionMode,
                    iccId: connection.iccId,
                    voiceinfo : {
                        cell : {
                            gsmCellId: connection.voice.cell.gsmCellId,
                            gsmLocationAreaCode: connection.voice.cell.gsmLocationAreaCode
                        },
                        connected : connection.voice.connected,
                        emergencyCallsOnly : connection.voice.emergencyCallsOnly,
                        lastKnownMcc : connection.voice.lastKnownMcc,
                        network : {
                            longName: connection.voice.network.longName,
                            mcc: connection.voice.network.mcc,
                            mnc: connection.voice.network.mnc,
                            shortName: connection.voice.network.shortName,
                            state: connection.voice.network.state
                        },
                        relSignalStrength : connection.voice.relSignalStrength,
                        roaming : connection.voice.roaming,
                        signalStrength : connection.voice.signalStrength,
                        state : connection.voice.state,
                        type : connection.voice.type
                    },
                    datainfo : {
                        cell : {
                            gsmCellId: connection.data.cell.gsmCellId,
                            gsmLocationAreaCode: connection.data.cell.gsmLocationAreaCode
                        },
                        connected : connection.data.connected,
                        emergencyCallsOnly : connection.data.emergencyCallsOnly,
                        lastKnownMcc : connection.data.lastKnownMcc,
                        network : {
                            longName: connection.data.network.longName,
                            mcc: connection.data.network.mcc,
                            mnc: connection.data.network.mnc,
                            shortName: connection.data.network.shortName,
                            state: connection.data.network.state
                        },
                        relSignalStrength : connection.data.relSignalStrength,
                        roaming : connection.data.roaming,
                        signalStrength : connection.data.signalStrength,
                        state : connection.data.state,
                        type : connection.data.type
                    },
                    apn: apnSettings
                };
                send_message(msg);
            }
            req.onerror = function(e) {
                if(DEBUG) console.log(TAG + "Cannot get ril settings: " + e);
            }

            dataConnectionState = connection.data.connected;
            voiceConnectionState = connection.voice.connected;
        }
    }

    if(connection.voice.network !== null) {
        if(DEBUG) console.log(TAG + "Sending initial connection state to ACL");
        sendStateToACL();
    }
    else {
        if (DEBUG) console.log(TAG + "connection.voice.network is NULL, no SIM card?");
    }

    connection.ondatachange = (function(e) {
        sendStateToACL();
    });

    connection.onvoicechange = (function(e) {
        sendStateToACL();
    });

    connection.oncfstatechange = (function(e) {
        sendStateToACL();
    });

    connection.ondataerror = (function(e) {
        sendStateToACL();
    });

    connection.onussdreceived = (function(e) {
        sendStateToACL();
    });
}

function initMediaWatcher() {
    pictures  = navigator.getDeviceStorage("pictures");
    console.log(TAG + "Setting up media watcher");

    pictures.addEventListener("change", function(event) {
        var reason = event.reason;
        var path = event.path;
        var msg = {
            type: MEDIA,
            subtype: MEDIA_SUBTYPE_SYNC,
            path: path,
            reason: reason
        };
        send_message(msg);
        console.log(TAG + 'Media change: file ' + path + ' has been ' + reason);
    });
}

function doInitialMediaSync() {
    console.log(TAG + "Doing initial media sync...");
    var msg = {
        type: MEDIA,
        subtype: MEDIA_SUBTYPE_SYNCALL
    };
    send_message(msg);
    console.log(TAG + "Media sync requested");
}

function initLocationMonitor() {
    var locOptions = {
        enableHighAccuracy: true,
        timeout: Infinity,
        maximumAge: 0
    };

    function locationSuccess(position) {
        var msg = { type:    LOCATION,
                            latitude:  position.coords.latitude,
                            longitude: position.coords.longitude,
            package: pkgName  };
        if(DEBUG) console.log(TAG + "position = " + position);
        if(DEBUG) console.log(TAG + "sending position " + position.coords.latitude + position.coords.longitude);
        send_message(msg);
    }

    function locationError(posErr) {
        if(DEBUG) console.log(TAG + "watchPosition error " + posErr.code + ", " + posErr.message);
    }

    positionWatchID = navigator.geolocation.watchPosition(locationSuccess, locationError, locOptions);


    if(DEBUG) console.log(TAG + "Position monitor is initialized, positionWatchID = " + positionWatchID);
}

function onLoad() {
    appInfo = document.getElementById("acl").dataset;
    appName = document.title;
    pkgName = appInfo.pkgname;
    aclInstallStatus = "UNKNOWN";

    l10n = navigator.mozL10n;

    TAG = TAG + "[" + appName + "] ";

    if (DEBUG) console.log(TAG + "Starting...");

    function startApp() {
        if (ENABLE_MEDIA_SYNC) initMediaWatcher();
        initTCPSocket();
        if (ENABLE_LOCATION_SYNC) initLocationMonitor();
        initSMSListener();
        createPlayer();
    }

    function onVersionAvailable() {
        if(DEBUG) console.log(TAG + "Current ACL version is '" + aclVersion +
                "'  min version is '" + appInfo.min_acl_rev + "'");
        if ((aclVersion === '') || (aclUpgraded === true)) {
            if(DEBUG) console.log(TAG + "ACL version is empty, ACL might not be installed.");
            showErrorPage();
            return;
        }

        if (versionCompare(appInfo.min_acl_rev, aclVersion, {maxLen:3}) <= 0) {
            getInstallStatus(startApp);
        } else {
            if(DEBUG) console.log(TAG + "ACL version is too old, need to upgrade...");
            var ver1 = get_l10n('alert_version1');
            var ver2 = get_l10n('alert_version2');

            alert(ver1 + ' ' + aclVersion + ', ' + ver2 + ' ' + appInfo.min_acl_rev);
            window.close();
        }
    }

    updateAclStatus(onVersionAvailable);
}

function waitForAppInstall() {
    sessionStorage.setItem('waitstate', 'appinstall');
    sessionStorage.setItem('appName', appName);
    if(pkgShortname !== null) {
        sessionStorage.setItem('pkgName', pkgShortname);
    }
    else {
        sessionStorage.setItem('pkgName', pkgName);
    }
    sessionStorage.setItem('header', get_l10n('pleasewait_header'));
    sessionStorage.setItem('msg1', get_l10n('pleasewait_msg1'));
    sessionStorage.setItem('msg2', get_l10n('pleasewait_msg2'));
    window.location.href = "/pleasewait.html";
}

function waitForACLInstallStatusChange() {
    cancelEventHandlers();
    sessionStorage.setItem('waitstate', 'aclInstallStatusChange');
    sessionStorage.setItem('header', get_l10n('pleasewait_header'));
    sessionStorage.setItem('appName', '');
    sessionStorage.setItem('msg2', '');
    if(aclInstallState === 'installing') {
        sessionStorage.setItem('msg1', get_l10n('pleasewait_installing'));
    }
    else if(aclInstallState === 'uninstalling') {
        sessionStorage.setItem('msg1', get_l10n('pleasewait_uninstalling'));
    }
    else if(aclInstallState === 'upgrading') {
        sessionStorage.setItem('msg1', get_l10n('pleasewait_upgrading'));
    }
    
    window.location.href = "/pleasewait.html";

}

function hashName(pkgName) {
    var hash = 5381;
    var i;
    for(i = 0; i < pkgName.length; i++) {
        hash = ((hash * 33) + pkgName.charCodeAt(i)) % 0x100000000;
    }
    return hash.toString(16);
}

function getInstallStatus(callback) {
    if(pkgShortname === null) {
        if (pkgName.length > 16) {
            pkgShortname = hashName(pkgName);
        }
        else {
            pkgShortname = pkgName;
        }
    }

    navigator.getFeature("acl." + pkgShortname + ".is").then(function(installStatus) {
        if (installStatus === 'installed' || installStatus === undefined) {
            // If installStatus is undefined then we're running an older version
            // of ACL.  Proceed to the version check.
            if(DEBUG) console.log("getInstallStatus: installStatus is " + installStatus);
            callback();
        }
        else if (installStatus === 'installing') {
            waitForAppInstall();
        }
        else {
            var errorString = get_l10n('alert_error');
            var installStatusString = get_l10n('alert_installstatus');
            var contactSupportString = get_l10n('alert_contactsupport');
            alert(errorString + " " + appName + " " + installStatusString + " " + installStatus + " " + contactSupportString);
            window.close();
        }
    });
}

function onBlur() {
    deletePlayer();

     if(DEBUG) console.log(TAG + "======= Window blurred...");

     var msg = { type:    WINDOW,
                 action:  ACTION_BLUR,
                 package: pkgName };

     send_message(msg);
}

function onFocus() {
    if(DEBUG) console.log(TAG + "======= Window onfocus...");

    //"visibilitychange" event could be too late, so force visability
    changeVisability(true);

    var msg = { type:    WINDOW,
                action:  ACTION_FOCUS,
                package: pkgName };

    send_message(msg);
}

function onUnload() {
    if(DEBUG) console.log(TAG + "======= Window unload...");
    var msg = { type:    WINDOW,
                action:  ACTION_EXIT,
                package: pkgName };

    send_message(msg);
    removeSMSListener();
}

function changeVisability(new_visible) {
    var visible = false;

    function doPause() {
        var pauseCmd = {
            command : "pause"
        };
        if (DEBUG) console.log(TAG + "onVisibilityChange: posting message " + JSON.stringify(pauseCmd));
        if (aclPlayer) aclPlayer.postMessage(JSON.stringify(pauseCmd));

        var msg = { type: WINDOW, action: ACTION_DEACTIVATE, package: pkgName };
        send_message(msg);
    }

    function doResume() {
        if ((aclVersion === '') || (aclUpgraded === true)) {
            if(DEBUG) console.log(TAG + "ACL version is empty, ACL might not be installed.");
            showErrorPage();
            return;
        }

        var msg = { type: WINDOW, action: ACTION_ACTIVATE, package: pkgName };
        send_message(msg);
        createPlayer();
        var resumeCmd = {
            command: "resume"
        };
        if (DEBUG) console.log(TAG + "onVisibilityChange: posting message " + JSON.stringify(resumeCmd));
        aclPlayer.postMessage(JSON.stringify(resumeCmd));
    }

    if(visible != new_visible) {
        if(DEBUG) console.log(TAG + "changeVisability: " + new_visible);

        visible = new_visible;
        if(visible) {
            updateAclStatus(doResume);
        } else {
            doPause();
        }
    }
}

function onVisibilityChange() {
    if (DEBUG) console.log(TAG + "onVisibilityChange: " +
            (document.hidden? "hidden" : "visible"));

    changeVisability(! document.hidden);
}

function versionCompare(v1, v2, options) {
    var zeroExtend = options && options.zeroExtend,
        v1parts = v1.split('.'),
        v2parts = v2.split('.');

    var v1Len = options.maxLen? Math.min(v1parts.length, options.maxLen) : v1parts.length;
    var v2Len = options.maxLen? Math.min(v2parts.length, options.maxLen) : v2parts.length;

    if (zeroExtend) {
        while (v1Len < v2Len) v1parts.push("0");
        while (v2Len < v1Len) v2parts.push("0");
    }

    for (var i = 0; i < v1Len; ++i) {
        if (v2Len == i) return 1;

        var v1Int = parseInt(v1parts[i], 10);
        var v2Int = parseInt(v2parts[i], 10);
        if (v1Int == v2Int) {
            var v1Lex = v1parts[i].substr((""+v1Int).length);
            var v2Lex = v2parts[i].substr((""+v2Int).length);
            if (v1Lex === '' && v2Lex !== '') return -1;
            if (v1Lex !== '' && v2Lex === '') return 1;
            if (v1Lex !== '' && v2Lex !== '') return v1Lex === v2Lex ? 0 : (v1Lex > v2Lex ? 1 : -1);
        } else if (v1Int > v2Int) {
            return 1;
        } else {
            return -1;
        }
    }

    if (v1Len != v2Len) return -1;

    return 0;
}


function cancelEventHandlers() {
    window.onblur = null;
    window.onfocus = null;
    window.onunload = null;
    window.onlanguagechange = null;
    document.removeEventListener("visibilitychange", onVisibilityChange, false);
    document.removeEventListener("touchstart", handleTouchEvent, false);
    document.removeEventListener("touchmove", handleTouchEvent, false);
    document.removeEventListener("touchend", handleTouchEvent, false);
}

function showErrorPage() {
    if (DEBUG) console.log(TAG + "ACL is installed = " + aclInstallState);

    // In all cases we no longer want to handle events; this routine simply
    // dispatches to the appropriate page
    cancelEventHandlers();

    if (aclUpgraded) {
        // alert("ACL has been upgraded.  Reboot required");
        window.location.href = "/reboot.html";
    }
    else if (aclInstallState === 'installed') {
        // Unable to connect to ACL, show the error page.
        if (DEBUG) console.log(TAG + "Redirecting to error.html");
        window.location.href = "/error.html";
    } else if (aclInstallState === 'uninstalling') {
        if (DEBUG) console.log(TAG + "ACL uninstalling, redirecting to download.html");
        window.location.href = "/download.html";
    } else if (aclInstallState === 'uninstalled' || aclInstallState == undefined) {
        // ACL is not fully installed.  If the webapp is available
        // to accept the EULA and install, then launch it.  If
        // that fails, assume it's not present and go to the
        // download page.

        if (DEBUG) console.log(TAG + "Launching ACL activity");
        var activity = new MozActivity ({
               name: "launchACL"
        });

        // If we were able to launch ACL then we were able to do
        // (or at least start) the install.  Our work here is done.
        activity.onsuccess = (function() {
            if (DEBUG) console.log(TAG + "ACL launch successful, closing window");
            window.close();
        });

        // If we were not able to launch ACL then it truly is not
        // installed.  Redirect to download screen.
        activity.onerror = (function() {
            if (DEBUG) console.log(TAG + "Redirecting to download.html");
            window.location.href = "/download.html";
        });
    }
    else {
        if (DEBUG) console.log(TAG + "Redirecting to error.html");
        window.location.href = "/error.html";
    }
}

function updateAclStatus(callback) {
    navigator.getFeature("acl.version").then(function(ver) {
        if (ver === undefined) {
            if(DEBUG) console.log(TAG + "ACL not supported on this device");
            cancelEventHandlers();
            window.location.href = "/unsupported.html";
        }

        var stateArray = ver.split(";");
        aclVersion = stateArray[4];
        if(stateArray.length > 5) {
            aclUpgraded = true;
        }
        else {
            aclUpgraded = false;
        }
        if(DEBUG) console.log(TAG + "updateAclStatus: aclVersion = " + aclVersion + " aclUpgraded = " + aclUpgraded);
        navigator.getFeature("acl.install_status").then(function(status) {
            if(DEBUG) console.log(TAG + "updateAclStatus: acl.install_status -> " + status);
            aclInstallState = status;
            if(DEBUG) console.log(TAG + "updateAclStatus: aclInstallState = " + aclInstallState);
            if (aclInstallState === 'installing' ||
                aclInstallState === 'uninstalling' ||
                aclInstallState === 'upgrading') {
                waitForACLInstallStatusChange();
            }
            else {
                callback();
            }
        });
    });
}


function handleContactOperation(data) {
    if (data.operation == "insert") {
         if (DEBUG) console.log(TAG + "insert a contact");
         if (data.familyName && data.givenName) {
              var contactData = {
                  familyName: [data.familyName],
                  givenName: [data.givenName]
              }
         } else if (data.familyName) {
              var contactData = {
                  familyName: [data.familyName]
              }
         } else if (data.givenName) {
              var contactData = {
                  givenName: [data.givenName]
              }
         } else {
              var contactData = {}
         }

         var person = new mozContact(contactData);
         person.tel = [{type: ["mobile"], value: data.phoneNumber, carrier: "myCarrier", pref: 1}];
         var saving = navigator.mozContacts.save(person);
         saving.onsuccess = function() {
              if (DEBUG) console.log(TAG + "new contact saved:" + person + "  id:" + person.id);
              gContactsDirection[person.id] = ANDROID2FFOS;

              var msg = { type: CONTACTS,
                  operation: "CONTACTADDED",
                  familyName: data.familyName,
                  givenName: data.givenName,
                  phoneNumber: data.phoneNumber,
                  rawAndroidId: data.rawAndroidId,
                  ffosId: person.id};
              send_message(msg);
         };
         saving.onerror = function(err) {
              console.error(err);
         };
    }
    else if (data.operation == "update") {
         if (DEBUG) console.log(TAG + "update a contact");
         var options = {
            filterValue: data.ffosId,
            filterBy: ['id'],
            filterOp: 'equals'
         }

        var search = navigator.mozContacts.find(options);

        search.onsuccess = function() {
            if (search.result.length === 1) {
                 if (DEBUG) console.log(TAG + "update a2f find one");
                 if (data.familyName) {
                     search.result[0].familyName = [data.familyName];
                 }
                 else {
                     search.result[0].familyName = null;
                 }
                 if (data.givenName) {
                     search.result[0].givenName = [data.givenName];
                 }
                 else {
                     search.result[0].givenName = null;
                 }
                 if (data.phoneNumber) {
                     search.result[0].tel = [{type: ["mobile"], value: data.phoneNumber, carrier: "myCarrier", pref: 1}];
                 }
                 else {
                     search.result[0].tel = [{type: ["mobile"], value: null, carrier: "myCarrier", pref: 1}];
                 }
                 var saving = navigator.mozContacts.save(search.result[0]);
                 saving.onsuccess = function() {
                     if (DEBUG) console.log(TAG + "update a2f contact saved");
                     gContactsDirection[search.result[0].id] = ANDROID2FFOS;
                     var msg = { type: CONTACTS,
                         operation: "CONTACTUPDATED",
                         familyName: data.familyName,
                         givenName: data.givenName,
                         phoneNumber: data.phoneNumber,
                         rawAndroidId: data.rawAndroidId,
                         ffosId: search.result[0].id};
                     send_message(msg);
                 };
                 saving.onerror = function(err) {
                     console.error(err);
                 };

            } else {
                 console.log(TAG + search.result.length + " update a2f, cannot update");
            }
        };
        search.onerror = function() {
            console.log(TAG + "could not search contacts for ID");
        };
    }
    else if (data.operation == "updateecho") {
         if (DEBUG) console.log(TAG + "updateecho a contact");
         var options = {
            filterValue: data.ffosId,
            filterBy: ['id'],
            filterOp: 'equals'
         }

        var search = navigator.mozContacts.find(options);

        search.onsuccess = function() {
            if (search.result.length === 1) {
                 if (DEBUG) console.log(TAG + "update a2f find one");
                 if (data.familyName) {
                     search.result[0].familyName = [data.familyName];
                 }
                 else {
                     search.result[0].familyName = null;
                 }
                 if (data.givenName) {
                     search.result[0].givenName = [data.givenName];
                 }
                 else {
                     search.result[0].givenName = null;
                 }
                 if (data.phoneNumber) {
                     search.result[0].tel = [{type: ["mobile"], value: data.phoneNumber, carrier: "myCarrier", pref: 1}];
                 }
                 else {
                     search.result[0].tel = [{type: ["mobile"], value: null, carrier: "myCarrier", pref: 1}];
                 }
                 var saving = navigator.mozContacts.save(search.result[0]);
                 saving.onsuccess = function() {
                     if (DEBUG) console.log(TAG + "update a2f contact saved");
                     gContactsDirection[search.result[0].id] = ANDROID2FFOS;
                     var msg = { type: CONTACTS,
                         operation: "CONTACTADDED",
                         ffosId: search.result[0].id,
                         rawAndroidId: data.rawAndroidId,
                         phoneNumber: data.phoneNumber};
                     send_message(msg);
                 };
                 saving.onerror = function(err) {
                     console.error(err);
                 };

            } else {
                 console.log(TAG + search.result.length + " update a2f, cannot update");
            }
        };
        search.onerror = function() {
            console.log(TAG + "could not search contacts for ID");
        };
    }
    else if (data.operation == "remove") {
         if (DEBUG) console.log(TAG + "remove a contact");
         var options = {
            filterValue: data.ffosId,
            filterBy: ['id'],
            filterOp: 'equals'
         }

        var search = navigator.mozContacts.find(options);

        search.onsuccess = function() {
            if (search.result.length === 1) {
                 if (DEBUG) console.log(TAG + "remove a2f find one");
                 gContactsDirection[search.result[0].id] = ANDROID2FFOS;
                 navigator.mozContacts.remove(search.result[0]);
                 var msg = { type: CONTACTS,
                     operation: "CONTACTREMOVED",
                     ffosId: search.result[0].id};
                 send_message(msg);
            }
            else {
                if (DEBUG) console.log(TAG + "remove a2f could fine one");
            }
        }
        search.onerror = function() {
            console.log(TAG + "could not search contacts for ID");
        };
    }
    else if (data.operation == "lastsync") {
        console.log(TAG+ "got lastsync:" + data.lastSync);
        gLastUpdate = parseInt(data.lastSync);
    }
}

function pictureAdded(picFile) {
    var prefix = "/storage/" + pictures.storageName + "/";
    var storageFile = picFile.replace(prefix, "");

    console.log(TAG + "pictureAdded: adding file " + storageFile);

    var getRequest = pictures.get(storageFile);

    getRequest.onsuccess = function() {
        var fileBlob = this.result;
        console.log(TAG + "Successfully got " + storageFile + " size " + fileBlob.size + " type " + fileBlob.type);
        var reader = new FileReader();

        reader.readAsArrayBuffer(fileBlob);
        reader.onload = function() {
            var contentsbuf = new Array(this.result);
            console.log(TAG + "contentsbuf is " + typeof contentsbuf);
            var newblob = new Blob(contentsbuf, { type : fileBlob.type });

            var deleteRequest = pictures.delete(storageFile);

            deleteRequest.onsuccess = function() {
                console.log(TAG + "Delete request successful.  Re-adding");
                console.log(TAG + "Adding file as " + storageFile + " size = " + newblob.size);
                var addRequest = pictures.addNamed(newblob, storageFile);

                addRequest.onsuccess = function() {
                    console.log(TAG + "Add request successful.");
                };

                addRequest.onerror = function() {
                    console.log(TAG + "Could not add file: " + this.error.name);
                };
            };

            deleteRequest.onerror = function() {
                console.log(TAG + "Could not delete file: " + this.error.name);
            };
        };
        reader.onerror = function() {
            console.log(TAG + "Could not read file: " + this.error.name);
        };
    };

    getRequest.onerror = function() {
        console.log(TAG + "Could not get file: " + this.error.name);
    };
}

function pictureDeleted(picFile) {
    var pictures = navigator.getDeviceStorage("pictures");
    var dummyBlob = new Blob(["dummy data"], {type: "image/jpeg"});
    var prefix = "/storage/" + pictures.storageName + "/";
    var storageFile = picFile.replace(prefix, "");

    console.log(TAG + "pictureDeleted: removing file " + storageFile);

    var addRequest;

    addRequest = pictures.addNamed(dummyBlob, storageFile);

    addRequest.onsuccess = function() {
        doLog("Add request successful: " + storageFile);

        var deleteRequest = pictures.delete(storageFile);

        deleteRequest.onsuccess = function() {
            doLog("delete request successful");
            window.close();
        };

        deleteRequest.onerror = function() {
            doLog("delete failed: " + this.error.name);
            window.close();
        };
    };

    addRequest.onerror = function() {
        doLog("add request failed: " + this.error.name);
        window.close();
    }

}
function onMediaSyncRequest(data) {
    console.log(TAG + "OnMediaSyncRequest");
    console.log(TAG + "type=" +data.storage_type + " name ="+ data.storage_name + " path ="+ data.storage_path);
    if(data.operation === MEDIA_SYNC_INSERT) {
        console.log(TAG + "Adding picture: " + data.storage_path);
        pictureAdded(data.storage_path);
    }
    else if (data.operation === MEDIA_SYNC_DELETE) {
        console.log(TAG + "Deleting picture: " + data.storage.path);
        pictureDeleted(data.storage_path);
    }
}

function initTCPSocket() {
    if (DEBUG) console.log(TAG + "Connecting to TCP socket...");

    if (sock_status == CONNECTING || sock_status == CONNECTED) {
        if (DEBUG) console.log(TAG + "Connecting or already connected!");
        return;
    }

    sock_status = CONNECTING;

    try {
        tcpsock = navigator.mozTCPSocket.open('127.0.0.1', PORT);

        tcpsock.onopen = function(evt) {
            if (DEBUG) console.log(TAG + "TCP socket opened....");
            sock_status = CONNECTED;
            initConnectionMonitor();
            if (DEBUG) console.log(TAG + "launching...");
            var msg = { type:    WINDOW,
                        action:  ACTION_LAUNCH,
                        package: pkgName };

            send_message(msg);

            onLanguageChange();
            if (ENABLE_CONTACT_SYNC) startContactsUpdate();
            if (ENABLE_MEDIA_SYNC) doInitialMediaSync();
        }

        tcpsock.onerror = function(evt) {
            console.error(TAG + "error occured: " + evt);
            updateAclStatus(showErrorPage);
        }

        tcpsock.onclose = function(evt) {
            if(DEBUG) console.log(TAG + "TCP socket closed " + evt);
            updateAclStatus(showErrorPage);
        }

        tcpsock.ondata = function (event) {
            msgBuffer += event.data;
            var i = msgBuffer.indexOf('\n');
            while (i !== -1) {
                var aclMsg = msgBuffer.slice(0, i);
                msgBuffer = msgBuffer.slice(i + 1);

                if (DEBUG)console.log(TAG + " received data: " + event.data);
                var jsonData = JSON.parse(decodeURIComponent(escape(aclMsg)));
                if(jsonData.type === WINDOW) {
                    if(jsonData.request == CMD_SHOW_DESKTOP) {
                        if (DEBUG) console.log(TAG + "Window minimizing");
                    } else {
                        console.error(TAG + "Unknown window request received");
                    }
                } else if (jsonData.type == NOTIFICATION) {
                    if (DEBUG)console.log(TAG + "Notification request");
                    handleNotification(jsonData.action, jsonData.title, jsonData.text,
                            jsonData.icon, jsonData.tag, jsonData.pkg);

                } else if (jsonData.type == WEB_ACTIVITY) {
                    if (DEBUG)console.log(TAG + "WebActivity request");
                    createMozActivity(jsonData.request);

                } else if (jsonData.type == POWER) {
                    if (DEBUG) console.log(TAG + "Power manager request = " + jsonData.request);
                    if (jsonData.request === ACQUIRE_WAKELOCK) {
                        if (DEBUG) console.log(TAG + "Acquiring wakelock");
                        if (wakelock == null) {
                            wakelock = window.navigator.requestWakeLock('screen');
                        } else {
                            if (DEBUG) console.log(TAG + "wakelock was already acquired!");
                        }
                    } else if (jsonData.request === RELEASE_WAKELOCK) {
                        if (DEBUG) console.log(TAG + "Releasing wakelock");
                        if (wakelock != null) {
                            wakelock.unlock();
                            wakelock = null;
                        } else {
                            if (DEBUG) console.log(TAG + "wakelock was already released!");
                        }
                    }
                } else if (jsonData.type == USERALERT) {
                    handleUserAlert(jsonData.severity, jsonData.text, pkgName);
                } else if (jsonData.type == CONTACTS) {
                    if (ENABLE_CONTACT_SYNC) handleContactOperation(jsonData.request);
                } else if (jsonData.type === SETTINGS) {
                    if(jsonData.request === SETTINGS_MEDIA_SYNC) {
                        if (ENABLE_MEDIA_SYNC) onMediaSyncRequest(jsonData);
                    }
                    else {
                        console.error(TAG + "Unknown settings request");
                    }
                } else {
                    console.error(TAG + "Unknown data received");
                }
                i = msgBuffer.indexOf('\n');
            }
        }
    } catch (err) {
        alert(err);
        return;
    }
}

function send_message(msg) {
    if (DEBUG) console.log(TAG + "sending  '" + JSON.stringify(msg) + "'");
    if (sock_status == CONNECTED) {
        try {
            tcpsock.send(JSON.stringify(msg)+"\n");
        } catch (err) {
            console.error(TAG + "send_message exception: " + err);
            updateAclStatus(showErrorPage);
        }
    } else
        console.error(TAG + "Failed to send message because not connected!!");
}

window.navigator.mozSetMessageHandler('notification', function onNotification(msg) {
    if(DEBUG) console.log(TAG + "onNotification() title='" + msg.title + "' clicked=" + msg.clicked);

    if (!msg.clicked)
        return;

    Notification.get().then(function(notifications) {
        if (notifications) {
            notifications.some(function(notification) {
                if (notification.title === msg.title) {
                    //console.log(TAG + "Found it: tag = " + notification.tag);
                    notification.close();
                    return true;
                }
            });
        }
    });

    getAppReference(function onApp(app) {
        app.launch();
    });
});

function onLanguageChange(ev) {
    if(DEBUG) console.log(TAG + "Language changed to '" + window.navigator.language + "'");
    var msg = { type: SETTINGS,
        package: pkgName,
        name: 'language',
        value: window.navigator.language };
    send_message(msg);
}

function handleTouchEvent(evt) {
    evt.preventDefault();
    if (DEBUG_TOUCH) console.log(TAG + "Dropping touch event " + evt);
}

// updates the local contacts database with the given contact
// if the given contact has an unknown ID, add it
// if the given contact has a known ID, then
//      if the given contact is newer than the one already in the local DB, update
function compareOneContact(inContact) {
    if (DEBUG) console.log(TAG + "compareOneContact name=" + inContact.name + "  tel=" + inContact.tel);
    var vFamilyName = "";
    if (inContact.familyName)
        vFamilyName = inContact.familyName[0];
    var vGivenName = "";
    if (inContact.givenName)
        vGivenName = inContact.givenName[0];
    var vPhoneNumber = "";
    if ((inContact.tel) && (inContact.tel != '')) {
        console.log(TAG + "===========not null contact, inContact.tel[0]="+ inContact.tel[0]);
        vPhoneNumber = inContact.tel[0].value;
        var i = 0;
        for (;inContact.tel[i];) {
            console.log(TAG + "===========not null contact iterate");
            if (inContact.tel[i].type == "mobile") {
               vPhoneNumber = inContact.tel[i].value;
               break;
            }
            i++
        }
    }
  if (gContacts[inContact.id]) {
    var existingRecord = gContacts[inContact.id];

    if (existingRecord.updated >= inContact.updated) {
      if (DEBUG_CONTACT) console.log(TAG + "exisiting same age as modified one");
    } else {
      gContacts[inContact.id] = inContact;
      if (DEBUG_CONTACT) console.log(TAG + "exisiting older than modified, updating");
      var msg = { type: CONTACTS,
                  operation: "UPDATE",
                  familyName: vFamilyName,
                  givenName: vGivenName,
                  phoneNumber: vPhoneNumber,
                  ffosId: inContact.id };
      send_message(msg);
    }
  } else {
    if (DEBUG_CONTACT) console.log(TAG + "adding new contact to DB " + inContact.id);
    var msg = { type: CONTACTS,
                operation: "ADD",
                familyName: vFamilyName,
                givenName: vGivenName,
                phoneNumber: vPhoneNumber,
                ffosId: inContact.id };
    send_message(msg);
    gContacts[inContact.id] = inContact;
  }
}

function removeOneContactByID(inRemovedContactID) {
  if (gContacts[inRemovedContactID]) {
    if (DEBUG_CONTACT) console.log(TAG + "found removed contact, removing", inRemovedContactID);
    var vFamilyName = "";
    if (gContacts[inRemovedContactID].familyName)
        vFamilyName = gContacts[inRemovedContactID].familyName[0];
    var vGivenName = "";
    if (gContacts[inRemovedContactID].givenName)
        vGivenName = gContacts[inRemovedContactID].givenName[0];
    var vPhoneNumber = "";
    if (gContacts[inRemovedContactID].tel[0])
        vPhoneNumber = gContacts[inRemovedContactID].tel[0].value;
    var i = 0;
    for (;gContacts[inRemovedContactID].tel[i];) {
        if (gContacts[inRemovedContactID].tel[i].type == "mobile") {
            vPhoneNumber = gContacts[inRemovedContactID].tel[i].value;
            break;
        }
        i++
    }
    var msg = { type: CONTACTS,
                operation: "REMOVE",
                familyName: vFamilyName,
                givenName: vGivenName,
                phoneNumber: vPhoneNumber,
                ffosId: inRemovedContactID };
    send_message(msg);
    gContacts[inRemovedContactID] = null;
  } else {
    if (DEBUG_CONTACT) console.log(TAG + "DID NOT find removed contact, doing nothing", inRemovedContactID);
  }
}

// find all contacts in the mozContacts DB and try updating the local DB
// with each one in turn, see compareOneContact()
function compareAllContacts() {

  if (DEBUG_CONTACT) console.log(TAG + "compareAllContacts: " + Object.keys(gContacts).length);

  var allContacts = navigator.mozContacts.getAll({sortBy: "familyName", sortOrder: "descending"});

  // TODO: maintain list of all known contact id's
  // TODO: make copy of that list, mark off the ones found during this iteration
  // TODO: delete the contacts whose id's are in our DB but not listed in these search results
  allContacts.onsuccess = function(event) {
    var cursor = event.target;

    if (cursor.result) {
      var updated = Date.parse(cursor.result.updated);
      if (DEBUG_CONTACT) console.log(TAG + "cursor.update:" + updated + " gLastUpdate:" + gLastUpdate + " date.now():" + Date.now());
      if (updated > gLastUpdate) {
        compareOneContact(cursor.result);
      }
      else {
        gContacts[cursor.result.id] = cursor.result;
      }
      cursor.continue();
    }
    else {
      if (DEBUG_CONTACT) console.log(TAG + "send syncdone");
      var msg = { type: CONTACTS,
                  operation: "SYNCDONE"};
      send_message(msg);
    }
  };

  allContacts.onerror = function(e) {
    console.log(TAG + "error " + e);
  };

  //gLastUpdate = Date.now();
  if (DEBUG_CONTACT) console.log(TAG + "lastupdat=" + gLastUpdate);
}

// for a given ID, find that contact in the mozContacts DB; if found, try updating the local DB
function getAndUpdateContactByID(inID) {
  if (DEBUG_CONTACT) console.log(TAG + "getAndUpdateContactByID", inID);

  var options = {
    filterValue: inID,
    filterBy: ['id'],
    filterOp: 'equals'
  }

  var search = navigator.mozContacts.find(options);

  search.onsuccess = function() {
    if (search.result.length === 1) {
      compareOneContact(search.result[0]);
    } else {
      console.log(TAG + search.result.length + " contact(s) for ID, cannot update");
    }
  };

  search.onerror = function() {
    console.log(TAG + "could not search contacts for ID");
  };
}

function recordContactByID(inID) {
  if (DEBUG_CONTACT) console.log(TAG + "recordContactByID", inID);

  var options = {
    filterValue: inID,
    filterBy: ['id'],
    filterOp: 'equals'
  }

  var search = navigator.mozContacts.find(options);

  search.onsuccess = function() {
    if (search.result.length === 1) {
      gContacts[search.result[0].id] = search.result[0];
    } else {
      console.log(TAG + search.result.length + " contact(s) for ID, cannot be recorded");
    }
  };

  search.onerror = function() {
    console.log(TAG + "record: could not search contacts for ID");
  };
}
// find all existing alarms for this app and removes them.
// useful because alarms accumulate during development
function killOldAlarms() {
  var oldAlarmsRequest = navigator.mozAlarms.getAll();

  oldAlarmsRequest.onsuccess = function() {
    this.result.forEach(function(alarm) {
      if (DEBUG_CONTACT) console.log(TAG + "found alarm", alarm.id);
      navigator.mozAlarms.remove(alarm.id);
    });
  }

  oldAlarmsRequest.onerror = function() {
    console.log(TAG + "could not retrieve old alarms");
  }
}

// schedule an alarm for some number of mSec in the future
function scheduleAlarm(inDelta) {
  if (DEBUG_CONTACT) console.log(TAG + "scheduleAlarm");
  var myDate  = new Date(Date.now() + inDelta);

  // This is arbitrary data pass to the alarm
  var data = {
    foo: "bar"
  }

  // TODO: probably these don't get unregistered! sorcerer's apprentice!
  var request = navigator.mozAlarms.add(myDate, "honorTimezone", data);

  request.onsuccess = function () {
    // console.log("The alarm has been scheduled");
  };

  request.onerror = function () {
    console.log(TAG +"An error occurred: " + this.error.name);
  };

}

function startContactsUpdate() {
//window.addEventListener('DOMContentLoaded', function() {

  // We'll ask the browser to use strict code to help us catch errors earlier.
  // https://developer.mozilla.org/Web/JavaScript/Reference/Functions_and_function_scope/Strict_mode
  'use strict';

  console.log(TAG + "startContactsUpdate");

  var msg = { type: CONTACTS,
              operation: "GETSYNC"};
  send_message(msg);
  // on launch, update all contacts
  //compareAllContacts();

  // listen for contact changes
  navigator.mozContacts.addEventListener('contactchange', function(updateEvent) {
    if (updateEvent.reason == 'remove') {
      if (gContactsDirection[updateEvent.contactID] == ANDROID2FFOS) {
         if (DEBUG_CONTACT) console.log(TAG +"contact removed echo in Firefox OS");
         gContactsDirection[updateEvent.contactID] = '';
      }
      else {
         if (DEBUG_CONTACT) console.log(TAG +"contact removed from Firefox OS");
         removeOneContactByID(updateEvent.contactID);
      }
    } else if (updateEvent.reason == 'create') {
      if (gContactsDirection[updateEvent.contactID] == ANDROID2FFOS) {
         if (DEBUG_CONTACT) console.log(TAG +"contact created echo in Firefox OS");
         gContactsDirection[updateEvent.contactID] = '';
         recordContactByID(updateEvent.contactID);
      }
      else {
         if (DEBUG_CONTACT) console.log(TAG +"contact created in Firefox OS");
         getAndUpdateContactByID(updateEvent.contactID);
      }
    } else if (updateEvent.reason == 'update') {
      if (gContactsDirection[updateEvent.contactID] == ANDROID2FFOS) {
         if (DEBUG_CONTACT) console.log(TAG +"contact updated echo in Firefox OS");
         gContactsDirection[updateEvent.contactID] = '';
         recordContactByID(updateEvent.contactID);
      }
      else {
         if (DEBUG_CONTACT) console.log(TAG +"contact updated in Firefox OS");
         getAndUpdateContactByID(updateEvent.contactID);
      }
    } else {
      if (DEBUG_CONTACT) console.log(TAG +"unknown update reason " + updateEvent.reason);
    }
  });

  // respond to alarms by updating all contacts and then scheduling another alarm
  navigator.mozSetMessageHandler("alarm", function (mozAlarm) {
    killOldAlarms();
    if (gLastUpdate > 0) {
       console.log(TAG + "last sync = " + gLastUpdate);
       compareAllContacts();
    }
    else {
       // note: artificially short time for demo
       scheduleAlarm(5000);
    }
  });

  // schedule the first alarm
  // note: artificially short time for demo
  scheduleAlarm(5000);
}

window.onload = onLoad;
window.onblur = onBlur;
window.onfocus = onFocus;
window.onunload = onUnload;
window.onlanguagechange = onLanguageChange;
document.addEventListener("visibilitychange", onVisibilityChange, false);
document.addEventListener("touchstart", handleTouchEvent, false);
document.addEventListener("touchmove", handleTouchEvent, false);
document.addEventListener("touchend", handleTouchEvent, false);

