/*global Util: true */

/**
 * Defines the Draggable class which makes an element draggable.
 * Based on code from Safari Developer Library.
 * 
 * @param {Element} elem an html element to make draggable - should be relative/fixed/absolutely positioned.
 * @param {function} dropCallBack called when the element is dropped, passed whether or not the drop was successful
 *        followed by this.
 * @param {function} canDropCallBack returns a boolean stating whether the element can be dropped
 *        at the current position, the element, which is passed in.
 * @param {function} onDragStart the function to call when the piece is started to be dragged. Passed the
 *        draggable DOM element.
 */
function Draggable(elem, dropCallBack, canDropCallBack, onDragStart){
    this.elem = elem;
    
    //get positions - simple calculation and assume it doesn't move..
    this.left = elem.offsetLeft;
    this.top = elem.offsetTop;
 
    this.dropCallBack = dropCallBack;
    this.canDropCallBack = canDropCallBack;
    this.touchEvents = Modernizr.touch;
    this.x = 0; //initial position assumed to be 0,0.
    this.y = 0;
    this.width = elem.offsetWidth; //simple calc and cache these for speed.
    this.height = elem.offsetHeight;
    this.ableToDrop = false; //if the drop can be made that the current position.
    this.canBeMoved = true; //false when dropping and moving back to prevent it being dragged again.

    //define event handlers..
    var that = this;
    var dragStart, dragMove, drop;

    dragStart = function(e){
        if(e.originalEvent)
            e = e.originalEvent;

        e.preventDefault(); //prevent native drag and drop..
		e.stopPropagation();
    
        if(!that.canBeMoved) {
            return;
        }

        if(e.targetTouches){
            if(e.targetTouches.length != 1){
            	return;
            }
        } else {
            e.targetTouches = [e]; //so it works for mouse too..   
        }
      
        //add listeners for other events (make it work on desktop too)..
        if(that.touchEvents){
            //TODO: this has broken the phone version..
            $(that.elem).on('touchmove', dragMove).on('touchend', drop);
        } else {
            $(document).on('mousemove', dragMove).on('mouseup', drop);
        }

        //Store positions..
        that.lastEventX = e.targetTouches[0].clientX;
        that.lastEventY = e.targetTouches[0].clientY;
        
        if(onDragStart) onDragStart(that.elem);
    };
    
    dragMove = function(e){
        if(e.originalEvent)
            e = e.originalEvent;

        e.preventDefault();
		e.stopPropagation();
        
        //Just track a single touch as more is a gesture..
        if(e.targetTouches){
            if(e.targetTouches.length != 1){
            	return;
            }
        } else {
            e.targetTouches = [e]; //so it works for mouse too..   
        }
        
        //update position vars..
        var leftDelta = e.targetTouches[0].clientX - that.lastEventX;
        var topDelta = e.targetTouches[0].clientY - that.lastEventY;      
        that.lastEventX = e.targetTouches[0].clientX;
        that.lastEventY = e.targetTouches[0].clientY;
        
        //actually move the element and update x and y..
		that.setPosition(that.x + leftDelta, that.y + topDelta);
           
        //Call the can drop element with it's position..
        if(that.canDropCallBack){
            var top = that.elem.offsetTop + that.y;
            var left = that.elem.offsetLeft + that.x;
            that.ableToDrop = that.canDropCallBack(top, left + that.width, top + that.height, left, this);
        }
    };

	//e - the event which caused the drop - if this is null assumes it's a forced
	//    drop.
	drop = function(e){
		var passedEvent = e,
			forced = typeof e === "undefined";

        if(e && e.originalEvent)
            e = e.originalEvent;

        // Prevent the browser from doing its default thing (scroll, zoom)
        if(e && e.preventDefault && e.stopPropagation){
        	e.preventDefault();
        	e.stopPropagation();
        }
        
        // Stop tracking when the last finger is removed from this element
        if (e && e.targetTouches && e.targetTouches.length > 0)
            return;

        if(that.touchEvents){
        	$(that.elem).off('touchmove', dragMove).off('touchend', drop);
        } else {
            $(document).off('mousemove', dragMove).off('mouseup', drop);
        }

        //revert the drag or call drop if able to or no event exists..
        if(!e || !that.ableToDrop){
            //move back to initial position slowly..
            that.canBeMoved = false; 
            that.setPosition(0, 0, true, function(){
            	that.canBeMoved = true;
            });
            if(that.dropCallBack) that.dropCallBack(false, forced);
        } else if(that.dropCallBack){
            that.dropCallBack(true, forced);
        }
    };

    //register on first touch..
    $(this.elem).on(this.touchEvents ? 'touchstart' : 'mousedown', dragStart);
	
	//add public method to allow dropping of piece..
	this.forceDrop = function(){
		drop();
	};
}

/**
 * Sets the position of this draggable element relative to where it started
 * using css3 transitions and updates it's x and y positions.
 * 
 * @param x {Number} the x location to move this draggable to.
 * @param y {Number} the y location to move this draggable to.
 * @param slow {boolean} default to false, to move the element slowly (taking 1/4 seconds)
 * @param onComplete {Function} a function to call when the movement is complete.
 */
Draggable.prototype.setPosition = function(x, y, slow, onComplete){

    var that = this;
    var done = false;

    //when the transition is over tidy up..
    var transitionEnded = function(){
    	done = true;
        //reset values..
        that.x = x;
        that.y = y;
		var thatStyle = that.elem.style;
		thatStyle.transitionDuration = '0ms';
        thatStyle.transform = 'translate3d(' + x + 'px, ' + y + 'px, 0)';
        $(that.elem).off('transitionEnd', transitionEnded);
        //call the given function..
        if(onComplete)
            onComplete();
    };
    
    if(slow){
        $(this.elem).on('transitionEnd', transitionEnded);
        this.elem.style.transitionDuration = '250ms'; 
        this.elem.style.transform = 'translate3d(' + x + 'px, ' + y + 'px, 0)';
   
        //the transitionEnd event doesn't always fire..
        setTimeout(function(){
            if(!done){
                transitionEnded();
            }
        }, 260);
    } else { //fast mode..
    	this.elem.style.transitionDuration = '0ms';
        this.elem.style.transform = 'translate3d(' + x + 'px, ' + y + 'px, 0)';
        this.x = x;
        this.y = y;
        if(onComplete){
            onComplete();
        }
    }
};

/**
 * Returns the current position of this draggable, given as an object with "top",
 * "right", "bottom" and "left" properties. 
 */
Draggable.prototype.getPosition = function(){
    var that = this;
    return {
        top: that.y + that.top,
        right: that.x + that.left + that.width,
        bottom: that.y + that.top + that.height,
        left: that.x + that.left
    };
};

/**
 * Returns the center position of this draggable, given as an object with "x"
 * and "y" properties.
 */
Draggable.prototype.getCenter = function(){
    var that = this;
    return {
        x: that.left + that.x + that.width/2,
        y: that.top + that.y + that.height/2
    };
};

