//Defines the Piece class - which represents pieces that can be moved on the game board,
//Note: when a peice is fixed it is no longer considered to be a Piece.
var Piece;
(function(){
    "use strict";

    var touchEvents = Modernizr.touch;

	/*
	 * Returns a randomly generated 3*3 grid.
	 */
	function getNewGrid(){
	    function getRandomSolidColourIndex(){
	        var colourIndex = Math.round((Puxxle.cssColours.length-1) * Math.seededRandom());
	        if(colourIndex === 0)
	            colourIndex = Puxxle.cssColours.length-1;
	        return colourIndex;
	    }

	    //pieces are 3*3 arrays..
		var grid = [[0,0,0],[0,getRandomSolidColourIndex(),0],[0,0,0]];
		 
		//50% chance of cross bits being
		if(Math.seededRandom() > 0.5)
		    grid[0][1] = getRandomSolidColourIndex(); //top
		if(Math.seededRandom() > 0.5)
		    grid[1][2] = getRandomSolidColourIndex(); //right
		if(Math.seededRandom() > 0.5)
		    grid[2][1] = getRandomSolidColourIndex(); //bottom
		if(Math.seededRandom() > 0.5)
		    grid[1][0] = getRandomSolidColourIndex(); //left
		
		//50% chance of corners if at least one of the cross bits is a 1..
		if(grid[0][1] && grid[1][0] && Math.seededRandom() > 0.5)
		    grid[0][0] = getRandomSolidColourIndex(); //top left
		if(grid[0][1] && grid[1][2] && Math.seededRandom() > 0.5)
		    grid[0][2] = getRandomSolidColourIndex(); //top right
		if(grid[2][1] && grid[1][2] && Math.seededRandom() > 0.5)
		    grid[2][2] = getRandomSolidColourIndex(); //bottom right
		if(grid[2][1] && grid[1][0] && Math.seededRandom() > 0.5)
		    grid[2][0] = getRandomSolidColourIndex(); //bottom left

		return grid;
	}

	/**
	 * Defines the peice class which wraps a 3*3 table and controls it's styling and logic.
	 *
	 * @param tableEl {HTMLTableElement} a 3 by 3 table.
	 */
	Piece = function(tableEl){
		this.tableEl = tableEl;
		
		//simple calc positional values..
		this.left = tableEl.offsetLeft;
		this.top = tableEl.offsetTop;
		this.allowRotate = true; //determines if this piece is allowed to rotate or not..
		
		this.nextGrid = getNewGrid(); //the next grid this will use..
		this.resetGrid(); //adds the "grid" object to this and gets the table cell classes accordingly. 
		
		//attach the rotating event..
		var that = this;
		var onClickHandler = function(event){
			that.rotateClockwise();
			event.stopPropagation(); //prevent bubble..
		};
		$(this.tableEl).on(touchEvents ? 'touchstart' : 'mousedown', onClickHandler);
	};

	/**
	 * Prevents the piece rotating by removing the associated event handler.
	 */
	Piece.prototype.preventRotate = function(){
		this.allowRotate = false;
	};
	
	/**
	 *	Allows the piece to rotate when pressed.
	 */
	Piece.prototype.makeRotate = function(){
		this.allowRotate = true;
	};

	/**
	 * Gives this piece a new random grid.  This function sets the grid to be a new random
	 * double array and updates the table cell's css class names accordingly.
	 */
	Piece.prototype.resetGrid = function(){
		this.grid = this.nextGrid;
		this.nextGrid = getNewGrid();
        this._matchTableWithGrid();
    };

    /** private */

	Piece.prototype._matchTableWithGrid = function(){
        var rows = this.tableEl.rows;
		for(var i = 0; i < 3; i++){
            for(var j = 0; j < 3; j++){
                rows[i].cells[j].className = Puxxle.cssColours[this.grid[i][j]];
            }
        }
    };
    
    /**
     * Returns the css class that gives the colour to the grid 
     * cell with the given value.
     */
    Piece.prototype.getCssColourForGrid = function(cellValue){
        return Puxxle.cssColours[cellValue];
    };
    
    /**
     * Returns a new copy of the 3x3 grid presenting this piece.
     */
    Piece.prototype.getCopyOfGrid = function(){
        var copy = new Array(3);
        for(var i = 0; i < 3; i++){
            copy[i] = new Array(3);
            var rowToCopy = this.grid[i];
            for(var j = 0; j < 3; j++){
                copy[i][j] = rowToCopy[j];
            }
        }
        return copy;
    };

	/**
	 * Returns a new copy of the 3x3 grid presenting the next grid for
	 * this piece.
	 */
	Piece.prototype.getCopyOfNextGrid = function(){
		var copy = new Array(3);
        for(var i = 0; i < 3; i++){
            copy[i] = new Array(3);
            var rowToCopy = this.nextGrid[i];
            for(var j = 0; j < 3; j++){
                copy[i][j] = rowToCopy[j];
            }
        }
        return copy;
    };

    /**
     * Rotates the piece by 90 degrees clockwise if allowRotate is true.
     */
    Piece.prototype.rotateClockwise = function(){
		if(!this.allowRotate) return; //only do it if allowed..
        
        var rotated = new Array(3);
        for(var i = 0; i < 3; i++){
            rotated[i] = new Array(3); 
            for(var j = 0; j < 3; j++){
                rotated[i][j] = this.grid[2-j][i];
            }
        }
        this.grid = rotated;
        this._matchTableWithGrid();
    	soundManager.playSfx('turn');
    };

})();

