/**
 * Defines the grid onto which elements can be dropped.
 */
var PuxxleGrid = function(tableEl){

    this.numRows = tableEl.rows.length;
    this.numColumns = tableEl.rows[0].cells.length;
    this.pathLengthToScore = 4; //min number of adjacent cells to score points..

    //store dom and internal representation of the table state..
    this.tableEl = tableEl;
    this.tableState = new Array(this.numRows);
    for(var row = 0; row < this.numRows; row++){
        this.tableState[row] = new Array(this.numColumns);
        for(var col = 0; col < this.numColumns; col++){
            this.tableState[row][col] = 0; //initially make a grid of zeros (blanks).
        }
    }
    //set initial positions..
    this.updatePosition();
};

(function(){
    //returns if the actual range intersects the drop range..
    function intersectsDrop(actMin, actMax, dropMin, dropMax){
        if(dropMin - actMin >= 0){
            return (actMax - dropMin >= 0);
        } else {
            return (dropMax - actMin >= 0);
        }
    }

    //returns whether or not this dropzone intersects the given Draggable..
    PuxxleGrid.prototype.intersects = function(draggable){
        var dragPos = draggable.getPosition();
        var intersects =  intersectsDrop(dragPos.top, dragPos.bottom, this.top, this.bottom) &&
                          intersectsDrop(dragPos.left, dragPos.right, this.left, this.right);
        return intersects;
    };
    
    //This class caches positions and widths of the elements for speed,
    //if these change, this needs to be called to reset them..
    PuxxleGrid.prototype.updatePosition = function(){
        //cache position vars - simple calc..
        this.left = this.tableEl.offsetLeft;
        this.top = this.tableEl.offsetTop;
        this.width = this.tableEl.offsetWidth;
        this.height = this.tableEl.offsetHeight;
        this.right = this.left + this.width;
        this.bottom = this.top + this.height;
        
        this.pixelsPerCol = this.height/this.numColumns;
        this.pixelsPerRow = this.width/this.numRows;
    };

    //returns the column and row of the cell that the given piece is intersecting
    //or nearest..
    //note that this function can return negative numbers in the
    //case that the top left hand edge should be moved over a position
    //outside the grid itself (to the left or the top)..
    //accepts both a Draggable instance and a jQuery instance..
    PuxxleGrid.prototype.getIntersectingGridPos = function(draggable){
        var dragPos = draggable.getPosition ? draggable.getPosition() : draggable.offset();

        var xDiff = dragPos.left - this.left + Puxxle.pieceHolderMargin;
        var yDiff = dragPos.top - this.top + Puxxle.pieceHolderMargin;

        var column = Math.round(xDiff / this.pixelsPerCol);
        column = (column < -2) ? -2 : column; //cope with overflow to the left..
        column = (column >= this.numColumns) ? this.numColumns -1 : column; //need to cope with overflow to the right..

        var row = Math.round(yDiff / this.pixelsPerRow); 
        row = (row < -2) ? -2 : row;  //cope with overflow to the top..
        row = (row >= this.numRows) ? this.numRows - 1 : row; //cope with overflow to the bottom..

        return { row: row, column: column };
    };

    /**
     * Returns whether or not the given PieceHolder holding the given piece is in a state
     * that allows for its piece to be dropped onto this grid.
     */
    PuxxleGrid.prototype.canFitPieceIn = function(pieceHolder, piece){
        if(this.intersects(pieceHolder)){ //Note: can make it more effficient by doing two together..
            var dragPos =  this.getIntersectingGridPos(pieceHolder);
            //check that the cells of the peice are ok can be put over this grid..
            for(var i = 0, col = dragPos.column; i < 3; i++, col++){
                for(var j = 0, row = dragPos.row; j < 3; j++, row++){
                    var pieceCell = piece.grid[j][i];
                    //case that the cell is off the edge of the main grid..
                    if(col < 0 || col >= this.numColumns || row < 0 || row >= this.numRows){
                        if(pieceCell){
                            return false;
                        }
                    } else if(pieceCell && this.tableState[row][col]){ //case that it's overlapping a non-empty cell..
                        return false;
                    }
                }
            }
            return true;
        }
        return false;
    };

    /**
     * Adds the given price to this grid starting in the top left hand corner.
     * Assumes that canFitPieceIn returns true.
     */
    PuxxleGrid.prototype.addPieceToGrid = function(piece, topLeftPos){
        for(var i = 0, col = topLeftPos.column; i < 3; i++, col++){
            for(var j = 0, row = topLeftPos.row; j < 3; j++, row++){
                var pieceCell = piece.grid[j][i];
                if(pieceCell){
                    this.tableState[row][col] = pieceCell; //update internel rep.
                    this.tableEl.rows[row].cells[col].className = piece.getCssColourForGrid(pieceCell); //update dom..
                }
            }
        }
    };

    /**
     * Returns if it's game over or not, optional takes the piece which has
     * just been dropped.  In the case that a piece is passed, it bases the
     * result on the "next value" of that piece.
     */
    PuxxleGrid.prototype.isGameOver = function(droppedPiece){

        var pieces = window.pieces; //TODO: odd use of global..
        var tableState = this.tableState;
        
        //returns a new grid which is the same as the given one but rotated by 90 degrees..
        function rotateState(pieceState){
           var rotated = new Array(3);
            for(var i = 0; i < 3; i++){
                rotated[i] = new Array(3); 
                for(var j = 0; j < 3; j++){
                    rotated[i][j] = pieceState[2-j][i];
                }
            }
            return rotated;
        }
        
        //Returns whether or not the 3*3 state would fit at the given position
        //if that were it's center..
        function canFitInCenter(pieceState, row, col){
            for(var i = 0; i < 3; i++){
                var pieceRow = pieceState[i];
                var rowNum = row - 1 + i;
                var tableRow = tableState[rowNum]; //can be undefined..
                for(var j = 0; j < 3; j++){
                    if(pieceRow[j]){
                        //is it a position on the grid..
                        var colNum = col - 1 + j;
                        if(rowNum < 0 || colNum < 0)
                            return false;
                        if(rowNum >= tableState.length || colNum >= tableRow.length)
                            return false;
                        //is it a taken position..
                        if(tableRow[colNum])
                            return false;
                    }
                }
            }
            return true;
        }

        var pieceStates = [];
        for(var i = 0, len = pieces.length; i < len; i++){
        	var piece = pieces[i];
        	var state = piece === droppedPiece ?
	    		piece.getCopyOfNextGrid() : piece.getCopyOfGrid();
            pieceStates.push(state);
        }

        //Note: currently simply does all positions, all pieces, all rotations..
        for(i = 0; i < 4; i++){
            for(var row = 0, rows = this.numRows; row < rows; row++){
                var stateRow = tableState[row];
                for(var col = 0, cols = this.numColumns; col < cols; col++){
                    var cell = stateRow[col];
                    if(cell) continue; //skip full cells..
                    for(var p = 0, numPieces = pieceStates.length; p < numPieces; p++){
                        var pieceState = pieceStates[p];
                        //see if the selected cell could be center of piece..
                        if(canFitInCenter(pieceState, row, col))
                            return false;  
                    }
                    
                }
            }
            //rotate all pieces, don't bother with the last time..
            if(i < 3){
                for(var p = 0, numPieces = pieceStates.length; p < numPieces; p++){
                    pieceStates[p] = rotateState(pieceStates[p]);
                }
            }
        }
        return true;
    }
    
    /**
     * Ensures that the internal representation of the gird matches the HTML
     * table.  This should be called when the classes of the table are edited outside
     * this class.
     */
    PuxxleGrid.prototype.resetGridState = function(){
        for(var row = 0; row < this.numRows; row++){
            this.tableState[row] = new Array(this.numColumns);
            for(var col = 0; col < this.numColumns; col++){
                var cellClass = this.tableEl.rows[row].cells[col].className;
                if(cellClass.indexOf("colour") >= 0){
                    this.tableState[row][col] = 1;
                } else {
                    this.tableState[row][col] = 0;
                }
            }
        }
    };
    
    //Sets the grid back to it's initial empty state..
    PuxxleGrid.prototype.emptyGrid = function(){
        for(var row = 0; row < this.numRows; row++){
            this.tableState[row] = new Array(this.numColumns);
            for(var col = 0; col < this.numColumns; col++){
                this.tableEl.rows[row].cells[col].className = "";
                this.tableState[row][col] = 0;
            }
        }
    };
    
    /**
     * Returns the [x,y] position of the given grid pos, which represents a position
     * to which a piece can be hauled.  A grid pos coordinate can be negative representing
     * the fact that a piece can be hauled to a position to the outside the grid itself.
     * The gridPos object must contain 'row' and 'column' properties as returned by 
     * getIntersectingGridPos.  
     */
    PuxxleGrid.prototype.getHaulPosition = function(gridPos){
    	//haul position is the top left hand edge to push a piece to,
        //in the case that the piece needs to be positioned over the
        //left or top of the cell the xAdjustment and yAdjustment should be added..
        var adjustedGridPos = { row: gridPos.row, column: gridPos.column };
        var yAdjustment = 0;
        if(gridPos.row < 0){ //case when the matching row is the top of the main grid..
            yAdjustment = this.pixelsPerRow * gridPos.row;
            adjustedGridPos.row = 0;
        }
        
        var xAdjustment = 0;
        if(gridPos.column < 0){ //case when the matching row is off to the left of the main grid..
            xAdjustment = this.pixelsPerCol * gridPos.column;
            adjustedGridPos.column = 0;
        }

        //get the nearest actual cell..
        var cell = this.tableEl.rows[adjustedGridPos.row].cells[adjustedGridPos.column];

        //use simple calc for position - note that tables are considered relative so you need to add the table offsets too..
        return {
        	x: cell.offsetLeft + this.left - Puxxle.pieceHolderMargin + xAdjustment,
    		y: cell.offsetTop + this.top - Puxxle.pieceHolderMargin + yAdjustment
    	};
    };
    
    
    /**
     * Hauls in the given piece holder, holding the given piece.  Picks the closest
     * cell to the top left and works from there.  Assumes that canFitPieceIn returns true.
     * Passed the event that event that triggered this, if this event as a row, column it
     * uses that cell instead of calculating (for playback purposes).
     * Executes onComplete when the haul in animation is complete is it is present.
     */ 
    PuxxleGrid.prototype.haulIn = function(pieceHolder, piece, onComplete){        
        var gridPos = this.getIntersectingGridPos(pieceHolder);
		var haulPosition = this.getHaulPosition(gridPos);
		
        //use simple calc for position - note that tables are considered relative so you need to add the table offsets too..
        var that = this;
        pieceHolder.setPosition(
            haulPosition.x - pieceHolder.left,
            haulPosition.y - pieceHolder.top,
            true,
            //onComplete add the actual piece to this grid..
            function(){
                //pass the original grid pos in as that's the one which needs to be used when updating the grid..
                that.addPieceToGrid(piece, gridPos);
                if(onComplete)
                    onComplete();
            }
        );
    };
    
    /**
     * Returns a list of table cells which will generate a score.  The logic is
     * that there needs to be a path of length - scoringPathLength.
     */ 
    PuxxleGrid.prototype.getScoringCells = function(){
       var that = this;

       //Simple function to return the entry for a cell which
       //is in the given row and column..
       function getCellEntry(row, col){
           return "[" + row + "," + col + "]";
       }
       //Inverse of above function..
       function getRowCol(cellEntry){
           var pos = cellEntry.slice(1,-1).split(",");
           pos[0] = parseInt(pos[0]);
           pos[1] = parseInt(pos[1]);
           return pos;
       }
       //Return the colour class or undefined if it doesn't exist..
       function getColourFromClass(className){
           var startIndex = className.indexOf('colour');
           //check that this is a coloured cell which is not currently scoring..
           if(startIndex > -1 && className.indexOf('score') < 0){
               className = className.substr(startIndex);
               var endIndex = className.indexOf(' ');
               if(endIndex > -1){
                   className = className.substr(0, endIndex);
               }
               return className;
           }
       }
       //Returns a map of "colour" -> {[i,j]" -> HTMLTableCell} which are solid.
       //Ensures each cell's className only needs to be looked at once.
       function getCellsByColour(){
           var colours = {};

           var table = that.tableEl;
           for(var col = 0, len = table.rows[0].cells.length; col < len; col++){
               for(var row = 0, rowLen = table.rows.length; row < rowLen; row++){
                   var cell = table.rows[row].cells[col];
                   var cellColour = getColourFromClass(cell.className);
                   if(cellColour){
                        var sameColour = colours[cellColour];
                        if(sameColour == undefined)
                           sameColour = colours[cellColour] = {};
                       sameColour[getCellEntry(row, col)] = cell;
                   }
               }
           }
           return colours;
       }
       //Returns a list of the adjacent cell entries..
       function getAdjacentCellEntries(cellEntry){
           var pos = getRowCol(cellEntry);

           //4 possible options..
           var adjacent = [];
           var row = pos[0];
           var col = pos[1];
           if(row > 0)
               adjacent.push(getCellEntry(row-1, col));
           if(row < that.numRows-1)
               adjacent.push(getCellEntry(row+1, col));
           if(col > 0)
               adjacent.push(getCellEntry(row, col-1));
           if(col < that.numColumns-1)
               adjacent.push(getCellEntry(row, col+1));

           return adjacent;
       }

       var paths = []; //all connected paths..
       var coloursMap = getCellsByColour();
       var colours = Object.keys(coloursMap);
       while(colours.length > 0){
           var cellsWithSameColour = coloursMap[colours.pop()];
           var toCheck = Object.keys(cellsWithSameColour);

           while(toCheck.length > 0){
               var toExpand = [toCheck.pop()];
               var currentPath = [toExpand[0]];
               while(toExpand.length > 0){
                   var next = toExpand.pop();
                   var adjacentCellEntries = getAdjacentCellEntries(next);
                   adjacentCellEntries.forEach(function(cellEntry){
                       if(toCheck.indexOf(cellEntry) !== -1){
                           toExpand.push(cellEntry);
                           currentPath.push(cellEntry);
                           toCheck = toCheck.filter(function(i){
                               return i !== cellEntry;
                           });
                       }
                   });
               }
               paths.push(currentPath);
           }
       }
  
       //return cells from all the scoring paths..
       var scoringCells = [];
       paths.forEach(function(path){
           if(path.length >= that.pathLengthToScore){
               path.forEach(function(cellEntry){
                   var pos = getRowCol(cellEntry);
                   scoringCells.push(that.tableEl.rows[pos[0]].cells[pos[1]]);    
               });
           }
       });
       return scoringCells;
    };
  
})();

