/**
 * Defines the Util singleton.
 */
var Util = {

    /**
     * Performs a binary search on the given array - which should be sorted
     * and returns the index of the element which is equal to the given value.
     * If not such value exists the value which is the closest by smaller is
     * given, if no smaller values exists, -1 is returned.
     */
     binaryFloor: function(sortedArr, comparator, value){
        var left = 0;
        var right = sortedArr.length - 1;
        while (left <= right){
            var mid = Math.round((left + right)/2);
            var comp = comparator(sortedArr[mid], value);
            if(comp === 0)
                return mid;
            else if (comp < 0)
                left = mid + 1;
            else
                right = mid - 1;
        }
        return right;
     },
          
     /**
      * Inserts the given value into the given sortedArray using the given
      * comparator.
      */
     insertToSortedArray: function(sortedArr, comparator, value){
         var position = this.binaryFloor(sortedArr, comparator, value);
         sortedArr.splice(position + 1, 0, value);
     },
	
	/**
	 * Returns whether the two values are within epsilon of each other.
	 * 
	 * @param x the first value
	 * @param y the second value
	 * @param epsilon the amount of error in the calculation allowed.
	 */
	isEqual: function(x, y, epsilon){
		return x > y ? x < y + epsilon : x + epsilon > y; 
	},
	
	/**
	 * Binds the given event to occur when the given element fires a transform end event.
	 * In the case that the event doesn't fire, it is triggered using a timeout after the
	 * given time plus a grace period of 100ms.
	 */
	bindToTransitionEndForSingleRun: function ($el, funcToExec, maxMSTillTransitionEnd) {
		var fired = false,
			timeout;
		var wrappedFunc = function () {
            if(fired) return;
			fired = true;
			clearTimeout(timeout);
			$el.unbind('transitionEnd', wrappedFunc);
			funcToExec();
		};
		$el.bind('transitionEnd', wrappedFunc);
		timeout = setTimeout(wrappedFunc, maxMSTillTransitionEnd + 100);
	},
	
	/**
	 * Attempts to retreive the JSON object from the local storage using
	 * the given key.  If nothing can be found, or parsing fails, the given
	 * default value is returned.
	 */
	getFromStorage: function(key, notFoundDefault){
		var localItem = localStorage.getItem(key);
		var toReturn;
		if(localItem) {
			try {
				toReturn = JSON.parse(localItem);
			} catch (e) {
				toReturn = notFoundDefault;
			}
		} else {
			toReturn = notFoundDefault;
		}
		return toReturn;
	},
	
	/**
	 * Stores the given value in local storage with the given key.
	 * returns whether or not the operation was a success.
	 */
	setInStorage: function(key, value) {
		var str;
		try {
			str = JSON.stringify(value);
			localStorage.setItem(key, str);
			return true;
		} catch (e) {
			return false;
		}
	},
	
	/**
	 * Formats the given number with commas, currently only works with +ive integers.
	 */
	formatNumber: function(number) {
		number = "" + number;
		if(number.length < 4){
			return number
		} else {
			var newNumber = "";
			for(var i = 0, len = number.length; i < len ; i++) {
				newNumber = number.charAt(len - i - 1) + newNumber;
				if (i > 0 && i < len - 1 && !((i + 1) % 3))
					newNumber = "," + newNumber;
			}
			return newNumber;
		}
	}
	
};

