(function() {
  /* global module, device, LocalFileSystem, console, Connection */
  "use strict";

  // TODO: event key to the url through options
  var EDUSERVER_URL = "http://eduevents.mobileicecube.com/api/v1/";
  var LOG_DIR_NAME = "_eduevents-log";

  var icecube = {};

  // Utility functions useful inside the Icecube components and accessible for
  // outsiders through IceCube.Utils
  var Utils = {};
  // Extend the given object with the properties from the rest of the arguments.
  // Properties in later arguments will overwrite properties from previous args.
  // Examples:
  //      > var obj = {}
  //      undefined
  //      > ice.Utils.extend(obj, {a: 1, b: 2})
  //      { a: 1, b: 2 }
  //      > obj
  //      { a: 1, b: 2 }
  //      > ice.Utils.extend({a: 1}, {b: 1}, {c: 1}, {d: 1})
  //      { a: 1, b: 1, c: 1, d: 1 }
  //      > ice.Utils.extend({}, {b: 1});
  //      { b: 1 }
  //      > ice.Utils.extend({a: 1, b: 1}, {a: 2, b: 2}, {b: 3})
  //      { a: 2, b: 3 }
  Utils.extend = function(obj) {
    Array.prototype.slice.call(arguments, 1).forEach(function(source) {
      if (source) {
        for (var prop in source) {
          obj[prop] = source[prop];
        }
      }
    });
    return obj;
  };
  // expose the Utils
  icecube.Utils = Utils;


  var EventHandler = function(edu, startTime) {
    this.edu = edu;
    this._startTime = startTime;
    this.init();
  };
  EventHandler.prototype = {
    _setLastActivityInterval: function() {
      // set interval to update lastActivity every X seconds (from pref?)
      this._interval = setInterval(function() {
        localStorage.setItem("lastActivity", new Date().getTime());
      }, 2000);
    },
    _logLastActivity: function() {
      // if we have a previous session (and it's lastActivity), log the close event
      var lastActivity = localStorage.getItem("lastActivity");
      if (lastActivity) {
        this.edu.logAppEvent("close", {}, new Date(JSON.parse(lastActivity)).getTime());
      }
    },
    init: function() {
      // set the initial state of online/offline
      var online = (navigator.connection.type !== Connection.NONE);
      this.edu.online(online);

      this.edu.initialize();

      // if we don't have a device "uuid", generate it and log newinstall event
      if (!localStorage.getItem("eduevents-uuid")) {
        var uid = new Date().getTime() + "" + Math.floor(Math.random()*10);
        localStorage.setItem("eduevents-uuid", uid);
        this.edu.logAppEvent("newinstall");
      }

      // handle last activity from previous session
      this._logLastActivity();
      // set last activity to now and set it to update
      localStorage.setItem("lastActivity", new Date().getTime());
      this._setLastActivityInterval();

      // data for the ready (=open) event
      var data = { locale: "unknown", online: online}; // default locale is unknown
      // if we know load start time, log loadtime
      if (this._startTime) { data.loadtime = new Date().getTime() - this._startTime.getTime(); }

      var globalCallback = function(loc) {
        // same callback is used for success and error
        if (loc && loc.value) { // only set locale if we have it
          data.locale = loc.value;
        }
        // log the open event
        this.edu.logAppEvent("open", data);
      }.bind(this);
      // check if the globalization plugin is installed
      if (navigator.globalization && navigator.globalization.getLocaleName) {
        navigator.globalization.getLocaleName(globalCallback, globalCallback);
      } else { // if not, simply log the event
          this.edu.logAppEvent("open", data);
      }

      //this.edu.logAppEvent("open", data);

      // set up the proper listeners for pause, resign, resume
      document.addEventListener("pause", this.pause.bind(this), false);
      document.addEventListener("resume", this.resume.bind(this), false);
      document.addEventListener("resign", this.pause.bind(this), false);
      // set up listeners for online and offline
      document.addEventListener("online", this.online.bind(this), false);
      document.addEventListener("offline", this.offline.bind(this), false);
    },
    pause: function() {
      // clear the lastActivity interval
      clearInterval(this._interval);
      //this.edu.logAppEvent("pause"); // don't log anything, resume will handle it
    },
    resume: function() {
      // check and log previous session close
      this._logLastActivity();
      // set interval to update lastActivity every X seconds
      this._setLastActivityInterval();
      this.edu.logAppEvent("resume");
    },
    online: function() {
      // set the isOnline flag to true
      this.edu.online(true);
    },
    offline: function() {
      // set the isOnline flag to false
      this.edu.online(false);
    }
  };

  var EduEvents = function(options) {
    this.options = Utils.extend({logEduEvents: true, logAppEvents: true}, options);
    // set default properties
    this._events = {
      appevents: [],
      eduevents: []
    };
    this._isOnline = false;
    this._eventhandler = new EventHandler(this, this.options.startTime);
  };
  var eduproto = EduEvents.prototype;
  eduproto.initialize = function() {
    // initialize the log_events array from localStorage
    var events = localStorage.getItem("eduevents");
    if (events) {
      this._events = JSON.parse(events);
    }
    if (this.online()) {
      this._syncExistingFiles();
    }
  };
  eduproto.online = function(newState) {
    if (typeof newState !== "undefined") {
      this._isOnline = !!newState; // force to boolean
    }
    return this._isOnline;
  };
  // log an event
  // never call before deviceready!
  // ready, pause, resume, online, offline events will be automatically logged
  eduproto.logEduEvent = function(type, objid, data, tstamp) {
    // add timestamp to event data (if not there already)
    var event = { type: type, objid: objid, data: data || {}, tstamp: tstamp || new Date().getTime() };
    // if we should not log the event, we return it filled with the data
    if (!this.options.logEduEvents) { return event; }
    // add the event data to logged events
    this._events.eduevents.push(event);
    // call function to check whether we should sync
    this._syncIfNecessary();
  };
  eduproto.logAppEvent = function(type, data, tstamp) {
    // add timestamp to event data (if not there already)
    var event = {type: type, data: data, tstamp: tstamp || new Date().getTime() };
    // if we should not log the event, we return it filled with the data
    if (!this.options.logAppEvents) { return; }
    // add the event data to logged events
    this._events.appevents.push(event);
    // call function to check whether we should sync
    this._syncIfNecessary();
  };
  eduproto._postFile = function(file, json, callback) {
    // - send the JSON obj to server
    var xhttp = new XMLHttpRequest({mozSystem: true}),
        self = this;
    // TODO: event key to the url
    xhttp.open("POST", EDUSERVER_URL + this.options.apikey + '/events/');
    xhttp.setRequestHeader("Content-type", "application/json");
    xhttp.onreadystatechange = function() {
      if (xhttp.readyState === 4) { // complete
        if (xhttp.status === 200 || xhttp.status === 0) {
          file.remove(callback, function() {
            self.logAppEvent("error", {msg: "removingFile"});
          });
        }
      }
    };
    xhttp.send(json);
    // - when successful (or 400), remove the file from disk
  };
  eduproto._error = function() {
    //self.logAppEvent("error", {msg: "writeFile"});
  };
  eduproto._writeFile = function(filename, json, callback) {
    var self = this,
        file;
    var haveFileWriter = function(fw) {
      fw.onwriteend = function() {
        if (typeof callback === "function") {
          callback(file);
        }
      };
      fw.write(json);
    };
    var haveFileEntry = function(f) {
      file = f; // set the scope file, to be used later
      f.createWriter(haveFileWriter, self.error);
    };
    this._getLogDir(function(dir) {
      dir.getFile("" + filename, { create: true }, haveFileEntry, self.error);
    });
  };
  eduproto._syncIfNecessary = function() {
    var events = this._events;
    // if more than 15 items in log_events
    if (events.appevents.length + events.eduevents.length >= 15) {
      // clone the event data
      var data = JSON.parse(JSON.stringify(this._events));
      // replace log_events with empty array
      this._events = { eduevents: [], appevents: []};
      // add device details to the data
      data.device = device.model;
      data.os = device.platform + ":" + device.version;
      data.uid = localStorage.getItem("eduevents-uuid");
      if (this.options.version) { data.appver = this.options.version; }
      // jsonize the data
      var jsondata = JSON.stringify(data);
      // write the JSON obj to disk (temp file, tstamp as name)
      this._writeFile("" + new Date().getTime() + ".json", jsondata, function(file) {
        // file was written successfully
        if (this.online()) { // if online
          // call _postFile with the filehandle and content
          this._postFile(file, jsondata);
        }
      }.bind(this));
    }
    // store the array into localStorage
    localStorage.setItem("eduevents", JSON.stringify(this._events));
  };

  eduproto._getLogDir = function(callback, error) {
    var self = this;
    var haveLogDir = function(dir) {
      self._logdir = dir;
      callback(self._logdir);
    };
    if (this._logdir) {
      haveLogDir(this._logdir);
      return;
    }
    var haveFileSystem = function(fs) {
      self._filesystem = fs;
      if (self._logdir) {
        haveLogDir(self._logdir);
      } else {
        fs.root.getDirectory(LOG_DIR_NAME, { create: true }, haveLogDir, error || self._error);
      }
    };
    if (this._filesystem) {
      haveFileSystem(this._filesystem);
    } else {
      window.requestFileSystem(LocalFileSystem.PERSISTENT, 0, haveFileSystem, error || this._error);
    }
  };

  // checks the log directory for files created when offline and sends them to the server
  eduproto._syncExistingFiles = function() {
    var self = this,
        // store current time to avoid sending duplicates of new files created during
        // processing of the existing files
        syncStartTime = new Date();

    this._getLogDir(function(dir) {
      var dirReader = dir.createReader();
      dirReader.readEntries(haveDirEntries, self._error);
    });

    var haveDirEntries = function(entries) {
      // a function that gets and processes first item in file list, and in the end
      // calls itself until there are no more files in the list of entries
      var handleNextFile = function() {
        var nextFile = entries.shift();
        if (nextFile && nextFile.isFile) { // entries includes subdirs, so check if we have a file

          // we need the file metadata, this function is called when we have that
          var haveFileMetadata = function(metadata) {
            // check that the fileentry was created before start of processing
            if (Date.parse(metadata.modificationTime) < syncStartTime) {
              // in order to read the file contents, we need file not fileentry
              // this function is called once we have the file
              var gotFile = function(file) {
                // create a filereader
                var reader = new FileReader();
                reader.onloadend = function(evt) {
                  // callback that is called after the file is read
                  if (evt.target.result) { // if we have result, call _postfile
                    self._postFile(nextFile, evt.target.result, function() {
                      handleNextFile(); // process next file
                    });
                  }
                };
                // start reading the file
                reader.readAsText(file);
              };
              // get the file object of the fileentry
              nextFile.file(gotFile, self._error);
            }
          };
          // get the metadata
          nextFile.getMetadata(haveFileMetadata, self._error);
        } else if (nextFile) { // we have an entry but it is not a file
          handleNextFile();
        }
      };
      // call the function to get the processing started
      handleNextFile();
    };
  };
  icecube.EduEvents = EduEvents;



  icecube.Dialog = function(opts) {
    // opts:
    // - title - dialog title
    // - msg - message body
    // - buttons (array of Strins)
    // - closeOnClick (default true if no buttons)
    // - noface (default to false)
    // - classname (class added to the icecube-dialog element)
    var dialog = document.createElement("div");
    dialog.classList.add("icecube-dialogwrapper");
    var body = '<div class="icecube-dialog ' + (opts.classname?opts.classname:'') + '"><h2 class="' +
        (opts.noface?'noface':'face') + '">' +
        (opts.title || "") + '</h2>' +
        '<div class="icecube-dialogbody">' + (opts.msg || "") + '</div>' +
        '<div class="icecube-dialogbuttons">';
    if (opts.buttons) {
      for (var i = opts.buttons.length; i--;) {
        body += '<button data-button-index="' + i + '">' + opts.buttons[i] + '</button>';
      }
    } else {
      body += '<button>OK</button>';
    }
    var closeOnClick = opts.closeOnClick?true:!opts.buttons;
    var closeDialog = function(status) {
      dialog.remove();
      if (opts.callback && typeof opts.callback === "function") {
        opts.callback(status);
      }
    };
    dialog.addEventListener("click", function(evt) {
      if (evt.target.nodeName.toLowerCase() === "button") {
        closeDialog(evt.target.dataset.buttonIndex);
      } else if (evt.target.classList.contains("icecube-dialogwrapper") && closeOnClick) {
        closeDialog();
      }
    });
    body += '</div></div>';

    dialog.innerHTML = body;
    document.body.appendChild(dialog);
  };

  var ShareButtons = function(msg, opts) {
    this.msg = msg;
    this.opts = opts || {};
  };
  var shareproto = ShareButtons.prototype;
  shareproto.html = function() {
    var html = '<div class="icecube-sharebuttons">';
    if (typeof this.opts.twitter === "undefined" || this.opts.twitter) {
      html += '<a href="#" class="icecube-twitter icecube-share">Twitter</a>';
    }
    if (typeof this.opts.facebook === "undefined" || this.opts.facebook) {
      html += '<a href="#" class="icecube-facebook icecube-share">Facebook</a>';
    }
    html += '</div>';
    return html;
  };
  shareproto.shareOnTwitter = function(evt) {
    if (evt) {
      evt.preventDefault();
      evt.stopPropagation();
    }
    if (window.plugins && window.plugins.socialsharing) {
      window.plugins.socialsharing.shareViaTwitter(this.msg, null, this.opts.url, function(shared) {
        if (shared) {
          if (this.opts.callback && typeof this.opts.callback === "function") {
            this.opts.callback();
          } else if (this.opts.eduevents) {
            this.opts.eduevents.logAppEvent("twittershare");
          }
        }
      }.bind(this));
    } else {
      console.warn("Sharing to Twitter but socialsharing plugin not loaded.");
    }
  };
  shareproto.shareOnFacebook = function(evt) {
    if (evt) {
      evt.preventDefault();
      evt.stopPropagation();
    }
    if (window.plugins && window.plugins.socialsharing) {
      window.plugins.socialsharing.shareViaFacebook(this.msg, null, this.opts.url, function(shared) {
        if (shared) {
          if (this.opts.callback && typeof this.opts.callback === "function") {
            this.opts.callback();
          } else if (this.opts.eduevents) {
            this.opts.eduevents.logAppEvent("facebookshare");
          }
        }
      }.bind(this));
    } else {
      console.warn("Sharing to Facebook but socialsharing plugin not loaded.");
    }
  };

  shareproto.attachListeners = function(elem) {
    var tmpShareButton = elem.querySelector(".icecube-twitter");
    if (tmpShareButton) {
      tmpShareButton.addEventListener("click", this.shareOnTwitter.bind(this));
    }
    tmpShareButton = elem.querySelector(".icecube-facebook");
    if (tmpShareButton) {
      tmpShareButton.addEventListener("click", this.shareOnFacebook.bind(this));
    }
  };
  icecube.ShareButtons = ShareButtons;


  var FEEDBACK_EMAIL = 'ville@mobileicecube.com';
  var Feedback = function(opts) {
    // opts supported:
    //  - appname (name of the application) [required]
    //  - appversion (version of the app) [optional]
    //  - email (the email address to send the feedback) [optional]
    this.opts = opts;
    var body = "I'd love to hear from you! Send your comments, suggestions, problems, and feature requests.\n\n\n" +
        "Feedback:\n\n\n";
    if ('device' in window) {
      body += "Device platform: " + device.platform + "\n" +
              "Device version: " + device.version + "\n" +
              "Device model: " + device.model + "\n";
    }
    if (opts.appversion) {
      body += "App version: " + opts.appversion + "\n";
    }
    this._body = body;
  };
  Feedback.prototype.emailUrl = function() {
    var url = 'mailto:' + encodeURIComponent('"' + this.opts.appname + ' Feedback" <' +
              (this.opts.email || FEEDBACK_EMAIL) + '>') + '?Subject=' + this.opts.appname + ' Feedback&body=' +
              encodeURIComponent(this._body);
    return url;
  };
  icecube.Feedback = Feedback;

  if (typeof module !== "undefined" && module.exports) {
    module.exports = icecube;
  } else if (typeof window !== "undefined") {
    window.IceCube = icecube;
  }
})();