var readerApp = angular.module('readerApp', [
  'ngAnimate',
  'ngTouch',
  'ui.router',
  'xml',
  'indexedDB',
  'ionic',
  'ngNotify',
  'monospaced.elastic',
  'cgBusy',
  'angular-google-analytics'
]);
angular.module('readerApp').config([
   '$indexedDBProvider', function(
    $indexedDBProvider
  )
  {
    const DATABASE_VERSION = 1;
    const DATABASE_NAME    = 'readerapp'

    $indexedDBProvider
      .connection(DATABASE_NAME)
      .upgradeDatabase(1, function(event, database, transaction) {
        // API cache
        database
          .createObjectStore('cache', {keyPath: 'cache_key'})
          .createIndex('expires', 'expires', {unique: false});

        // Images cache
        database
          .createObjectStore('images', {keyPath: 'url'})
          .createIndex('last_access', 'last_access', {unique: false});

        // Images cache
        database
          .createObjectStore('payments', {keyPath: 'orderNumber'})
          .createIndex('bookId', 'bookId', {unique: true});

        // Key-value store for settings, current user, etc.
        database
          .createObjectStore('keyval', {keyPath: 'key'});

        // 
        database
          .createObjectStore('my_books', {keyPath: 'id'});

        database
          .createObjectStore('book_file_chunks', {keyPath: 'id'})
          .createIndex('book_key', 'book_key', {unique: false});

        database
          .createObjectStore('book_file_images', {keyPath: 'id'})
          .createIndex('book_key', 'book_key', {unique: false});
      });

    // Shortcut for clear api cache
    window.clearApiCache = function() {
      var idb,
          connection,
          objectStore;

      idb = indexedDB.open(DATABASE_NAME, DATABASE_VERSION);
      idb.onsuccess = function(event) {
        connection  = event.target.result;
        objectStore = connection.transaction('cache', 'readwrite').objectStore('cache');
        objectStore.clear();
      };
    };

    // Shortcut for clear MyBook cache
    window.clearMyBooks = function() {
      var idb,
          connection,
          transaction,
          objectStore;

      idb = indexedDB.open(DATABASE_NAME, DATABASE_VERSION);
      idb.onsuccess = function(event) {
        connection  = event.target.result;
        transaction = connection.transaction(['my_books', 'keyval'], 'readwrite');

        objectStore = transaction.objectStore('my_books');
        objectStore.clear();

        objectStore = transaction.objectStore('keyval');
        objectStore.delete('my_books_last_update');
        objectStore.delete('my_books_last_login');
      };
    };


    // Shortcut for clear BookFile cache
    window.clearBookFiles = function() {
      var idb,
          connection,
          transaction,
          objectStore;

      idb = indexedDB.open(DATABASE_NAME, DATABASE_VERSION);
      idb.onsuccess = function(event) {
        connection  = event.target.result;
        transaction = connection.transaction(['book_file_chunks', 'book_file_images'], 'readwrite');

        objectStore = transaction.objectStore('book_file_chunks');
        objectStore.clear();

        objectStore = transaction.objectStore('book_file_images');
        objectStore.clear();
      };
    };
  }
]);

angular.module('readerApp').run([
   'KeyVal', function(
    KeyVal
  )
  {
    // Fill initial data
    KeyVal.get('settings_download_after_purchase', {reject: false}).then(function(result) {
      if (result === undefined) {
        KeyVal.set('settings_download_after_purchase', true);  
      }
    })
  }
]);
angular.module('readerApp').controller('PreloaderCtl', [
  '$scope', '$timeout', '$q', 'Book', 'Comment', function(
    $scope,  $timeout,   $q,   Book,   Comment
  )
  {

    $timeout(function() {
      $scope.preloaderHide = true;
    }, 1000)


  }
]);
angular.module('readerApp').controller('SidebarCtl', [
   '$scope', '$stateParams', '$rootScope', '$q', 'Book', 'Person', 'User', 'Search', 'MyBook', function(
    $scope,   $stateParams,   $rootScope,   $q,   Book,   Person,   User,   Search,   MyBook
  )
  {
    $scope.currentUser = User.current();

    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    $scope.bookShowCount = 3;
    $scope.showAllBookResult = function () {
      $scope.bookShowCount = 100500
    };

    $scope.personShowCount = 3;
    $scope.showAllPersonResult = function () {
      $scope.personShowCount = 100500
    };

    $scope.seriesShowCount = 3;
    $scope.showAllSeriesResult = function () {
      $scope.seriesShowCount = 100500
    };


    $scope.searchFocusChange = function(inFocus, $event) {
      $rootScope.searchOpened = inFocus;
      if(!inFocus) {
        $scope.searchString = null;
      }
    };

    $scope.searchString = null;
    $scope.$watch('searchString', function(nV, oV) {
      if(!nV || nV == '' || nV == oV || nV.length < 3) {
        if(!nV || nV == '') {
          $scope.books = $scope.persons = $scope.series = $scope.best =[]
        }
        return;
      }
      var promise = Search(nV);

      $scope.searchPpromise = $q.all([promise, $scope.myBookStatesPromise]);

      promise.then(function(data) {
        console.log('found', data);
        $scope.books   = data.books;
        $scope.persons = data.persons;
        $scope.series  = data.series;

        if (data.best) {
          $scope.best = [data.best];
        } else {
          $scope.best = null;
        }

      })

    });

    $scope.closeDrawer = function() {
      $rootScope.drawerOpened = false;
    }
  }
]);
angular.module('readerApp').controller('HeaderCtl', [
   '$scope', '$state', '$stateParams', '$q', 'Book', '$ionicHistory', '$rootScope', function(
    $scope,   $state,   $stateParams,   $q,   Book,   $ionicHistory,   $rootScope
  )
  {
    if ($state.current.enable_menu == undefined) {
      $scope.enable_menu = true
    } else {
      $scope.enable_menu = $state.current.enable_menu;
    }

    angular.forEach($stateParams, function(value, key) {
      $scope[key] = value;
    });

    $scope.filter = null;
    $scope.$watch('filter', function(nV, oV) {
      $rootScope.$broadcast('filter.set', nV)
    });

    $scope.filterOpened = false;
    $scope.toggleFilter = function () {
      $scope.filterOpened = !$scope.filterOpened
      if(!$scope.filterOpened) {
        $scope.filter = null
      }
    }

  }
]);
angular.module('readerApp').controller('CatalogCtl', [
   '$scope', '$http', '$q', 'Book', '$stateParams', '$ionicLoading', '$ionicModal', '$state', 'AppError', 'MyBook', 'BookFileDownload', function(
    $scope,   $http,   $q,   Book,   $stateParams,   $ionicLoading,   $ionicModal,   $state,   AppError,   MyBook,   BookFileDownload
  )
  {
    var currentPage     = 0;
    var perPage         = 10;

    var scopeParams = {
      editors_choice: {rating: 'hot'},
      novelty:        {sort:   'time_desc', min_person_rating: 6},
      most_popular:   {sort:   'pop_desc'},
      sequence:       {sequence: $stateParams.sequenceId, sort: 'time_desc'}
    };

    $scope.books    = [];
    $scope.canLoadMore = true;

    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    $scope.doRefresh = function () {
      currentPage = 0;

      $scope.nextPage(true, function() {
        $scope.$broadcast('scroll.refreshComplete');
      }, true);
    };

    $scope.loadMore = function () {
      $scope.nextPage(false, function() {
        $scope.$broadcast('scroll.infiniteScrollComplete');
      });
    };

    // For book list (read with current book)
    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };

    $scope.nextPage = function (forceReload, callback, clearList) {
      if (forceReload == undefined) {
        forceReload = false;
      }

      var books = Book.collection(
        angular.extend({}, scopeParams[$stateParams.scope], {limit: [currentPage * perPage, perPage].join(',')}),
        {forceReload: forceReload}
      );
      
      $scope.pagePromise = $q.all({
        books:  books.$promise,
        states: $scope.myBookStatesPromise
      });

      $scope.pagePromise.then(function(result) {
        if(clearList) {
          $scope.books = []
        }

        angular.forEach(result.books, function(value){
          $scope.books.push(value)
        });

        if(result.books.length > 0) {
          currentPage = currentPage +1;
        } else {
          $scope.canLoadMore = false;
        }

        if(callback != undefined) {
          callback()
        }
        
      }, function(error) {
        error = AppError.wrap(error);

        if (currentPage == 0) {
          $scope.pageError = error;
        } else {
          alert("Не удалось загрузить данные: " + error.message);
        }

        throw error;
      });

      true
    };

  }
]);
angular.module('readerApp').controller('MyCtl', [
   '$scope', '$http', '$q', '$state', '$ionicModal', 'ngNotify', 'Book', 'Genre', 'User', 'MyBook', 'AppError', 'BookFile', 'BookFileDownload', function(
    $scope,   $http,   $q,   $state,   $ionicModal,   ngNotify,   Book,   Genre,   User,   MyBook,   AppError,   BookFile,   BookFileDownload
  ) {

    var isArchived = function(myBook) {
      return  myBook.isArchived() && (!$scope.filter ? true : myBook.book.title.toLowerCase().includes($scope.filter.toLowerCase()))
    };
    var isNotArchived = function(myBook) {
      return !myBook.isArchived() && (!$scope.filter ? true : myBook.book.title.toLowerCase().includes($scope.filter.toLowerCase()))
    };

    $scope.filter = null;
    $scope.$on('filter.set', function(ev, filterValue) {
      $scope.filter = filterValue
    });

    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    var myCollectionSuccess = function(collection) {
      var bookInfoPromises = [];

      $scope.myBooksCollection = collection;
      
      angular.forEach(collection.myBooks, function(myBook) {
        if (myBook.book && myBook.book.$promise) {
          bookInfoPromises.push(myBook.book.$promise);
        }
      });

      $q.all(bookInfoPromises).catch(function(error) {
        $scope.pageError = AppError.wrap(error);
      });
    };

    $scope.reloadPage = function(forceReload) {
      var myBookPromise = MyBook.collection({forceReload: forceReload}).then(myCollectionSuccess);

      var pageFatalFailure = function(error) {
        error = AppError.wrap(error);

        if (error.systemName == "authorizationFailed") {
          ngNotify.set('Сессия просрочена. Пожалуйста, войдите еще раз', 'error');

          User.signOut().then(function() {
            $state.go('ui.account');
          })
        } else {
          $scope.pageError = error;
        }
      };

      var pageReloadFailure = function(error) {
        if (AppError.is(error)) {
          if (error.systemName == 'notWifiConnection' && !forceReload) {
            return;

          }
          
          alert("Не удалось обновить список: " + error.message);
        }
      }

      $scope.myBookPromise = myBookPromise.then(
        null,
        pageFatalFailure,
        pageReloadFailure
      );

      if (!forceReload) {
        $scope.pagePromise = $q.all([$scope.myBookPromise, $scope.myBookStatesPromise]);
      }

      return $scope.myBookPromise;
    }

    $scope.reloadPage();
    
    $scope.doRefresh = function() {
      $scope.reloadPage(true).finally(function() {
        $scope.$broadcast('scroll.refreshComplete');
      });
    };

    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };

    $scope.goToBook = function (book, hideModal) {
      $state.go('ui.book', {book_id: book.id})

      if ($modal && hideModal) {
        $modal.hide();
        $modal = null;
      }
    };

    var myBooksCollectionUpdated = function() {
      var collection,
          i;

      if ($scope.myBooksCollection && $scope.myBooksCollection.myBooks) {
        collection = $scope.myBooksCollection.myBooks;
      } else {
        collection = [];
      }
      
      $scope.myNotArchived = [];
      $scope.myArchived    = [];

      for (i = 0; i < collection.length; ++i) {
        if (isArchived(collection[i])) {
          $scope.myArchived.push(collection[i]);
        }

        if (isNotArchived(collection[i])) {
          $scope.myNotArchived.push(collection[i]);
        }
      }
    }

    $scope.$watchCollection('myBooksCollection.myBooks', myBooksCollectionUpdated);
    $scope.$watch('filter', function(newVal, oldVal) {
      if (newVal != oldVal) {
        myBooksCollectionUpdated();
      }
    });

    var $modal = null;

    $ionicModal.fromTemplateUrl('templates/my/modal.html', {
      scope:     $scope,
      animation: 'slide-in-up'
    }).then(function(modal) {
      $modal = modal;
      window.modal = $modal;
    });

    $scope.bookMenu = function(myBook) {
      $scope.menuForBook = myBook;
      $modal.show();
    };

    $scope.removeBook = function(book) {
      var removeSuccess = function() {
        myBooksCollectionUpdated();
      }

      var removeFailure = function(error) {
        error = AppError.wrap(error);
        ngNotify.set('Ошибка при перемещении "' + book.book.title  + '" в архив: ' + error.message, 'error');
      }

      book.moveToArchive(true).then(removeSuccess, removeFailure);

      if ($modal) {
        $modal.hide();
        $modal = null;
      }
    };

    $scope.returnBook = function(book) {
      var returnSuccess = function() {
        myBooksCollectionUpdated();
      }

      var returnFailure = function(error) {
        error = AppError.wrap(error);
        ngNotify.set('Ошибка при перемещении "' + book.book.title  + '" из архива: ' + error.message, 'error');
      }

      book.returnFromArchive(true).then(returnSuccess, returnFailure);

      if ($modal) {
        $modal.hide();
        $modal = null;
      }
    };

    $scope.closeModal = function() {
      $modal.hide();
    }

    $scope.$on('$ionicView.enter', function(){
      myBooksCollectionUpdated();  
    });

}]);
angular.module('readerApp').controller('BookmarksCtl', [
   '$scope', '$http', '$state', '$q', 'ngNotify', 'Book', 'Genre', 'User', 'AppError', 'MyBook', 'BookFileDownload', function(
    $scope,   $http,   $state,   $q,   ngNotify,   Book,   Genre,   User,   AppError,   MyBook,   BookFileDownload
    )
  {
    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    $scope.fetchBookmarks = function (forceReload) {
      var bookmarksSuccess = function(result) {
        $scope.books = result;
      }

      var bookmarksFailure = function(error) {
        error = AppError.wrap(error);

        if (error.systemName == "authorizationFailed") {
          ngNotify.set('Сессия просрочена. Пожалуйста, войдите еще раз', 'error');

          User.signOut().then(function() {
            $state.go('ui.account');
          })
        } else {
          $scope.pageError = error;
        }

        throw error;
      };

      $scope.booksPromise = Book.bookmarks({}, {forceReload: forceReload}).$promise.then(
        bookmarksSuccess,
        bookmarksFailure
      );

      if (!forceReload) {
        $scope.pagePromise = $q.all([$scope.booksPromise, $scope.myBookStatesPromise]);
      }

      return $scope.booksPromise;
    };

    $scope.doRefresh = function () {
      $scope.fetchBookmarks(true).finally(function() {
        $scope.$broadcast('scroll.refreshComplete');
      });
    };

    $scope.fetchBookmarks();

    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };
  }
]);
angular.module('readerApp').controller('BookCtl', [
   '$scope', '$state', '$stateParams', '$q', '$ionicModal', '$ionicLoading', 'Book', 'Comment', 'Genre', 'Basket', 'BookFileDownload', 'BookFileDownloadRegistry', 'BookFile', 'MyBook', 'AppError', function(
    $scope,   $state,   $stateParams,   $q,   $ionicModal,   $ionicLoading,   Book,   Comment,   Genre,   Basket,   BookFileDownload,   BookFileDownloadRegistry,   BookFile,   MyBook,   AppError
  )
  {

    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    $scope.reloadPage = function() {
      $scope.book     = Book.element($stateParams.book_id);
      $scope.related  = Book.collection({art: $stateParams.book_id, rating: 'with', limit: '0,3'});
      $scope.comments = Comment.collection({art: $stateParams.book_id});

      var fetchGenres = function(book) {
        genrePromises = book.genres.map(function(genreType) {
          return Genre.element(genreType).$promise;
        });

        return $q.all(genrePromises).then(function(genres) {
          $scope.genres = genres;

          return book;
        });
      };

      var commentsSuccess = function(comments) {
        $scope.lastComment   = comments[0];
        $scope.countComments = comments.length;
      };

      var bookFileFullSuccess = function(result) {
        $scope.bookFileFullSaved = result;
      };

      var bookFileTrialSuccess = function(result) {
        $scope.bookFileTrialSaved = result;
      };


      var bookPromise          = $scope.book.$promise.then(fetchGenres),
          commentsPromise      = $scope.comments.$promise.then(commentsSuccess),
          relatedPromise       = $scope.related.$promise,
          bookFileFullPromise  = BookFile.isSaved($stateParams.book_id).then(bookFileFullSuccess),
          bookFileTrialPromise = BookFile.isSaved($stateParams.book_id, true).then(bookFileTrialSuccess);

      var pageFailure = function(error) {
        $scope.pageError = AppError.wrap(error);
      };

      $scope.pagePromise = $q.all({
        book:          bookPromise,
        comments:      commentsPromise,
        related:       relatedPromise,
        bookFileFull:  bookFileFullPromise,
        bookFileTrial: bookFileTrialPromise,
        states:        $scope.myBookStatesPromise
      }).catch(
        pageFailure
      );
    };

    $scope.reloadPage();

    // For current book
    $scope.downloadCurrentBook = function(isTrial) {
      var download, downloadPromise;

      var downloadSuccess = function(result) {
        if (isTrial) {
          $scope.bookFileTrialSaved = true;
        } else {
          $scope.bookFileFullSaved = true;
        }

        return result;
      };

      var downloadFailure = function(error) {
        error = AppError.wrap(error);

        if (error.systemName == 'notError') {
          throw error;
        }

        if (isTrial) {
          alert("Ошибка скачивания книги: " + error.message);
        } else {
          alert("Ошибка скачивания фрагмента: " + error.message);
        }

        throw error;
      }

      if (isTrial) {
        download = BookFileDownload.downloadTrial($stateParams.book_id);
      } else {
        download = BookFileDownload.downloadFull($stateParams.book_id);
      }

      downloadPromise = download.promise.then(
        downloadSuccess,
        downloadFailure
      );

      return downloadPromise;
    };

    $scope.readTrial = function() {
      var go = function(result) {
        Basket.putToBasket($scope.book.id);
        
        $state.go('ui.reader_trial', {book_id: $stateParams.book_id});
      };

      if ($scope.bookFileTrialSaved) {
        go();
      } else {
        $scope.downloadCurrentBook(true).then(go);
      }
    };

    $scope.readFull = function() {
      var go = function() {
        $state.go('ui.reader_full', {book_id: $scope.book.id});
      };

      if ($scope.bookFileFullSaved) {
        go();
      } else {
        $scope.downloadCurrentBook(false).then(go);
      }
    };

    // For book list (read with current book)
    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };

  }
]);
angular.module('readerApp').controller('BookAuthorsCtl', [
   '$scope', '$stateParams', 'Book', 'AppError', function(
    $scope,   $stateParams,   Book,   AppError
    )
  {
    $scope.reloadPage = function() {
      $scope.book = Book.element($stateParams.book_id);
      $scope.pagePromise = $scope.book.$promise;

      $scope.pagePromise.catch(function(error) {
        $scope.pageError = AppError.wrap(error);
      });
    }

    $scope.reloadPage();
  }
]);
angular.module('readerApp').controller('BookCommentsCtl', [
   '$scope', '$stateParams', '$q', 'Comment', 'AppError', function(
    $scope,   $stateParams,   $q,   Comment,   AppError
    )
  {
    $scope.fetchComments = function (forceReload) {
      var commentsSuccess = function(result) {
        $scope.comments = result;
      }

      var commentsFailure = function(error) {
        $scope.pageError = AppError.wrap(error);
        throw error;
      };

      $scope.commentsPromise = Comment.collection({art: $stateParams.book_id}, {forceReload: forceReload}).$promise.then(
        commentsSuccess,
        commentsFailure
      );

      if (!forceReload) {
        $scope.pagePromise = $scope.commentsPromise;
      }

      return $scope.commentsPromise;
    }

    $scope.doRefresh = function () {
      $scope.fetchComments(true).finally(function() {
        $scope.$broadcast('scroll.refreshComplete');
      });
    };

    $scope.fetchComments();
  }
]);
angular.module('readerApp').controller('CommentsHeaderCtl', [
   '$scope', '$stateParams', '$q', 'Book', '$ionicHistory', 'User', function(
    $scope,   $stateParams,   $q,   Book,   $ionicHistory,   User
  )
  {
    $scope.currentUser = User.current();

    angular.forEach($stateParams, function(value, key) {
      $scope[key] = value;
    });

    $scope.goBack = function() {
      //window.history.back();
      $ionicHistory.goBack();
    }

  }
]);
angular.module('readerApp').controller('BookCommentsNewCtl', [
   '$scope', '$stateParams', '$state', '$q', 'Comment', 'AppError', 'User', function(
    $scope,   $stateParams,   $state,   $q,   Comment,   AppError,   User
    )
  {
    $scope.reloadPage = function() {
      var currentUserSuccess = function(currentUser) {
        $scope.mail = currentUser.email;
      };

      var currentUserFailure = function(error) {
        $scope.pageError = error;
      };

      $scope.currentUser = User.current();
      $scope.pagePromise = $scope.currentUser.$promise.then(
        currentUserSuccess,
        currentUserFailure
      );
    };

    $scope.reloadPage();

    // TODO: redirect to login if not signed in
    // TODO: remember user nickname

    $scope.nickname = "";
    $scope.mail     = "";
    $scope.message  = "";
    $scope.rating   = null;

    // TODO: оценка
    // TODO: email обязателен для коммента,
    //       но не обязателен для юзера

    $scope.create = function() {
      var success = function() {
        alert("Ваш комментарий успешно добавлен и появится после прохождения модерации");
        $state.go("^");
      }

      var failure = function(error) {
        alert("Не удалось добавить отзыв: " + AppError.wrap(error).message);
      }

      var params = {
        nickname: $scope.nickname,
        message:  $scope.message,
        art:      $stateParams.book_id,
        rating:   $scope.rating
      };

      if ($scope.mail) {
        params.mail = $scope.mail;
      }

      $scope.submitPromise = Comment.create(params);
      $scope.submitPromise.then(success, failure);
    }
  }
]);
angular.module('readerApp').controller('SettingsCtl', [
   '$scope', '$q', 'KeyVal', 'User', function(
    $scope,   $q,   KeyVal,   User
    )
  {
    settingsList     = ['wifi_only', 'download_after_purchase'];
    settingsPromises = {}

    var fetchSettings = function(index, settings) {
      var systemName;

      if (index >= settingsList.length) {
        return settings;
      } else {
        settings   = settings || {};
        index      = index || 0;
        systemName = settingsList[index];

        var getSuccess = function(value) {
          settings[systemName] = value;
          return fetchSettings(index + 1, settings);
        }

        var getFailure = function(error) {
          settings[systemName] = null;
          return fetchSettings(index + 1, settings);
        }

        return KeyVal.get("settings_" + systemName).then(
          getSuccess,
          getFailure
        );
      }
    }

    // Queueing 'set' calls for KeyVal storage to prevent
    // race conditions
    var currentSetDeferred = $q.defer();
    var currentSetPromise  = currentSetDeferred.promise;

    currentSetDeferred.resolve();

    $scope.pagePromise = fetchSettings().then(function(settings) {
      User.current().$promise.then(function(currentUser) {
        $scope.settings    = settings;
        $scope.currentUser = currentUser;

        angular.forEach(settingsList, function(systemName) {
          $scope.$watch('settings.' + systemName, function(newVal, oldVal) {
            if (newVal != oldVal) {
              currentSetPromise = currentSetPromise.then(function() {
                return KeyVal.set('settings_' + systemName, newVal);
              });
            }
          });
        });
      });
    });

  }
])
angular.module('readerApp').controller('GenresIndexCtl', [
   '$scope', '$http', '$q', '$stateParams', 'Genre', 'AppError', function(
    $scope,   $http,   $q,   $stateParams,   Genre,   AppError
    )
  {
    $scope.fetchGenres = function (forceReload) {
      var genreSuccess = function(result) {
        $scope.genres = result;
      };
      
      var genreFailure = function(errors) {
        $scope.pageError = AppError.wrap(errors);
        throw errors;
      };

      $scope.genresPromise = Genre.collection({}, {forceReload: forceReload}).then(
        genreSuccess,
        genreFailure
      );

      if (!forceReload) {
        $scope.pagePromise = $scope.genresPromise;
      }

      return $scope.genresPromise;
    };

    if ($stateParams.genre || $stateParams.title) {
      // Page with child genres
      if ($stateParams.genre) {
        $scope.genres = $stateParams.genre.children;
      } else if ($stateParams.title) {
        $scope.pagePromise = Genre.element($stateParams.title).$promise;
        $scope.pagePromise.then(function(result) {
          $scope.genres = result.children;
        });
      }

      $scope.doRefresh = function () {
        $scope.$broadcast('scroll.refreshComplete');
      }
    } else {
      // Page with root genres
      $scope.fetchGenres();

      $scope.doRefresh = function () {
        $scope.fetchGenres(true).finally(function() {
          $scope.$broadcast('scroll.refreshComplete');
        });
      };
    }
  }]);
angular.module('readerApp').controller('GenresShowCtl', [
   '$scope', '$http', '$q', '$ionicScrollDelegate', 'Book', 'Genre', '$stateParams', 'AppError', 'MyBook', 'BookFileDownload', function(
    $scope,   $http,   $q,   $ionicScrollDelegate,   Book,   Genre,   $stateParams,   AppError,   MyBook,   BookFileDownload
    )
  {
    var currentPages    = {'new': 0, 'most_popular': 0},
        perPage         = 10;

    $scope.canLoadMore  = {'new': true, 'most_popular': true};

    $scope.loading = true;
    $scope.type    = 'new';
    $scope.books   = [];

    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    $scope.loadMore = function () {
      var load = function () {
        loadBooks(undefined, function() {
          $scope.$broadcast('scroll.infiniteScrollComplete');
        });
      };

      if($scope.genre.$promise) {
        $scope.genre.$promise.then(function() { load() })
      } else {
        load()
      }
    };

    var loadBooks = function (type, callback) {
      var requestParam, type;

      if (!type) {
        type = $scope.type
      } else if (type != $scope.type) {
        $scope.type  = type;
        $scope.books = [];

        currentPages[$scope.type] = 0;
        $ionicScrollDelegate.scrollTop();
      }

      switch (type) {
        case 'new':
          requestParam = {sort: 'time_desc'};
          break;
        case 'most_popular':
          requestParam = {sort: 'pop_desc'};
          break;
      }
      requestParam.genre = $scope.genre.id || $scope.genre.children.map(function(e) { return e.id});

      if (!$scope.genre.id) {
        // Use min rating only for root genres
        requestParam.min_person_rating = 6;
      }

      $scope.loading = true;

      $scope.bookPromise = Book.collection(
        angular.extend(requestParam, {limit: [currentPages[$scope.type] * perPage, perPage].join(',')})
      ).$promise;

      var success = function(result) {
        angular.forEach(result, function(value){
          $scope.books.push(value)
        });

        if(result.length > 0) {
          currentPages[$scope.type] = currentPages[$scope.type] + 1;
        } else {
          $scope.canLoadMore[$scope.type] = false;
        }

        if(callback != undefined) {
          callback()
        }
      }

      var failure = function(error) {
        error = AppError.wrap(error);

        if (currentPages[$scope.type] == 0) {
          $scope.pageError = error;
        } else {
          alert("Не удалось загрузить данные: " + error.message);
        }

        throw error;
      }

      $scope.pagePromise = $scope.bookPromise.then(success, failure);
      $scope.pagePromise = $q.all([$scope.pagePromise, $scope.myBookStatesPromise]);
    };

    $scope.loadBooks = loadBooks;

    if($stateParams.genre) {
      $scope.genre = $stateParams.genre
    } else if($stateParams.title) {
      $scope.genre = Genre.element($stateParams.title);
    }

    // For book list (read with current book)
    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };

  }
]);
angular.module('readerApp').controller('AccountRecoverPasswordCtl', [
   '$scope', '$state', 'User', 'AppError', function(
    $scope,   $state,   User,   AppError
    )
  {
    $scope.email  = "";

    $scope.submit = function() {
      var recoverEmail = $scope.email;
      
      var success = function(result) {
        alert("Инструкция по восстановлению отправлена на почту " + recoverEmail);
        $state.go('^');
      }

      var failure = function(error) {
        alert('Ошибка восстановления пароля: ' + AppError.wrap(error).message);
      }

      $scope.recoverPromise = User.passwordRecover(recoverEmail).then(
        success,
        failure
      );
    }
  }
])
angular.module('readerApp').controller('AccountRegisterCtl', [
   '$scope', '$state', 'User', 'AppError', function(
    $scope,   $state,   User,   AppError
    )
  {
    $scope.login    = "";
    $scope.password = "";

    $scope.params = {
      email:       null,
      first_name:  null,
      middle_name: null,
      last_name:   null,
      city:        null,
      phone:       null,
      www:         null,
      birth_day:   null, // YYYY-MM-DD
      male:        null  // 'm' || 'f'
    };

    $scope.submit = function() {
      var success = function(result) {
        alert("Регистрация завершена успешно. Письмо со ссылкой для подтверждения адреса отправлена на почту.");
        $state.go('^');
      }

      var failure = function(error) {
        alert('Не удалось зарегистрироваться: ' + AppError.wrap(error).message);
      }

      $scope.signUpPromise = User.signUp(
        $scope.login,
        $scope.password,
        $state.params
      ).then(
        success,
        failure
      );
    }
  }
])
angular.module('readerApp').controller('AccountHeaderCtl', [
   '$scope', '$state', 'User', function(
    $scope,   $state,   User
    )
  {
    $scope.currentUser = User.current();

    $scope.signOut = function() {
      var signOutFailure = function() {
        alert('Ошибка: не удалось выйти из аккаунта');
      }

      var signOutSuccess = function() {
        $state.go('ui.account');
      }

      User.signOut().then(
        signOutSuccess,
        signOutFailure
      );
    }
  }
])
angular.module('readerApp').controller('AccountProfileCtl', [
   '$scope', '$state', 'ngNotify', 'User', 'AppError', function(
    $scope,   $state,   ngNotify,   User,   AppError
  )
  {
    $scope.reloadPage = function() {
      $scope.currentUser = User.current();
      $scope.currentUser.$promise.then(function(currentUser) {
        $scope.formParams = {
          first_name:  currentUser.first_name  || "",
          middle_name: currentUser.middle_name || "",
          last_name:   currentUser.last_name   || "",
          phone:       currentUser.phone       || ""
        }
      });

      $scope.pagePromise = $scope.currentUser.$promise;
    };

    $scope.reloadPage();

    $scope.submit = function() {
      var success = function() {
        ngNotify.set('Профиль обновлен успешно.', {type: 'success'});
        $state.go('^');
      };


      var failure = function(error) {
        error = AppError.wrap(error);
        alert('Ошибка обновления профиля: ' + error.message);
      }

      $scope.submitPromise = User.update($scope.formParams).then(
        success,
        failure
      );
    }
  }
])
angular.module('readerApp').controller('AccountCtl', [
   '$scope', '$state', 'User', 'AppError', '$ionicScrollDelegate', function(
    $scope,   $state,   User,   AppError,   $ionicScrollDelegate
  )
  {
    $scope.currentUser = User.current();
    $scope.currentUser.$promise.catch(function(error) {
      console.log("Ошибка регистрации пользователя: " + error.message);
    });

    $scope.$watch('currentUser.session_id', function(newVal) {
      if (newVal) {
        $scope.currentUser.$promise.then(function() {
          User.getBalance().then(function(balance) {
            $scope.balance = balance;
          });
        });
      }
    });

    $scope.login    = "";
    $scope.password = "";

    $scope.signIn = function() {
      var success = function() {
        $ionicScrollDelegate.scrollTop();
      }

      var failure = function(error) {
        alert('Не удалось войти: ' + AppError.wrap(error).message);
      }

      $scope.signInPromise = User.signIn(
        $scope.login,
        $scope.password
      ).then(
        success,
        failure
      );
    }
  }
])
angular.module('readerApp').controller('PersonCtl', [
   '$scope', '$stateParams', '$q', 'Book', 'Person', 'AppError', 'MyBook', 'BookFileDownload', function(
    $scope,   $stateParams,   $q,   Book,   Person,   AppError,   MyBook,   BookFileDownload
    )
  {
    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    var personSuccess = function(newPerson) {
      $scope.loadTab = function(new_tab) {
        if ($scope.type == new_tab) {
          return;
        } else {
          $scope.type = new_tab;
        }

        switch ($scope.type) {
          case 'new':
            if (!$scope.new_books) {
              $scope.new_books = Book.collection({person: newPerson.id, sort: 'time_desc'});
              return $scope.new_books.$promise.catch(function(error) {
                alert("Не удалось загрузить новинки автора: " + AppError.wrap(error).message);
              });
            }

            break;
          case 'most_popular':
            if (!$scope.pop_books) {
              $scope.pop_books = Book.collection({person: newPerson.id, sort: 'pop_desc'});
              return $scope.pop_books.$promise.catch(function(error) {
                alert("Не удалось загрузить популярные книги: " + AppError.wrap(error).message);
              });
            }

            break;
          case 'series':
            if (!$scope.sequences) {
              var books = Book.collection({person: newPerson.id, sort: 'pop_desc'}),
                  defer = $q.defer();

              $scope.sequences = []
              $scope.sequences.$promise = defer.promise;

              return books.$promise.then(function(data) {
                $scope.sequences = [].concat.apply([], data.map(function(e) {
                  return e.sequences
                })).unique();
              }, function(error) {
                alert("Не удалось загрузить серии автора: " + AppError.wrap(error).message);
              });
            }

            break;
        }
      };

      return $scope.loadTab('new');
    };

    var personFailure = function(error) {
      $scope.pageError = AppError.wrap(error);
    };

    $scope.reloadPage = function() {
      $scope.person        = Person.element($stateParams.personId);
      $scope.personPromise = $scope.person.$promise.then(personSuccess, personFailure);
    }

    $scope.reloadPage();

    $scope.downloadBook = function(bookId) {
      var downloadFailure = function(error) {
        // Error is empty when user press 'stop download' button
        if (error) {
          alert("Ошибка загрузки книги: " + AppError.wrap(error).message);
        }
      };

      BookFileDownload.downloadFull(bookId).promise.catch(downloadFailure)
    };
  }
]);
angular.module('readerApp').controller('PurchaseWithBankCardCtl', [
    '$scope', '$state', '$q', '$http', 'Book', 'Payments', 'URLBuilder', 'ngNotify', '$ionicHistory', function(
     $scope,   $state,   $q,   $http,   Book,   Payments,   URLBuilder,   ngNotify,   $ionicHistory
  )
  {
    $scope.fieldTranslations = {
      'country':  'Страна',
      'mail':     'E-mail',
      'phone':    'Номер телефона',
      'bank':     'Банк'
    };


    $scope.years = [];

    var currentYear = (new Date).getFullYear();

    for(var i = currentYear; i < currentYear + 20; i ++ ) {
      $scope.years.push(i.toString().substr(2,4));
    }

    $scope.months = [];
    for(var i = 1; i <= 12; i ++ ) { $scope.months.push(i < 10 ? "0" + i : i) }

    $scope.form = {
      CardNumber: [],
      CardExpDate: [],
      Phone: [],
      Country: "RU"
    };

    var countries = function (bookId) {
      var url     = URLBuilder.build("/static/country-codes.xml"),
          defered = $q.defer();

      $http.get(url)
        .success(function(response, status, headers, config) {
          defered.resolve(response['country-codes']['country']);
        })
        .error(function(response, status, headers, config) {
          defered.reject(response, status, headers, config);
        });

      return defered.promise;
    };

    var bookPromise = Book.element($state.params.bookId).$promise.then(function(data) {
      //data.price = 20;
      $scope.price = data.price;
      $scope.book  = data;

      var req = Payments.creditCardInit(data.price);
      req.then(function(data) {
        angular.forEach(data.params, function(param) {
          if(param.key == undefined) {
            $scope.form[param.name] = param.value;
          }
        });
        $scope.paymentConfig = data;

        setTimeout(function () {
          $.autotab.refresh();
        }, 500);
      });

      countriesPromise = countries().then(function(data) {
        $scope.countries = data;
      });

      $scope.pagePromise = $q.all({
        countries:    countriesPromise,
        payment:      req
      });

    });

    $scope.pagePromise = bookPromise;


    $scope.template = function(key) {
      return 'templates/purchase/bank_card/_el_' + key + '.html'
    };

    $scope.hasKey = function(value, index, array) {
      return (value.key != undefined && ['name', 'number', 'cvv', 'expires'].indexOf(value.key) == -1)
    };

    $scope.submit = function () {
      var form = angular.copy($scope.form),
          normalized = {};

      angular.forEach(form, function(field, key) {
        if(angular.isArray(field)) {
          normalized[key] = field.join('')
        } else {
          normalized[key] = field
        }
      });

      console.log($scope.paymentConfig.url, normalized)

      var req = Payments.creditCardProcess($scope.paymentConfig.url, normalized, $state.params.bookId, {
        price:    $scope.book.price,
        bookId:   $scope.book.id,
        bookName: $scope.book.title,
        currency: 'RUB'
      });
      
      req.then(function() {
        console.log('argumants1', arguments);

        $ionicHistory.goBack();
      }, function(response) {
        console.log('argumants2', arguments);

        if(response.errorDesc) {
          ngNotify.set([
              '<b>', response.errorDesc[0], '</b> ', response.errorDesc[1]
          ].join(''), {
            duration: 10000,
            type: 'error',
            html: true
          });
        } else {
          ngNotify.set('При обработке платежа возникла неизвестная ошибка, попробуйте воспользоваться другим способом оплаты.', {
            duration: 10000,
            type: 'error'
          });
        }
      });

      $scope.processPromise = req;
    }

  }
]);
angular.module('readerApp').controller('PurchaseWithMobileCommerceCtl', [
   '$scope', '$state', 'Book', 'Payments', '$ionicHistory', '$ionicLoading', 'ngNotify', function(
    $scope,   $state,   Book,   Payments,   $ionicHistory,   $ionicLoading,   ngNotify
  )
  {
    $scope.phone = {
      code:   "",
      number: ""
    };

    Book.element($state.params.bookId).$promise.then(function(data) {
      $scope.price = data.price;
      $scope.book  = data;
    });

    $scope.submit = function() {
      payment = Payments.mobileCommerceInit([$scope.phone.code, $scope.phone.number].join(''), $scope.price, $state.params.bookId, {
        price:    $scope.book.price,
        bookId:   $scope.book.id,
        bookName: $scope.book.title,
        currency: 'RUB'
      });

      payment.then(function(response) {
        $ionicHistory.goBack();
      }, function() {
        ngNotify.set('При обработке платежа возникла неизвестная ошибка, попробуйте воспользоваться другим способом оплаты.', {
          duration: 10000,
          type: 'error'
        });
      });

      $scope.processPromise = payment
    }

    setTimeout(function () {
      $.autotab.refresh();
    }, 1);
  }
]);
angular.module('readerApp').controller('ReaderCtl', [
   '$scope', '$q', '$stateParams', 'Book', 'BookFile', 'User', 'URLBuilder', 'AppError', 'MyBook', function(
    $scope,   $q,   $stateParams,   Book,   BookFile,   User,   URLBuilder,   AppError,   MyBook
    )
  {
    $scope.myBookStatesPromise = MyBook.states().then(function(states) {
      $scope.myBookStates = states;
    });

    if (!$stateParams.is_trial) {
      MyBook.collection().then(function(collection) {
        collection.moveUp($stateParams.book_id);
      });
    }

    var success = function(results) {
      var version = '1.2',
          book = results.book,
          user = results.user,
          file = results.file;

      var canvas = document.getElementById('reader-canvas');

      var localBookId    = book.id;
      var localBookmarks = new LocalBookmarks.LocalBookmarksClass(localBookId.toString());

      var progressBar = new BarClassRe.BarClass('progress', '#reader-footer .progressbar', true, function(val) {
        reader.GoToPercent(val);
      });

      var readerSite   = new FB3ReaderSite.ExampleSite(canvas, progressBar);
      var dataProvider = file.provider();
      var ppCache      = new FB3PPCache.PPCache();
      var readerDOM    = new FB3DOM.DOM(readerSite, readerSite.Progressor, dataProvider, ppCache);
      
      var bookmarksProcessor = new FB3Bookmarks.LitResBookmarksProcessor(readerDOM, user.session_id, localBookmarks.GetCurrentArtBookmarks());
      bookmarksProcessor.FB3DOM.Bookmarks.push(bookmarksProcessor);
      
      var reader = new FB3Reader.Reader(book.uuid, true, readerSite, readerDOM, bookmarksProcessor, version, ppCache);

      reader.NColumns = 1;
      reader.Init([0]);


      $scope.nextPage = function() {
        reader.PageForward();
      };

      $scope.prevPage = function() {
        reader.PageBackward();
      };

      $scope.swipeLeft = function () {
        $scope.nextPage();
      };

      $scope.swipeRight = function () {
        $scope.prevPage();
      };

      window.reader = reader;
      window.bp = bookmarksProcessor;

      $scope.openContents  = function() {
        $scope.contentsOpened  = true;
        $scope.bookmarksOpened = false;
        $scope.contents        = file.contents(reader);
      };

      $scope.openBookmarks = function() {
        $scope.bookmarksOpened = true;
        $scope.contentsOpened  = false;
        $scope.bookmarks       = file.bookmarks(reader);
      };

      $scope.bookmarks = file.bookmarks(reader);
      window.bookmarks = $scope.bookmarks;

      $scope.closeContents  = function() { $scope.contentsOpened  = false };
      $scope.closeBookmarks = function() { $scope.bookmarksOpened = false };

      $scope.toggleContents  = function() {
        $scope.contentsOpened ? $scope.closeContents() : $scope.openContents()
      };

      $scope.toggleBookmarks  = function() {
        $scope.bookmarksOpened ? $scope.closeBookmarks() : $scope.openBookmarks()
      };

      $scope.goto = function(toc) {
        reader.GoTO([toc]);
        $scope.closeContents();
      };

      $scope.gotoBookmark = function(bookmark) {
        reader.GoTO(reader.Bookmarks.Bookmarks[bookmark.number].Range.From);
        $scope.closeBookmarks();
      }

      $scope.addBookmark = function() {
        if ($scope.hasBookmarkOnCurrentPos) {
          var BookmarksToDelete = bookmarksProcessor.GetBookmarksInRange(1);

          for (var j = 0; j < BookmarksToDelete.length; j++) {
              for (var i = 0; i < bookmarksProcessor.Bookmarks.length; i++) {
                  if (BookmarksToDelete[j].ID == bookmarksProcessor.Bookmarks[i].ID) {
                      bookmarksProcessor.Bookmarks[i].Detach();
                      break;
                  }
              }
          }

          $scope.hasBookmarkOnCurrentPos = false;

          $scope.storeBookmarks();
          reader.Redraw();

          return;
        };

        var bookmark = new FB3Bookmarks.Bookmark(bookmarksProcessor),
            i;

        var range = reader.GetVisibleRange();

        if (!range) {
          range = bookmarksProcessor.Bookmarks[0].Range;
        } else if (range.From.length > 1 && range.From[0] != range.To[0]) {
          var NextEl = reader.FB3DOM.GetElementByAddr([1 + range.From[0]]);

          if (NextEl && NextEl.TagName != 'title') {
            range.From = [1 + range.From[0]];
          }
          range.To = range.From;
        }

        var ObjectPos = reader.FB3DOM.GetElementByAddr(range.From).Position();

        if (!ObjectPos || ObjectPos.length < 1) {
          return undefined;
        }

        bookmark.Range.From = ObjectPos.slice(0);
        bookmark.Range.To = ObjectPos;
        bookmark = bookmark.RoundClone(true);
        bookmark.Group = 1

        function NoteCheckTag(Note) {
            var pos = Note.Range.From[0];

            while (Note.Owner.FB3DOM.Childs[pos].TagName == 'empty-line') {
                pos++;
                if (Note.Owner.FB3DOM.Childs[pos]) {
                    Note.Range.From = pos.slice(0);
                    Note.Range.To = pos;
                } else {
                    break;
                }
            }
            return Note;
        }

        function PrepareTitle(str) {
          return str.replace(/\[\d+\]|\{\d+\}/g, '');
        }

        function GetTitleFromTOC(Range, TOC) {
          var TOC = TOC || reader.TOC();
          for (var j = 0; j < TOC.length; j++) {
            var row = TOC[j];
            var xps = FB3Reader.PosCompare(Range.From, [row.s]);
            var xpe = FB3Reader.PosCompare(Range.To, [row.e]);
            if (xps >= 0 && xpe <= 1) {
              var title = row.t;
              if (row.c) {
                var childTitle = GetTitleFromTOC(Range, row.c);
                if (childTitle) {
                  title = childTitle;
                }
              }
              if (title === undefined) {
                return undefined;
              }
              return PrepareTitle(title);
            }
          }
        }

        bookmark = NoteCheckTag(bookmark);
        bookmark.Title = GetTitleFromTOC(bookmark.Range).substr(0, 100);

        if (!bookmark.Title) {
            bookmark.Title = 'Закладка';
        }

        bookmarksProcessor.AddBookmark(bookmark);

        $scope.storeBookmarks();

        reader.Redraw();
      }


      // private MakeContent() {
      //   this.PrepareData();
      //   this.ObjList.innerHTML = this.ParseWindowData();
      //   this.SetHandlers();
      // }
      // private ParseWindowData() {
      //   var html = '';
      //   if (!this.WindowData.length) {
      //     return '<li><div class="bookmark-top">Нет заметок/закладок</div></li>';
      //   }
      //   var title = '';
      //   for (var j = 0; j < this.WindowData.length; j++) {
      //     if (this.WindowData[j].TemporaryState == 1) {
      //       continue;
      //     }
      //     var bookmark = this.WindowData[j];
      //     var text = bookmark.MakePreviewFromNote();
      //     html += '<li data-n="' + bookmark.N + '" ' +
      //       'data-id="' + bookmark.ID + '" ' +
      //       (bookmark.Group == 3 || bookmark.Group == 5 ? 'class="' + bookmark.Class + '" ' : '') +
      //       '>';
      //     if (title != bookmark.Title) {
      //       html += '<div class="bookmark-top">' + bookmark.Title + '</div>';
      //     }
      //     title = bookmark.Title;
      //     html += '<div class="bookmark-body">';
      //     html += '<div class="bookmark-text">' +
      //       '<span class="icon-type icon-type-' + bookmark.Group + '"></span>' +
      //       '<a href="javascript:void(0);" data-e="' + bookmark.Range.From + '">' + text + '</a></div>';
      //     if (bookmark.Group == 3 || bookmark.Group == 5) {
      //       html += this.CommentObj.MakeComment(bookmark.Note[1]);
      //     }
      //     html += '<div class="bookmark-buttons">' +
      //       '<a class="drop-bookmark action-icon" title="Удалить" ' +
      //         'data-id="' + bookmark.ID + '" href="javascript:void(0);">x</a>';
      //     if (bookmark.Group == 3) {
      //       html += this.CommentObj.MakeButton(bookmark.N);
      //     }
      //     html += this.ShareListObj.MakeButton(bookmark.N);
      //     html += '</div>' +
      //       '</div>' +
      //       '</li>';
      //   }
      //   return html;
      // }

      
      reader.CanvasReadyCallback = function (Range) {
        function findFirstPageBookmark(Range) {
          var pageBookmarks = bookmarksProcessor.GetBookmarksInRange(1, Range);

          if (bookmarksProcessor.GetBookmarksInRange(1, Range).length) {
            return pageBookmarks[0];
          } else {
            return false;
          }
        }

        $scope.firstPageBookmark = findFirstPageBookmark(Range);

        window.firstPageBookmark = $scope.firstPageBookmark;

        if ($scope.firstPageBookmark) {
          $scope.hasBookmarkOnCurrentPos = true;
        } else {
          $scope.hasBookmarkOnCurrentPos = false;
        }
      };

      $scope.canvasAddBookmark = function($event) {
        var target   = $event.changedTouches[0],
            bookmark = new FB3Bookmarks.Bookmark(bookmarksProcessor),
            elem     = reader.ElementAtXY(target.clientX, target.clientY);

        bookmark.TemporaryState = 1;
        bookmark.Group = 1

        bookmark.Range.From = elem;
        bookmark.Range.To   = elem;
        bookmark = bookmark.RoundClone(true);

        bookmarksProcessor.AddBookmark(bookmark);

        $scope.storeBookmarks();

        reader.Redraw();
      }


      $scope.removeBookmark = function(bookmark) {
        var index = $scope.bookmarks.indexOf(bookmark);
        
        $scope.bookmarks.splice($scope.bookmarks.indexOf(bookmark), 1);

        reader.Bookmarks.Bookmarks[bookmark.number].Detach();
        $scope.storeBookmarks();

        $scope.bookmarks = file.bookmarks(reader);
        reader.Redraw();
      }

      $scope.storeBookmarks = function() {
        // Store bookmark local
        localBookmarks.StoreBookmarks(bookmarksProcessor.MakeStoreXML());

        // Store bookmark remote
        bookmarksProcessor.Store();
      }

      $scope.canvasClick = function($event) {
        $event.pageX;
        $event.pageY;

        var canvas       = document.getElementById('reader-canvas');
            canvasXCoord = $event.pageX - canvas.offsetLeft,
            canvasWidth  = canvas.offsetWidth;

        if (canvasXCoord >= 0 && canvasXCoord <= canvasWidth) {
          if ((canvasXCoord / canvasWidth) > 0.2) {
            $scope.nextPage();
          } else {
            $scope.prevPage();
          }
        }
      };

      $scope.book = book
    }

    var failure = function(errors) {
      $scope.pageError = AppError.wrap(errors);
    }

    $q.all({
      book:  Book.element($stateParams.book_id).$promise,
      user:  User.current().$promise,
      file:  BookFile.find($stateParams.book_id, $stateParams.is_trial),
      state: $scope.myBookStatesPromise
    }).then(success, failure);

    $scope.trial = $stateParams.is_trial
  }
]);
angular.module('readerApp').factory('URLBuilder', [
  function()
  {
    return {
      build: function(path, params) {
        var serializeParams = function(params) {
          var strings = [],
              p;
          
          for(prop in params) {
            if(angular.isArray(params[prop])) {
              if (params[prop].$comma) {
                var values = [];

                angular.forEach(params[prop], function(el) {
                  values.push(encodeURIComponent(el));
                });

                strings.push(encodeURIComponent(prop) + "=" + values.join(','));
              } else {
                angular.forEach(params[prop], function(el) {
                  strings.push(encodeURIComponent(prop) + "=" + encodeURIComponent(el));
                });
              }
            } else {
              strings.push(encodeURIComponent(prop) + "=" + encodeURIComponent(params[prop]));
            }
          }
            
          return strings.join("&");
        };

        var paramsString = serializeParams(params || {});

        if (paramsString) {
          return("http://ff-read.litres.ru" + path + "?" + serializeParams(params));
        } else {
          return("http://ff-read.litres.ru" + path);
        }
      }
    }
  }]);
angular.module('readerApp').factory('KeyVal', [
    '$indexedDB', '$q', 'AppError', function(
     $indexedDB,   $q,   AppError
  )
  {
    const STORE_NAME = 'keyval';

    var keyCheck = function(key) {
      if (typeof(key) != 'string') {
        throw AppError.application('ключ не является строкой');
      }

      if (key.length <= 0) {
        throw AppError.application('длина ключа не может быть нулевой');
      }      
    };

    var defaultOptions = {
      reject: true // Reject promise if key not found
    };

    // In-memory cache for KeyVal store
    var cache = {};

    window.keyValCache = cache;

    return {
      get: function(key, options) {
        options = angular.extend({}, defaultOptions, options);

        var storeOpened = function(store) {
          return store.find(key);
        };

        var findSuccess = function(result) {
          cache[key] = result;
          return result.val;
        };

        var findFailure = function() {
          if (options.reject) {
            throw AppError.databaseKeyNotFound(key);
          } else {
            return undefined;
          }
        };

        if (cache.hasOwnProperty(key)) {
          return $q.when(findSuccess(cache[key]));
        } else {
          return $indexedDB.openStore(STORE_NAME, storeOpened, 'readonly').then(
            findSuccess,
            findFailure
          );
        }

      },

      set: function(key, val) {
        keyCheck(key);

        var storeObject = {key: key, val: val};

        var storeOpened = function(store) {
          return store.upsert(storeObject);
        };

        var upsertSuccess = function() {
          cache[key] = storeObject;
          return val;
        };

        var upsertFailure = function() {
          throw AppError.databaseFailure();
        };

        return $indexedDB.openStore(STORE_NAME, storeOpened, 'readwrite').then(
          upsertSuccess,
          upsertFailure
        );
      },

      del: function(key) {
        var storeOpened = function(store) {
          return store.delete(key);
        };

        var deleteSuccess = function() {
          delete cache[key];
        };

        var deleteFailure = function() {
          throw AppError.databaseFailure();
        };

        return $indexedDB.openStore(STORE_NAME, storeOpened, 'readwrite').then(
          deleteSuccess,
          deleteFailure
        );
      }
    };
  }]);
angular.module('readerApp').factory('APICache', [
    '$q', '$interval', '$indexedDB', function(
     $q,   $interval,   $indexedDB
  )
  {
    const STORE_NAME     = 'cache';
    const DAYS_TO_DELETE = 30;

    const COLLECTION_KEY_PREFIX = 'c';
    const ELEMENT_KEY_PREFIX    = 'e';

    var wifiOnly = false;

    var defaultSerializeParams = function(params) {
      if (angular.isObject(params)) {
        return JSON.stringify(params);
      } else {
        throw new Error("Unsupported params type for serialize: " + typeof(params));
      }
    };

    var APICache = function(name, options) {
      this.name            = name;
      this.expires         = options.expires || 1;
      this.serializeParams = options.serializeParams || defaultSerializeParams;

      this.fetchCollection        = options.fetchCollection;
      this.splitCollectionElemets = options.splitCollectionElemets;
      this.fetchElement           = options.fetchElement;

      if (!angular.isString(name) || name.length <= 0) {
        throw new TypeError("name should be string with length >= 0");
      }

      if (!angular.isFunction(options.fetchCollection)) {
        throw new TypeError("option fetchCollection is not a function");
      }

      if (!angular.isFunction(options.splitCollectionElemets) && options.splitCollectionElemets) {
        throw new TypeError("option splitCollectionElemets is not a function");
      }

      if (!angular.isFunction(options.fetchElement) && options.fetchElement) {
        throw new TypeError("option fetchElement is not a function");
      }
    };

    APICache.prototype.collection = function(params, options) {
      var cacheKey = COLLECTION_KEY_PREFIX + ":" + this.name + ":"  + this.serializeParams(params || {}),
          self     = this;

      options = options || {};

      var cacheFound = function(cache) {
        if (self.isUseful(cache)) {
          return cache.value;
        } else {
          return cacheNotFoundOrExpired();
        }
      };

      var cacheNotFoundOrExpired = function() {
        return self.fetchCollection(params).then(function(collection) {
          self.storeCollection(cacheKey, collection, options);
          return collection;
        });
      };

      var openStoreSuccess = function(store) {
        return store.find(cacheKey);
      }

      if (options.forceReload) {
        return cacheNotFoundOrExpired();
      } else {
        return $indexedDB
          .openStore(STORE_NAME, openStoreSuccess, 'readonly')
          .then(cacheFound, cacheNotFoundOrExpired);
      }

    };

    APICache.prototype.element = function(searchKeyValue, options) {
      var cacheKey = ELEMENT_KEY_PREFIX + ":" + this.name + ":" + searchKeyValue,
          self     = this;

      options = options || {};

      if (!self.fetchElement) {
        throw new Error("Single element request not supported");
      }

      var cacheFound = function(cache) {
        if (self.isUseful(cache)) {
          return cache.value;
        } else {
          return cacheNotFoundOrExpired();
        }
      };

      var cacheNotFoundOrExpired = function() {
        return self.fetchElement(searchKeyValue).then(function(element) {
          self.storeElement(element, options);
          return element.value;
        });
      }

      var openStoreSuccess = function(store) {
        return store.find(cacheKey);
      }

      if (options.forceReload) {
        return cacheNotFoundOrExpired();
      } else {
        return $indexedDB
          .openStore(STORE_NAME, openStoreSuccess, 'readonly')
          .then(cacheFound, cacheNotFoundOrExpired);
      }
    };

    APICache.prototype.storeCollection = function(collectionCacheKey, collectionSource, options) {
      var self = this,
          expires,
          collection = angular.copy(collectionSource);

      expires = new Date();
      expires.setMinutes(expires.getMinutes() + self.expires);

      var openStoreSuccess = function(store) {
        store.upsert({
          cache_key: collectionCacheKey,
          expires:   expires,
          value:     collection,
          permanent: !!options.permanent
        });

        if (self.splitCollectionElemets) {
          var nextElement = self.splitCollectionElemets(collection),
              element;

          for (element = nextElement(); element; element = nextElement()) {
            self.storeElement(element, angular.extend({}, options, {store: store}));
          }
        }
      };

      return $indexedDB.openStore(STORE_NAME, openStoreSuccess, 'readwrite');
    }

    APICache.prototype.storeElement = function(element, options) {
      options = options || {};

      var expires = options.expires,
          store   = options.store,
          self    = this;

      if (!expires) {
        expires = new Date();
        expires.setMinutes(expires.getMinutes() + self.expires);
      }

      var upsertElement = function(store) {
        elementCacheKey = ELEMENT_KEY_PREFIX + ":" + self.name + ":" + element.pk;

        store.upsert({
          cache_key: elementCacheKey,
          expires:   expires,
          value:     element.value,
          permanent: !!options.permanent
        });
      }

      if (store) {
        upsertElement(store);
      } else {
        $indexedDB.openStore(STORE_NAME, upsertElement, 'readwrite');
      }
    }

    APICache.prototype.isUseful = function(cacheElement) {
      var isConnected;

      if (navigator.connection !== undefined) {
        if (wifiOnly) {
          isConnected = (navigator.connection.type == "wifi");
        } else {
          isConnected = (navigator.connection.type != "none");
        }
      } else if (navigator.onLine !== undefined) {
        isConnected = navigator.onLine;
      } else {
        isConnected = true;
      }

      return cacheElement && (cacheElement.expires > new Date() || !isConnected);
    }

    // Clean old cached values
    var cleanCache = function() {
      var deleteValues = function(caches) {
        var openStoreForDelete = function(store) {
          angular.forEach(caches, function(cache) {
            if (!cache.permanent) {
              store.delete(cache.cache_key);
            }
          })
        }

        $indexedDB
          .openStore(STORE_NAME, openStoreForDelete, 'readwrite')
      }

      var openStoreSuccess = function(store) {
        var dateToFilter;

        dateToFilter = new Date();
        dateToFilter.setDate(dateToFilter.getDate() - DAYS_TO_DELETE);

        var query = store.query().$index('expires').$lte(dateToFilter);
        return store.eachWhere(query);
      }

      $indexedDB
        .openStore(STORE_NAME, openStoreSuccess, 'readonly')
        .then(deleteValues);
    }

    // Clear APICache very 5 minutes
    $interval(cleanCache, 1000 * 60 * 5);

    return {
      create: function(name, options) {
        return new APICache(name, options);
      },

      setWifiOnly: function(wifiOnlyValue) {
        wifiOnly = wifiOnlyValue;
      }
    };
  }]);
angular.module('readerApp').factory('IMGCache', [
    '$q', '$http', '$interval', '$timeout', '$indexedDB', 'deviceStorage', 'uuid', 'KeyVal', 'AppError', function(
     $q,   $http,   $interval,   $timeout,   $indexedDB,   deviceStorage,   uuid,   KeyVal,   AppError
  )
  {
    const STORE_NAME = "images";
    const CACHE_SIZE = 1000; // Images count
    var   folder     = deviceStorage.folder('litres/images');

    var fetchImage = function(url) {
      var getSuccess = function(response) {
        var contentType = response.headers('content-type') || "image/jpeg",
            fileName    = uuid(),
            blob        = new Blob([response.data], {type: contentType});

        var putSuccess = function(file) {
          var openStoreSuccess = function(store) {
            store.upsert({
              url:         url,
              file_name:   fileName,
              last_access: new Date(),
              size:        file.fileSize
            });
          };

          $indexedDB.openStore(STORE_NAME, openStoreSuccess, 'readwrite');
        }

        folder.put(blob, fileName).then(putSuccess);
        
        return window.URL.createObjectURL(blob);
      }

      return $http.get(url, {responseType: "arraybuffer"}).then(getSuccess);
    };

    var cleanCache = function() {
      var deleteFiles = function(files) {
        var openStoreForDelete = function(store) {
          angular.forEach(files, function(file) {
            store.delete(file.url);
          });
        }

        return $indexedDB
          .openStore(STORE_NAME, openStoreForDelete, 'readwrite').finally(function() {
            angular.forEach(files, function(file) {
              folder.del(file.fileName);
            });
          });
      }

      var openStoreSuccess = function(store) {
        var cleanCount;

        var requestSuccess = function(result) {
          var total = result.length - CACHE_SIZE,
              fileName,
              filesToDelete = [],
              i;

          for (i = 0; i < total; ++i) {
            filesToDelete.push({url: result[i].url, fileName: result[i].file_name});
          }

          return deleteFiles(filesToDelete);
        }

        var countSuccess = function(count) {
          cleanCount = count - CACHE_SIZE;

          if (cleanCount > 0) {
            var query = store.query().$index('last_access').$asc();
            return store.eachWhere(query);
          } else {
            throw "reject";
          }
        }

        return store.count().then(
          countSuccess
        ).then(
          requestSuccess
        );
      }

      return $indexedDB
        .openStore(STORE_NAME, openStoreSuccess, 'readonly')
        .then();
    }

    // Shrink cache for CACHE_SIZE images evert 5 minutes
    $interval(cleanCache, 1000 * 60 * 5);

    // Buffer for batch update of image 'last_access' fields
    var lastAccessUpdateBuffer = [];

    var updateLastAccess = function() {
      var buffer = angular.copy(lastAccessUpdateBuffer);
      
      lastAccessUpdateBuffer = [];

      var openStoreSuccess = function(store) {
        angular.forEach(buffer, function(img) {
          img.last_access = new Date();
          store.upsert(img);
        });
      }

      $indexedDB.openStore(STORE_NAME, openStoreSuccess, 'readwrite');
    }

    // Update 'last_access' for images every 30 seconds
    $interval(updateLastAccess, 1000 * 30);

    return {
      get: function(url) {
        var openStoreSuccess = function(store) {
          return store.find(url);
        };

        var fetchThisImage = function() {
          if (navigator.connection == undefined) {
            // Networking API is not supported, we cannot
            // check WiFi connection
            return fetchImage(url);
          }

          if (navigator.connection.type == "wifi") {
            // Current connection is WiFi, it's ok regardless
            // 'Only Wifi' option
            return fetchImage(url);
          }

          // Check current option state
          return KeyVal.get('settings_wifi_only', {reject: false}).then(function(value) {
            if (value) {
              // Options is enabled, interrupt request
              throw AppError.notWifiConnection();
            } else {
              // Option is disabled, continue request
              return fetchImage(url);
            }
          });
        };

        var findSuccess = function(cache) {
          lastAccessUpdateBuffer.push(cache);

          var storageGetSuccess = function(file) {
            return window.URL.createObjectURL(file);
          };

          return folder.get(cache.file_name).then(
            storageGetSuccess,
            fetchThisImage
          );
        };

        return $indexedDB
          .openStore(STORE_NAME, openStoreSuccess, 'readonly')
          .then(findSuccess, fetchThisImage);
      },

      clean: function() {
        return cleanCache();
      }
    }
  }]);
angular.module('readerApp').factory('AppError', function() {
  var AppError = function(message, systemName) {
    Error.call(this, message);

    this.message    = message;
    this.name       = 'AppError';
    this.systemName = systemName;
  }

  AppError.prototype = Object.create(Error.prototype);
  AppError.prototype.constructor = AppError;

  return {
    create: function(message, systemName) {
      return new AppError(message, systemName);
    },

    serverUnavailable: function() {
      return new AppError('сервер недоступен', 'serverUnavailable');
    },

    connectionTimeout: function() {
      return new AppError('превышено время ожидания ответа', 'connectionTimeout');
    },

    unexpectedResponseStatus: function(status) {
      return new AppError('сервер вернул статус ' + status.toString(), 'unexpectedResponseStatus');
    },

    unexpectedResponseData: function() {
      return new AppError('неожиданный результат запроса', 'unexpectedResponseData');
    },

    validation: function(errorMessage) {
      return new AppError(errorMessage, 'validation');
    },

    databaseFailure: function() {
      return new AppError('ошибка базы данных', 'databaseFailure');
    },

    databaseKeyNotFound: function(key) {
      return new AppError('ключ не найден: ' + key, 'databaseKeyNotFound');
    },

    authorizationFailed: function() {
      return new AppError('ошибка авторизации', 'authorizationFailed');
    },

    application: function(errorMessage) {
      return new AppError('Внутренняя ошибка приложения: ' + message, 'application');
    },

    notWifiConnection: function() {
      return new AppError('Получение данных разрешено только при использовании Wi-Fi соединения', 'notWifiConnection');
    },

    notError: function() {
      return new AppError('Не ошибка', 'notError')
    },

    unknown: function(arguments) {
      console.error('unknown error:', arguments);

      var appError = new AppError('неизвестная ошибка', 'unknown');
      appError.arguments = arguments;

      return appError;
    },

    wrap: function(error) {
      if (error instanceof AppError) {
        return error;
      } else {
        return this.unknown(error);
      }
    },

    is: function(error) {
      return(error instanceof AppError);
    }
  };
});
angular.module('readerApp').factory('deviceId', [
    'KeyVal', 'uuid', function(
     KeyVal,   uuid
  ) {
    const DEVICE_ID_STORE_KEY = 'device_id'

    var deviceIdPromise ;

    return function() {
      if (deviceIdPromise === undefined) {
        var deviceIdNotFound = function() {
          var deviceId = "ff_" + uuid();

          return KeyVal.set(DEVICE_ID_STORE_KEY, deviceId).then(function() {
            return deviceId;
          });
        };

        deviceIdPromise = KeyVal.get(DEVICE_ID_STORE_KEY).catch(deviceIdNotFound);
      }

      return deviceIdPromise;
    }
  }]);
angular.module('readerApp').factory('deviceStorage', [
    '$q', function(
     $q
  )
  {
    var Folder = function(storage, name) {
      this.storage = storage;
      this.name    = name;
    };

    Folder.prototype.get = function(name) {
      return this.storage.get(this.name + "/" + name);
    };

    Folder.prototype.put = function(blob, name) {
      return this.storage.put(blob, this.name + "/" + name);
    };

    Folder.prototype.del = function(name) {
      return this.storage.del(this.name + "/" + name);
    }


    var DeviceStorage = function() {
      this.storage = navigator.getDeviceStorage("sdcard");
    };

    DeviceStorage.prototype.put = function(blob, name) {
      var defered = $q.defer(),
          self    = this,
          request;

      if (name === undefined) {
        request = this.storage.add(blob);
      } else {
        request = this.storage.addNamed(blob, name);
      }

      request.onsuccess = function() {
        var getSuccess = function(file) {
          defered.resolve(file);
        }

        var getFailure = function(error) {
          defered.reject(error);
        }

        self.get(name).then(getSuccess, getFailure);
      }

      request.onerror = function() {
        defered.reject(request.error);
      }

      return defered.promise;
    };

    DeviceStorage.prototype.get = function(name) {
      var defered = $q.defer(),
          request;

      request = this.storage.get(name);
      
      request.onsuccess = function() {
        defered.resolve(request.result);
      }

      request.onerror = function() {
        defered.reject(request.error);
      }

      return defered.promise;
    }

    DeviceStorage.prototype.del = function(name) {
      var defered = $q.defer(),
          request;

      request = this.storage.delete(name);
      
      request.onsuccess = function() {
        defered.resolve(request.result);
      }

      request.onerror = function() {
        defered.reject(request.error);
      }

      return defered.promise;
    }

    DeviceStorage.prototype.folder = function(name) {
      return new Folder(this, name);
    };

    return new DeviceStorage();
  }]);
angular.module('readerApp').factory('uuid', function() {
  return function() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
      var r = Math.random() * 16 | 0,
          v = (c == 'x') ? r : (r & 0x3 | 0x8);

      return v.toString(16);
    });
  };
});
angular.module('readerApp').factory('TaskRunner', [
    '$q', 'AppError', function(
     $q,   AppError
  )
  {
    var TaskRunner = function(tasks, maxConcurrentCount, workerFunction) {
      if (!angular.isArray(tasks)) {
        throw AppError.application('tasks should be an array');
      }

      if (!angular.isNumber(maxConcurrentCount)) {
        throw AppError.application('maxConcurrentCount should be a positive integer');
      }

      if (!angular.isFunction(workerFunction)) {
        throw AppError.application('workerFunction should be a function');
      }

      this.tasks              = tasks;
      this.maxConcurrentCount = maxConcurrentCount;
      this.taskCount          = tasks.length;
      this.workerFunction     = workerFunction;
      this.nextTaskIndex      = 0;

      this.defered = $q.defer();
      this.promise = this.defered.promise;
      this.results = {};
      this.stopped = false;
    };

    TaskRunner.prototype.runNext = function() {
      var currentTaskIndex = this.nextTaskIndex,
          self = this;

      if (this.stopped) {
        return;
      }

      if (currentTaskIndex < this.taskCount) {
        this.nextTaskIndex++;

        var nextSuccess = function(result) {
          self.results[currentTaskIndex] = result;
          self.defered.notify({index: currentTaskIndex, result: result});
          self.runNext();
        }

        var nextFailure = function(result) {
          self.defered.reject(result);
        };
        
        try {
          nextPromise = this.workerFunction(this.tasks[currentTaskIndex], currentTaskIndex);
        } catch (error) {
          this.defered.reject(AppError.wrap(error));
        };

        nextPromise.then(nextSuccess, nextFailure);
      } else if (Object.keys(this.results).length == this.taskCount) {
        this.defered.resolve(this.results);
      }
    }

    TaskRunner.prototype.run = function() {
      do {
        this.runNext();
      } while (this.nextTaskIndex < Math.min(this.taskCount, this.maxConcurrentCount));

      return this.promise;
    };

    TaskRunner.prototype.stop = function() {
      this.stopped = true;
      
      if (this.defered.promise.$$state.status == 0) {
        this.defered.reject(AppError.notError());
      }
    }

    return {
      create: function(tasks, maxConcurrentCount, workerFunction) {
        return new TaskRunner(tasks, maxConcurrentCount, workerFunction);
      }
    }
  }]);
angular.module('readerApp').factory('BookFileDownload', [
    '$http', '$q', 'URLBuilder', 'User', 'AppError', 'TaskRunner', 'BookFile', 'BookFileDownloadRegistry', '$timeout', 'MyBookStates', function(
     $http,   $q,   URLBuilder,   User,   AppError,   TaskRunner,   BookFile,   BookFileDownloadRegistry,   $timeout,   MyBookStates
  )
  {
    const CONCURRENT_DOWNLOADS = 4;

    // We cannot use any kind of `eval` here, make real JSON from response
    // to use standard parser
    //
    // For example:
    //   {Hello: "world"} => {"Hello": "world"}
    var normalizeJSON = function(source) {
      var inString = false,
          inKey    = false,
          result   = "",
          currChar,
          prevChar,
          i;

      var isQuoteChar = function(char) {
        // return ["'", '"'].indexOf(char) >= 0;
        return char == '"';
      };

      var isKeyChar = function(char) {
        return !!char.match(/^[a-zA-Z][a-zA-Z0-9_]?$/);
      };

      for (i = 0; i < source.length; ++i) {
        prevChar = currChar;
        currChar = source[i];

        if (!inKey && !inString) {
          if (isQuoteChar(currChar)) {
            inString = true;
            result   = result + currChar;
          } else if (isKeyChar(currChar)) {
            inKey    = true;
            result   = result + '"' + currChar;
          } else {
            result   = result + currChar;
          }

          continue;
        }

        if (inKey) {
          if (isKeyChar(currChar)) {
            result = result + currChar;
          } else {
            result = result + '"' + currChar;
            inKey  = false;
          }

          continue;
        }

        if (inString) {
          result = result + currChar;

          if (isQuoteChar(currChar) && prevChar != "\\") {
            inString = false;
          }

          continue;
        }
      }

      return result;
    };


    var parseResponse = function(source) {
      return JSON.parse(normalizeJSON(source));
    };


    var httpFailure = function(response) {
      if (AppError.is(response)) {
        throw response;
      } else if (response.status == 0) {
        throw AppError.serverUnavailable();
      } else if (response.status) {
        throw AppError.unexpectedResponseStatus(response.status.toString());
      } else {
        throw AppError.unknown();
      }
    };


    var BookFileDownload = function (bookId, isTrial) {
      BookFileDownloadRegistry.set(bookId, isTrial, this);

      if (!isTrial) {
        MyBookStates.addDownload(bookId, this);
      }

      this.defered = $q.defer();
      this.stopped = false;
      this.result  = {};
      this.promise = this.defered.promise;
      this.isTrial = isTrial;
      this.bookId  = bookId;

      this.stats = {
        partsTotal:  undefined,
        partsReady:  0,
        imagesTotal: undefined,
        imagesReady: 0,
        images:      false
      };

      this.httpConfig = {
        // Do not apply standard JSON transformation here
        transformResponse: [function (data) { return parseResponse(data); }]
      };

      var unregister = function() {
        BookFileDownloadRegistry.del(bookId, isTrial);

        if (!isTrial) {
          MyBookStates.delDownload(bookId);
        }
      }

      this.promise.finally(unregister);
    };

    BookFileDownload.prototype.start = function(baseURL) {
      var self = this;

      if (self.stopped) {
        return;
      }

      var httpTokSuccess = function(response) {
        var toc        = response.data,
            readyCount = 0,
            result     = {toc: toc},
            partFiles,
            runner;

        if (self.stopped) {
          return;
        }

        self.stats.partsTotal = toc['Parts'].length;

        partFiles = toc['Parts'].map(function(part) {
          return part.url;
        });

        var imagesCollect = function(parts) {
          var collectFromList = function(list) {
            if (!angular.isArray(list)) {
              return [];
            }

            var res = [],
                i;

            for (i = 0; i < list.length; ++i) {
              res = res.concat(collectFromPart(list[i]));
            }

            return res;
          };

          var collectFromPart = function(part) {
            if (!angular.isObject(part)) {
              return [];
            }

            if (part.t == 'image' && part.s) {
              var image = {
                src:    part.s,
                width:  part.w,
                height: part.h
              }

              return [image].concat(collectFromList(part.c)).concat(collectFromPart(part.f));
            }

            return collectFromList(part.c).concat(collectFromPart(part.f));
          };

          var images = [],
              partKey,
              part,
              k;

          for (partKey in parts) {
            images = images.concat(collectFromList(parts[partKey]));
          }

          return images;
        }

        var partsSuccess = function(parts) {
          var imagesLinks     = imagesCollect(parts),
              imageReadyCount = 0;

          self.stats.imagesTotal = imagesLinks.length;
          self.stats.imagesTotal.images = true;

          result['parts'] = parts;

          var imagesSuccess = function(images) {
            result['images'] = images;

            return BookFile.save(self.bookId, self.isTrial, result).then(function(bookFile) {
              if (!self.isTrial) {
                MyBookStates.addBought(self.bookId, true);
              }

              self.defered.resolve(bookFile);
            });
          };

          var imagesFailure = function(error) {
            self.defered.reject(error);
          };

          var imagesNotify  = function() {
            imageReadyCount++;
            self.stats.imagesReady++;

            self.defered.notify({
              ready:  imageReadyCount,
              total:  imagesLinks.length,
              images: true
            });
          };

          self.runner = TaskRunner.create(imagesLinks, CONCURRENT_DOWNLOADS, function(fileName, partNum) {
            return self.downloadImage(baseURL, fileName, partNum);
          });

          self.defered.notify({ready: 0, total: imagesLinks.length, images: true});
          return self.runner.run().then(imagesSuccess, imagesFailure, imagesNotify);
        };

        var partsFailure = function(error) {
          self.defered.reject(error);
        };

        var partsNotify  = function() {
          readyCount++;

          self.stats.partsReady++;
          self.defered.notify({
            ready:  readyCount,
            total:  partFiles.length,
            images: false
          });
        };

        self.runner = TaskRunner.create(partFiles, CONCURRENT_DOWNLOADS, function(fileName, partNum) {
          return self.downloadPart(baseURL, fileName, partNum);
        });

        self.defered.notify({ready: 0, total: partFiles.length, images: false});
        self.runner.run().then(partsSuccess, partsFailure, partsNotify);
      }

      $http.get(baseURL + 'toc.js', self.httpConfig).then(
        httpTokSuccess, httpFailure
      );
    }

    BookFileDownload.prototype.downloadPart = function(baseURL, fileName, partNum) {
      var partSuccess = function(response) {
        return response.data;
      };

      // Code for progress bar debug
      // var self = this;

      // return $timeout(function() {
      //   return $http.get(baseURL + fileName, self.httpConfig).then(
      //     partSuccess,
      //     httpFailure
      //   );
      // }, 5000 + Math.floor(Math.random() * 1500));

      return $http.get(baseURL + fileName, this.httpConfig).then(
        partSuccess,
        httpFailure
      );
    };


    BookFileDownload.prototype.downloadImage = function(baseURL, image) {
      var imageSuccess = function(response) {
        var contentType = response.headers('content-type') || "binary/octet-stream";
        
        return angular.extend({}, image, {
          blob: new Blob([response.data], {type: contentType})
        });
      };

      var httpConfig = {responseType: "arraybuffer"};

      return $http.get(baseURL + image.src, httpConfig).then(
        imageSuccess,
        httpFailure
      );
    };

    BookFileDownload.prototype.stop = function() {
      this.stopped = true;

      if (this.runner) {
        this.runner.stop();
      } else if (this.defered.promise.$$state.status == 0) {
        this.defered.reject(AppError.notError());
      }
    };


    var downloadTrial = function(bookId) {
      var download = new BookFileDownload(bookId, true);

      var idMesh = function(id) {
        var meshTmp = '0000',
            mesh = [];

        meshTmp = meshTmp.substring(0, 8 - id.length) + id;
        mesh.push(meshTmp.substring(0, 2));
        mesh.push(meshTmp.substring(2, 4));
        mesh.push(meshTmp.substring(4, 6));
        mesh.push(meshTmp);
        
        return mesh;
      }

      var idMeshPart = idMesh(bookId.toString()).join('/');
      var baseURL    = URLBuilder.build('/static/trials/' + idMeshPart + '.json/');

      download.start(baseURL);

      return download;
    };


    var downloadFull = function(bookId) {
      var download = new BookFileDownload(bookId, false);

      User.current().$promise.then(function(currentUser) {
        var baseURL = URLBuilder.build('/pages/catalit_download_book/', {
          art: bookId,
          sid: currentUser.session_id,
          type: 'json',
        }) + '&filename=';

        download.start(baseURL);
      });

      return download;
    };

    return {
      downloadTrial: downloadTrial,
      downloadFull:  downloadFull
    };
  }]);
angular.module('readerApp').factory('BookFileDownloadRegistry', [
    '$q', function(
     $q
  )
  {
    var registry = {};

    var registerKey = function(bookId, isTrial) {
      return bookId.toString() + ":" + isTrial.toString();
    }

    return {
      get: function(bookId, isTrial) {
        var key = registerKey(bookId, isTrial);

        if (!registry[key]) {
          registry[key] = {
            download: null
          };
        }

        return registry[key];
      },

      set: function(bookId, isTrial, downloadInstance) {
        var key = registerKey(bookId, isTrial);

        bookId = bookId.toString();

        if (!registry[key]) {
          registry[key] = {};
        }

        registry[key].download = downloadInstance;
      },

      del: function(bookId, isTrial) {
        var key = registerKey(bookId, isTrial),
            reg = registry[key];

        if (reg) {
          reg.download = null;
        }
      }
    };
  }]);
angular.module('readerApp').factory('BookFile', [
    '$http', '$q', '$indexedDB', '$timeout', 'BookDataProvider', function(
     $http,   $q,   $indexedDB,   $timeout,   BookDataProvider
  )
  {
    var BookFile = function(chunks, images) {
      this.chunks = chunks;
      this.images = images;
    };

    BookFile.prototype.provider = function() {
      return new BookDataProvider(this);
    };

    BookFile.prototype.getChunk = function(chunk) {
      if (chunk == 'toc') {
        return this.chunks.toc;
      } else {
        return this.chunks.parts[chunk];
      }
    };

    BookFile.prototype.getImage = function(image) {
      return this.images[image]
    };

    BookFile.prototype.contents = function(reader) {
      var tocLevelProcess = function(tocLevel) {
        var result = [],
            i;

        for (i = 0; i < tocLevel.length; ++i) {
          var toc  = tocLevel[i],
              item = {};

          if (toc.t) {
            item.name = toc.t;
            item.toc  = toc.s;
          }

          if (toc.bookmarks && toc.bookmarks.g0) {
            item.current = true;
          }

          if (toc.c) {
            item.children = tocLevelProcess(toc.c);
          }

          result.push(item);
        }

        return result;
      }

      return tocLevelProcess(reader.TOC());
    }


    BookFile.prototype.bookmarks = function(reader) {
      var bookmarks = [],
          i;

      for (i = 1; i < reader.Bookmarks.Bookmarks.length; ++i) {
        bookmarks.push({
          number: reader.Bookmarks.Bookmarks[i].N,
          title:  reader.Bookmarks.Bookmarks[i].Title,
          body:   reader.Bookmarks.Bookmarks[i].MakePreviewFromNote()
        });
      };

      return bookmarks;
    }

    // BookFile.prototype.setAttr = function(attr, value) {
    //   attrs = this.getAttrs()
    // };


    var bookStoreKey = function(bookId, isTrial) {
      if (isTrial) {
        return "trial:" + bookId.toString();
      } else {
        return "full:" + bookId.toString();
      }
    }

    return {
      find: function(bookId, isTrial) {
        var openStoreSuccess = function(chunksStore, imagesStore) {
          var chunksPromise = chunksStore.eachWhere(
            chunksStore.query().$index('book_key').$eq(bookStoreKey(bookId, isTrial))
          );

          var imagesPromise = imagesStore.eachWhere(
            chunksStore.query().$index('book_key').$eq(bookStoreKey(bookId, isTrial))
          );

          return $q.all({
            chunks: chunksPromise,
            images: imagesPromise
          });
        }

        var findSuccess = function(results) {
          var chunksResult = results.chunks,
              imagesResult = results.images,
              parts  = [],
              images = {},
              part,
              toc,
              i;

          if (chunksResult.length == 0) {
            // Book not found, reject promise
            throw "not_found";
          }

          for (i = 0; i < chunksResult.length; ++i) {
            part = chunksResult[i];

            if (part.chunk_key == 'toc') {
              toc = part.chunk;
            } else {
              parts[part.chunk_key] = part.chunk;
            }
          }

          for (i = 0; i < imagesResult.length; ++i) {
            images[imagesResult[i].image_key] = imagesResult[i].image;
          }

          return new BookFile({toc: toc, parts: parts}, images);
        }

        return $indexedDB
          .openStores(['book_file_chunks', 'book_file_images'], openStoreSuccess, 'readonly')
          .then(findSuccess);
      },

      isSaved: function(bookId, isTrial) {
        var openStoreSuccess = function(store) {
          return store.eachWhere(
            store.query().$index('book_key').$eq(bookStoreKey(bookId, isTrial))
          );
        };

        var findSuccess = function(chunks) {
          return chunks.length > 0;
        }

        return $indexedDB
          .openStore('book_file_chunks', openStoreSuccess, 'readonly')
          .then(findSuccess);
      },

      save: function(bookId, isTrial, data) {
        var images = [];

        var openStoreSuccess = function(chunksStore, imagesStore) {
          var bookKey = bookStoreKey(bookId, isTrial),
              chunks = [],
              prop,
              imageKey,
              chunkKey;

          var pushChunk = function(chunkKey, chunk) {
            chunks.push({
              id:        bookKey + ":" + chunkKey.toString(),
              book_key:  bookKey,
              chunk_key: chunkKey.toString(),
              chunk:     chunk
            });
          }

          pushChunk('toc', data['toc']);

          for (chunkKey in data['parts']) {
            pushChunk(chunkKey, data['parts'][chunkKey]);
          }

          for (prop in data['images']) {
            imageKey = data['images'][prop].src;

            images.push({
              id:        bookKey + ":" + imageKey.toString(),
              book_key:  bookKey,
              image_key: imageKey,
              image:     data['images'][prop]
            });
          }

          return $q.all({
            chunks: chunksStore.upsert(chunks),
            images: imagesStore.upsert(images)
          });
        }

        var upsertSuccess = function() {
          var imagesObject = {},
              i;

          for (i = 0; i < images.length; ++i) {
            imagesObject[images[i].image_key] = images[i].image;
          }

          return new BookFile(data, imagesObject);
        }

        var upsertFailure = function(error) {
          throw error;
        }

        return $indexedDB
          .openStores(['book_file_chunks', 'book_file_images'], openStoreSuccess, 'readwrite')
          .then(upsertSuccess);
      },

      remove: function(bookId, isTrial) {
        var removeAllKeys = function(results) {
          var doRemoveAllKeys = function(chunksStore, imagesStore) {
            var chunk,
                image,
                i;

            for (i = 0; i < results.chunks.length; ++i) {
              chunk = results.chunks[i];
              chunksStore["delete"](chunk.id); 
            }

            for (i = 0; i < results.images.length; ++i) {
              image = results.images[i];
              imagesStore["delete"](image.id);
            }
          }

          return $indexedDB
            .openStores(['book_file_chunks', 'book_file_images'], doRemoveAllKeys, 'readwrite');
        };

        var openStoreSuccess = function(chunksStore, imagesStore) {
          var chunksPromise = chunksStore.eachWhere(
            chunksStore.query().$index('book_key').$eq(bookStoreKey(bookId, isTrial))
          );

          var imagesPromise = imagesStore.eachWhere(
            chunksStore.query().$index('book_key').$eq(bookStoreKey(bookId, isTrial))
          );

          return $q.all({
            chunks: chunksPromise,
            images: imagesPromise
          });
        };

        return $indexedDB
          .openStores(['book_file_chunks', 'book_file_images'], openStoreSuccess, 'readonly')
          .then(removeAllKeys);
      }
    };
  }]);
angular.module('readerApp').factory('BookDataProvider', [
    '$http', '$q', '$indexedDB', '$timeout', function(
     $http,   $q,   $indexedDB,   $timeout
  )
  {    
    var BookDataProvider = function(bookFile) {
      this.bookFile     = bookFile;
      this.currentReqId = 0;

      this.BaseURL   = "http://litres.ru/";
    };

    BookDataProvider.prototype.ArtID2URL = function (artId, chunk) {
      var image;

      if (this.bookFile.getChunk(chunk || "toc")) {
        return artId.toString() + ":" + (chunk || "toc").toString();
      }

      var image = this.bookFile.getImage(chunk);

      if (image) {
        return window.URL.createObjectURL(image.blob);
      }

      console.error('error: unknown chunk:', chunk);
      return '';
    }

    BookDataProvider.prototype.Request = function(
      partIdentity,
      callback,
      progressor,
      customData
    ) {
      var chunkId = partIdentity.split(':')[1],
          chunk   = this.bookFile.getChunk(chunkId),
          self    = this;

      progressor.HourglassOn(this, false, 'Loading ' + partIdentity);

      $timeout(function() {
        progressor.HourglassOff(this);
        callback.call(this, chunk, customData);
      }, 0);
    };

    BookDataProvider.prototype.Reset = function() {
      // Nothing to do here
    };

    return BookDataProvider;
  }]);
angular.module('readerApp').factory('MyBook', [
    '$q', '$indexedDB', 'Book', 'KeyVal', 'AppError', 'User', 'BookFileDownloadRegistry', 'MyBookStates', 'BookFile', 'BookFileDownload', 'Folder', function(
     $q,   $indexedDB,   Book,   KeyVal,   AppError,   User,   BookFileDownloadRegistry,   MyBookStates,   BookFile,   BookFileDownload,   Folder
  )
  {
    // IndexedDB store name
    const STORE_NAME = "my_books";

    // My books cache lifetime in minutes
    const LIFETIME   = 5;

    const LAST_UPDATE_STORE_KEY = "my_books_last_update";
    const LAST_LOGIN_STORE_KEY  = "my_books_last_login";

    const ARCHIVE_FOLDER_ID = 0;
    const NOT_IN_FOLDER     = -1;


    var currentCollection,        // In-memory cache for current MyBooks collection
        currentCollectionPromise; // Single 'initial loading' promise, used to prevent
                                  // parallel collection loading


    var MyBookCollection = function(myBooks) {
      this.myBooks  = [];
      this.byId     = {};
      this.maxOrder = 0;

      if (myBooks) {
        for (var i = 0; i < myBooks.length; ++i) {
          this.push(myBooks[i]);
        }
      }
    };

    MyBookCollection.prototype.reset = function() {
      this.myBooks  = [];
      this.byId     = {};
      this.maxOrder = 0;
    }

    MyBookCollection.prototype.push = function(myBook) {
      if (!(myBook instanceof MyBook)) {
        throw AppError.application("MyBookCollection#push argument should be MyBook instance");
      }

      if (!this.findMy(myBook.id)) {
        this.myBooks.push(myBook);
        this.byId[myBook.id.toString()] = myBook;

        var order = myBook.getAttr('order');

        if (order) {
          if (order > this.maxOrder) {
            this.maxOrder = order;
          }
        } else {
          myBook.setAttr('order', this.nextOrder());
        }
      }
    };

    MyBookCollection.prototype.size = function() {
      return this.myBooks.length;
    }

    MyBookCollection.prototype.moveUp = function(bookId) {
      var myBook = this.findMy(bookId);

      if (myBook) {
        myBook.setAttr('order', this.nextOrder());
      }
    };

    MyBookCollection.prototype.nextOrder = function() {
      this.maxOrder += 1;
      return this.maxOrder;
    }

    MyBookCollection.prototype.findMy = function(bookId) {
      return this.byId[bookId.toString()];
    };

    var MyBook = function(attributes, book, index) {
      this.id              = parseInt(attributes.id);
      this.downloadRegFull = BookFileDownloadRegistry.get(attributes.id, false);

      this.attributes = attributes;
      this.book       = book || Book.element(this.attributes.id, {permanent: true});
    }

    MyBook.prototype.moveToArchive = function(syncWithServer) {
      var self = this;

      BookFile.remove(self.id, false);
      MyBookStates.addBought(self.id, false);

      if (syncWithServer) {
        self.archivePromise = Folder.put(self.id, ARCHIVE_FOLDER_ID).then(function() {
          return self.setAttr('archive', true);
        });

        return self.archivePromise;
      } else {
        return self.setAttr('archive', true);
      }
    }

    MyBook.prototype.returnFromArchive = function(syncWithServer) {
      var self  = this,
          state = MyBookStates.get()[self.id.toString()];

      if (state && state.state == 'my' && !state.download) {
        BookFileDownload.downloadFull(self.id);
      }

      if (syncWithServer) {
        self.archivePromise = Folder.put(self.id, NOT_IN_FOLDER).then(function() {
          return self.setAttr('archive', false);
        });

        return self.archivePromise;
      } else {
        return self.setAttr('archive', false);
      }
    }

    MyBook.prototype.isArchived = function() {
      return this.getAttr('archive');
    }

    MyBook.prototype.setAttr = function(attr, val, options) {
      if (!options) {
        options = {};
      }

      this.attributes[attr] = val;

      if (options.save !== false) {
        return this.save();
      }
    }

    MyBook.prototype.getAttr = function(attr) {
      return this.attributes[attr];
    }

    MyBook.prototype.file = function() {
      throw "TODO";
    };

    MyBook.prototype.save = function(store) {
      var self = this;

      var storeOpened = function(store) {
        return store.upsert(self.attributes);
      };

      if (store) {
        return storeOpened(store);
      } else {
        return $indexedDB.openStore(STORE_NAME, storeOpened, 'readwrite');
      }
    }

    MyBook.init = function(bookId, book, isArchived) {
      return new MyBook({
        id:       bookId,
        progress: 0,
        archive:  isArchived || false
      }, book);
    }

    var collection = function(options) {
      options = options || {}
      // TODO: reload if promise rejected

      if (currentCollectionPromise === undefined ||
          currentCollectionPromise.$$state.status === 2 ||
          options.forceReload) {

        var fetchSuccess = function(result) {
          var isDownloadPromises = [];

          angular.forEach(result, function(myBook) {
            isDownloadPromises.push(BookFile.isSaved(myBook.id, false));
          });

          return $q.all(isDownloadPromises).then(function(isDownloadResults) {
            if (currentCollection) {
              currentCollection.reset();
            } else {
              currentCollection = new MyBookCollection();
            }

            angular.forEach(result, function(myBook, index) {
              MyBookStates.addBought(myBook.id, isDownloadResults[index]);
              currentCollection.push(myBook);
            });

            return currentCollection;
          });


        };

        var checkCacheExpires = function(cachedBooks) {
          var getSuccess = function(lastUpdateAt) {
            var expireAt;

            expireAt = new Date();
            expireAt.setMinutes(expireAt.getMinutes() - LIFETIME);

            if (!lastUpdateAt || lastUpdateAt < expireAt) {
              throw cachedBooks;
            } else {
              return cachedBooks;
            }
          };

          var getFailure = function() {
            throw cachedBooks;
          }

          if (options.forceReload) {
            throw cachedBooks;
          } else {            
            return KeyVal.get(LAST_UPDATE_STORE_KEY).then(
              getSuccess,
              getFailure
            );
          }
        };

        var fetchAndMergeMyBooks = function(cachedBooks) {
          var cachedBooksById = {},
              defered = $q.defer(),
              errorIsFatal;


          if (!angular.isArray(cachedBooks) || cachedBooks.length <= 0) {
            cachedBooks = [];

            // If cachedBooks is not array, we have no cached MyBook's yet.
            // So, all load errors should lock myBook page (we reject promise)
            errorIsFatal = true;
          } else {
            // If cachedBooks is array, we already cached some MyBook's. So,
            // all load errors should be promise notifications (we show it in
            // alert, and then resolve promise with cached data)
            errorIsFatal = false;
          }

          angular.forEach(cachedBooks, function(cachedBook) {
            cachedBooksById[cachedBook.id] = cachedBook;
          });

          var fetchServerMyBooksSuccess = function(fetchedBooks) {
            KeyVal.set(LAST_UPDATE_STORE_KEY, new Date());

            angular.forEach(fetchedBooks, function(fetchedBook) {
              // TODO: use single transaction to speed up books saving
              //       (already supported in .save method)
              
              if (cachedBooksById[fetchedBook.id]) {
                var cachedArchived  = cachedBooksById[fetchedBook.id].getAttr('archive'),
                    fetchedArchived = fetchedBook.getAttr('archive');

                cachedBooksById[fetchedBook.id].book = fetchedBook.book;

                if (cachedArchived && !fetchedArchived) {
                  cachedBooksById[fetchedBook.id].returnFromArchive();
                } else if (!cachedArchived && fetchedArchived) {
                  cachedBooksById[fetchedBook.id].moveToArchive();
                } else {
                  cachedBooksById[fetchedBook.id].save();
                }
              } else {
                fetchedBook.save();
                cachedBooks.push(fetchedBook);
              };
            });

            defered.resolve(cachedBooks);
          }

          var fetchServerMyBooksFailure = function(error) {
            error = AppError.wrap(error);

            if (errorIsFatal || error.systemName == "authorizationFailed") {
              defered.reject(error);
            } else {            
              defered.notify(error);
              defered.resolve(cachedBooks);
            }
          }

          fetchServerMyBooks().then(
            fetchServerMyBooksSuccess,
            fetchServerMyBooksFailure
          );

          return defered.promise;
        };

        var checkLastLogin = function() {
          var currentUser = User.current();
          var lastLogin   = KeyVal.get(LAST_LOGIN_STORE_KEY, {reject: false});

          var promiseSuccess = function(result) {
            if (result.currentUser.login != result.lastLogin) {
              return clearCachedMyBooks().then(function() {
                return KeyVal.set(LAST_LOGIN_STORE_KEY, result.currentUser.login).then(function() {
                  // This exception should reject promise and cause books refetch
                  throw null;
                });
              });
            }
          };

          return $q.all({
            currentUser: currentUser.$promise,
            lastLogin:   lastLogin
          }).then(
            promiseSuccess
          );
        }

        currentCollectionPromise = checkLastLogin()
          .then (fetchCachedMyBooks)
          .then (checkCacheExpires)
          .catch(fetchAndMergeMyBooks)
          .then (fetchSuccess);
      }
      
      return currentCollectionPromise;
    };

    var fetchCachedMyBooks = function() {
      var storeOpened = function(store) {
        return store.getAll();
      };

      var getAllSuccess = function(results) {
        return results.map(function(attributes) {
          return new MyBook(attributes);
        });
      };

      return $indexedDB.openStore(STORE_NAME, storeOpened, 'readonly').then(
        getAllSuccess
      );
    }

    var clearCachedMyBooks = function(clearInMemoryCache) {
      var storeOpened = function(store) {
        // TODO: remove books from FS or DB
        store.clear();
        MyBookStates.clear();
      };

      return $indexedDB.openStore(STORE_NAME, storeOpened, 'readwrite').then(function() {
        if (clearInMemoryCache) {
          currentCollectionPromise = undefined;
          currentCollection        = undefined;
        }
      });
    }

    var fetchServerMyBooks = function() {
      var collectionSuccess = function(results) { 
        var my = results.my.map(function(book, index) {
          return MyBook.init(book.id, book, false);
        });

        var archive = results.archive.map(function(book, index) {
          return MyBook.init(book.id, book, true);
        });

        return my.concat(archive);
      };

      var myPromise      = Book.my({}, {forceReload: true, permanent: true}).$promise,
          archivePromise = Book.my({my_folder: 0}, {forceReload: true, permanent: true}).$promise;

      return $q.all({
        my:      myPromise,
        archive: archivePromise
      }).then(
        collectionSuccess
      );
    }


    return {
      collection: collection,

      push: function(book) {
        var self = this;

        if (!angular.isObject(book)) {
          return Book.element(book, {permanent: true}).$promise.then(function(book) {
            return self.push(book);
          });
        }

        return BookFile.isSaved(book.id, false).then(function(isDownload) {
          var myBook = MyBook.init(book.id, book);

          MyBookStates.addBought(myBook.id, isDownload);

          return myBook.save().then(function() {
            self.collection().then(function(currentCollection) {
              currentCollection.push(myBook);
            });
          });
        });

      },

      states: function() {
        return this.collection().then(function() {
          return MyBookStates.get();
        })
      },

      clear: function() {
        return clearCachedMyBooks(true);
      }
    };
  }]);
angular.module('readerApp').factory('MyBookStates', [
    '$q', function(
     $q
  )
  {
    var states = {};

    return {
      get: function() {
        return states;
      },

      addBuyInProcess: function(bookId, orderId) {
        bookId = bookId.toString();

        if (!states[bookId] || states[bookId].state == 'buy') {
          states[bookId] = {};
          states[bookId].state = 'buy';
          states[bookId].orderId = orderId
        }
      },

      delBuyInProcess: function(bookId) {
        bookId = bookId.toString();

        if (states[bookId] && states[bookId].state == 'buy') {
          delete states[bookId];
        }
      },

      addBought: function(bookId, isDownload) {
        bookId = bookId.toString();

        states[bookId] = {};

        if (isDownload) {
          states[bookId].state = 'download';
        } else {
          states[bookId].state = 'my';
        }

        return states[bookId];
      },

      addDownload: function(bookId, downloadInstance) {
        var state = this.addBought(bookId, false);

        state.download = downloadInstance;
        return state;
      },

      delDownload: function(bookId) {
        bookId = bookId.toString();
 
        if (states[bookId]) {
          delete states[bookId].download;
        }

        return state;
      },

      clear: function() {
        states = {};
      }
    }
  }]);
angular.module('readerApp').factory('sidHttpInterceptor', [
   '$injector', function(
    $injector
  ) {
    var currentUserCache;

    // Avoid circular dependency between User service (User->$http->sidHttpInterceptor)
    // and sidHttpInterceptor (sidHttpInterceptor->User): inject User when it already
    // initialized
    var currentUser = function() {
      return currentUserCache || (currentUserCache = $injector.get('User').current());
    }

    var requestInterceptor = function(config) {
      var session_id = currentUser().session_id;

      if (config.sidHttpInterceptorDisable || session_id === undefined) {
        return config;
      }

      switch (config.method) {
        case 'GET':
          if (config.params === undefined) {
            config.params = {}
          }

          if (config.params.sid === undefined) {
            config.params.sid = currentUser().session_id;
          }

          break;

        case 'POST':
        case 'PUT':
          if (config.data === undefined) {
            config.data = {}
          }

          if (config.data.sid === undefined) {
            config.data.sid = currentUser().session_id;
          }

          break;          
      }

      return config;
    };

    return {
      request: requestInterceptor
    }
  }]);
angular.module('readerApp').factory('debugHttpInterceptor', [
   '$q', 'uuid', function(
    $q,   uuid
  ) {

    var logFilter = function(requestConfig) {
      // Skip requests and responses for app files (templates, etc.)
      return requestConfig.url.startsWith('http');
    }

    var requestInterceptor = function(config) {
      config.uniqueId = uuid();

      if (logFilter(config)) {
        console.log('http request [' + config.uniqueId + ']: ' + config.method + ' ' + config.url);
        console.log('http data:  ', config.data);
      }

      return config;
    }

    var requestErrorInterceptor = function(rejection) {
      console.log('http request error:', rejection);
      throw $q.reject(rejection);
    }

    var responseInterceptor = function(response) {
      if (logFilter(response.config)) {
        console.log('http response [' + response.config.uniqueId + ']: ' + response.status.toString() + ' ' + response.data);
      }

      return response;
    }

    var responseErrorItercetor = function(rejection) {
      console.log('http response error [' + rejection.config.uniqueId + ']: ', rejection);
      throw $q.reject(rejection);
    }

    return {
      request:       requestInterceptor,
      response:      responseInterceptor
    };
  }]);
angular.module('readerApp').factory('wifiHttpInterceptor', [
   '$injector', 'KeyVal', 'AppError', function(
    $injector,   KeyVal,   AppError
  ) {

    var requestInterceptor = function(config) {

      if (config && config.wifiHttpInterceptorDisable) {
        return config;
      }

      if (config && angular.isString(config.url)) {
        if (!config.url.startsWith("http")) {
          // It is local request (app://)
          return config;
        }
      }

      if (navigator.connection == undefined) {
        // Networking API is not supported, we cannot
        // check WiFi connection
        return config;
      }

      if (navigator.connection.type == "wifi") {
        // Current connection is WiFi, it's ok regardless
        // 'Only Wifi' option
        return config;
      }

      // Check current option state
      return KeyVal.get('settings_wifi_only', {reject: false}).then(function(value) {
        if (value) {
          // Options is enabled, interrupt request
          throw AppError.notWifiConnection();
        } else {
          // Option is disabled, continue request
          return config;
        }
      });
    }

    return {
      request: requestInterceptor
    }
  }]);
angular.module('readerApp').factory('Book', [
    '$http', '$q', '$sce', 'URLBuilder', 'APICache', 'User', 'AppError', 'BookFileDownloadRegistry', function(
     $http,   $q,   $sce,   URLBuilder,   APICache,   User,   AppError,   BookFileDownloadRegistry
  )
  {
    const DEFAULT_REQUEST_PARAMS = {
      app:      64,
      currency: 'RUB',
      lang:     'ru'
    };

    var fetchCollection = function(params) {
      var url = URLBuilder.build("/pages/catalit_browser/");

      var httpSuccess = function(response) {
        if (response.data["catalit-fb2-books"] !== undefined) {
          User.setBalance(response.data["catalit-fb2-books"]["_account"]);

          return response.data;
        } else if (response.data["catalit-authorization-failed"] !== undefined) {
          throw AppError.authorizationFailed();
        } else {
          throw AppError.unexpectedResponseData();
        }
      };

      var httpFailure = function(response) {
        if (AppError.is(response)) {
          throw response;
        } else if (response.status == 0) {
          throw AppError.serverUnavailable();
        } else if (response.status) {
          throw AppError.unexpectedResponseStatus(response.status.toString());
        } else {
          throw AppError.unknown();
        }
      };

      return $http.post(url, params).then(
        httpSuccess,
        httpFailure
      );
    }

    var fetchElement = function(token) {
      return fetchCollection({art: token}).then(function(collectionList) {
        return splitCollectionElemets(collectionList)();
      });
    }

    var transformBook = function(book) {
      var bookModel = {
        id:           book._hub_id,
        uuid:         book.text_description.hidden['document-info']['id'],
        title:        book.text_description.hidden['title-info']['book-title'],
        description:  $sce.trustAsHtml(( (new X2JS()).json2xml_str(book.text_description.hidden['title-info']['annotation']) )),
        price:        parseFloat(book._price),
        recenses:     book._recenses ? book._recenses : 0,
        rating:       parseInt(book._rating),
        basket:       book._basket,
        document_info: {
          date: book.text_description.hidden['document-info']['date']['__text']
        },

        cover: {
          full:    book._cover,
          preview: book._cover_preview
        },

        sequences: [],
        people:    [],

        //author: { }
      };

      if(book.text_description.hidden['title-info'].author) {
        bookModel.author = {
          first_name: book.text_description.hidden['title-info'].author["first-name"],
          last_name:  book.text_description.hidden['title-info'].author["last-name"],
          id:         book.text_description.hidden['title-info'].author.id
        }
      }

      bookModel.rating_voted = (
        (parseInt(book._voted1) || 0) +
        (parseInt(book._voted2) || 0) +
        (parseInt(book._voted3) || 0) +
        (parseInt(book._voted4) || 0) +
        (parseInt(book._voted5) || 0)
      )

      if (book.sequences) {
        var sequences;

        if (angular.isArray(book.sequences.sequence)) {
          sequences = book.sequences.sequence;
        } else {
          sequences = [book.sequences.sequence];
        }

        angular.forEach(sequences, function(sequence) {
          bookModel.sequences.push({
            id:    sequence._id,
            title: sequence._name
          });
        });
      }

      if (book.persons) {
        var people;

        if (angular.isArray(book.persons.persons)) {
          people = book.persons.persons;
        } else {
          people = [book.persons.persons];
        }

        angular.forEach(people, function(person) {
          bookModel.people.push({
            id:          person._id,

            first_name:  person._first,
            middle_name: person._middle,
            last_name:   person._last,
            role:        person._role
          });
        });
      }

      if (book._has_trial) {
        var urlBase = URLBuilder.build('/static/trials/');
        var urlBook = bookModel.id.toString();
        
        while (urlBook.length < 8) {
          urlBook = "0" + urlBook;
        }

        bookModel.trial_url = (
          urlBase + urlBook.substr(0, 2) + "/" + urlBook.substr(2, 2) + "/" +
                    urlBook.substr(4, 2) + "/" + urlBook + ".fb2.zip"
        );
      }

      if (book.text_description.hidden['publish-info']) {
        bookModel.publish_info = {
          isbn:      book.text_description.hidden['publish-info']['isbn'],
          publisher: book.text_description.hidden['publish-info']['publisher']
        };
      }
      
      var genre = book.text_description.hidden['title-info'].genre;
      
      if (angular.isArray(genre)) {
        bookModel.genres = genre;
      } else {
        bookModel.genres = [genre];
      }

      bookModel.genres = bookModel.genres.unique();

      bookModel.downloadRegTrial = BookFileDownloadRegistry.get(book._hub_id, true);
      bookModel.downloadRegFull  = BookFileDownloadRegistry.get(book._hub_id, false);
      
      return bookModel;
    };

    var splitCollectionElemets = function(collectionList) {
      var currentIndex = 0;

      collectionList = collectionList['catalit-fb2-books']['fb2-book'];

      if(collectionList === undefined) {
        collectionList = [];
      } else {
        if (!angular.isArray(collectionList)) {
          collectionList = [collectionList];
        }
      }

      return function() {
        if (currentIndex < collectionList.length) {
          currentElement = collectionList[currentIndex++];

          return {
            pk:    currentElement._hub_id,
            value: currentElement
          };
        }
      };
    }

    var cache = APICache.create('book', {
      expires:                30, // minutes
      fetchCollection:        fetchCollection,
      fetchElement:           fetchElement,
      splitCollectionElemets: splitCollectionElemets
    });

    return {
      collection: function(params, options) {
        var $promise,
            result,
            ret,
            i;

        params = angular.extend({}, DEFAULT_REQUEST_PARAMS, params);

        if(options == undefined) {
          options = {}
        }

        var fillRet = function(collection) {
          collection = collection['catalit-fb2-books']['fb2-book'] || [];

          if (!angular.isArray(collection)) {
            collection = [collection];
          }

          result = collection.map(transformBook);

          ret.splice(0, ret.length);

          for (i = 0; i < result.length; ++i) {
            ret.push(result[i]);
          }

          return result;
        }

        $promise = cache.collection(params, options).then(fillRet, null, fillRet);

        ret = [];
        ret.$promise = $promise;

        return ret;
      },

      element: function(token, options) {
        var $promise,
            result,
            prop,
            ret;

        var fillRet = function(element) {
          result = transformBook(element);

          for (prop in ret) {
            if (prop != '$promise') {
              delete ret[prop];
            }
          }

          for (prop in result) {
            ret[prop] = result[prop];
          }

          return ret;
        }

        $promise = cache.element(token, options).then(fillRet, null, fillRet);

        ret = {}
        ret.$promise = $promise;

        return ret;
      },

      bookmarks: function(additionalParams, options) {
        var self = this,
            ret  = [];

        ret.$promise = User.current().$promise.then(function(currentUser) {
          var collectionSuccess = function(collection) {
            angular.forEach(collection, function(book) {
              if (book.basket == "2" || book.basket == "1") {
                ret.push(book);
              }
            });

            return ret;
          };

          return self.collection(
            angular.extend({sid: currentUser.session_id, basket: true}, additionalParams),
            options
          ).$promise.then(
            collectionSuccess
          );
        });

        return ret;
      },

      my: function(additionalParams, options) {
        var self = this,
            ret  = [];

        ret.$promise = User.current().$promise.then(function(currentUser) {
          var collectionSuccess = function(collection) {
            angular.forEach(collection, function(book) {
              ret.push(book);
            });

            return ret;
          };

          var collectionFailure = function(error) {
            throw error;
          }

          return self.collection(
            angular.extend({sid: currentUser.session_id, my: true}, additionalParams),
            options
          ).$promise.then(
            collectionSuccess,
            collectionFailure
          );
        });

        return ret;
      }
    }

  }]);
angular.module('readerApp').factory('Person', [
    '$http', '$q', '$sce', 'URLBuilder', 'APICache', 'User', 'AppError', function(
     $http,   $q,   $sce,   URLBuilder,   APICache,   User,   AppError
  )
  {
    var fetchCollection = function(params) {
      var url     = URLBuilder.build("/pages/catalit_persons/");

      var httpSuccess = function(response) {
        return response.data;
      }

      var httpFailure = function(response) {
        if (AppError.is(response)) {
          throw response;
        } else if (response.status == 0) {
          throw AppError.serverUnavailable();
        } else if (response.status) {
          throw AppError.unexpectedResponseStatus(response.status.toString());
        } else {
          throw AppError.unknown();
        }
      }

      return $http.post(url, params).then(
        httpSuccess,
        httpFailure
      );
    }

    var fetchElement = function(personId) {
      var params = {},
          key;

      // personId can be litres internal id (hub_id, numbers only) or
      // global UUID
      if (personId.toString().search("-") >= 0) {
        key = 'person';
      } else {
        key = 'hub_id';
      }

      params[key] = personId;

      return fetchCollection(params).then(function(collectionList) {
        return splitCollectionElemets(collectionList)();
      });
    }

    var transformPerson = function(person) {
      var personModel = {
        id:          person._id,
        first_name:  person['first-name'],
        middle_name: person['middle-name'],
        last_name:   person['last-name'],
        photo:       person.photo
      }

      if (person.title) {
        personModel.full_name = person.title.main;
      } else {
        personModel.full_name = (person.first_name.toString() +
                                 person.middle_name.toString() + 
                                 person.last_name.toString());
      }

      if (person.text_descr_html) {
        personModel.bio = $sce.trustAsHtml(
          (new X2JS()).json2xml_str(person.text_descr_html.hidden)
        );
      };

      return personModel;
    };

    var splitCollectionElemets = function(collectionList) {
      var currentIndex = 0;

      collectionList = collectionList['catalit-persons']['subject'];

      if(collectionList === undefined) {
        collectionList = [];
      } else {
        if (!angular.isArray(collectionList)) {
          collectionList = [collectionList];
        }
      }

      return function() {
        if (currentIndex < collectionList.length) {
          currentElement = collectionList[currentIndex++];

          return {
            pk:    currentElement._id,
            value: currentElement
          };
        }
      };
    }

    var cache = APICache.create('person', {
      expires:                30, // minutes

      fetchCollection:        fetchCollection,
      fetchElement:           fetchElement,
      splitCollectionElemets: splitCollectionElemets
    });

    return {
      collection: function(params, options) {
        var $promise,
            result,
            ret,
            i;

        if(options == undefined) {
          options = {}
        }

        var fillRet = function(collection) {
          collection = collection['catalit-persons']['subject'] || [];

          if (!angular.isArray(collection)) {
            collection = [collection];
          }

          result = collection.map(transformPerson);

          ret.splice(0, ret.length);

          for (i = 0; i < result.length; ++i) {
            ret.push(result[i]);
          }

          return result;
        }

        $promise = cache.collection(params, options).then(fillRet, null, fillRet);

        ret = [];
        ret.$promise = $promise;

        return ret;
      },

      element: function(token, options) {
        var $promise,
            result,
            prop,
            ret;

        var fillRet = function(element) {
          result = transformPerson(element);

          for (prop in ret) {
            if (prop != '$promise') {
              delete ret[prop];
            }
          }

          for (prop in result) {
            ret[prop] = result[prop];
          }

          return result;
        }

        $promise = cache.element(token, options).then(fillRet, null, fillRet);

        ret = {}
        ret.$promise = $promise;

        return ret;
      }
    }

  }]);
angular.module('readerApp').factory('Basket', [
    '$http', '$q', 'URLBuilder', 'AppError', 'User', 'Book', function(
     $http,   $q,   URLBuilder,   AppError,   User,   Book
  )
  {
    return {
      request: function(hubId, type) {
        var currentUserSuccess = function(currentUser) {
          return $http.post("http://robot.litres.ru/pages/catalit_manage_basket/", {
            sid:    currentUser.session_id,
            hub_id: hubId,
            type:   type
          });
        }

        var httpSuccess = function(response) {
          if (response.data["catalit-basket-ok"] !== undefined) {
            // Reload bookmarks cache
            Book.bookmarks({}, {forceReload: true});

            return "OK";
          } else if (response.data["catalit-authorization-failed"] !== undefined) {
            throw AppError.authorizationFailed();
          } else if (response.data["catalit-basket-error"] !== undefined) {
            throw AppError.validation(response.data["catalit-basket-error"]["_error"]);
          } else {
            throw AppError.unexpectedResponseData();
          }
        };

        var httpFailure = function(response) {
          if (AppError.is(response)) {
            throw response;
          } else if (response.status == 0) {
            throw AppError.serverUnavailable();
          } else if (response.status) {
            throw AppError.unexpectedResponseStatus(response.status.toString());
          } else {
            throw AppError.unknown();
          }
        };

        return User.current().$promise.then(
          currentUserSuccess
        ).then(
          httpSuccess,
          httpFailure
        );
      },

      putToBasket: function(hubId) {
        return this.request(hubId, 1);
      },

      putToBookmarks: function(hubId) {
        return this.request(hubId, 2);
      },

      remove: function(hubId) {
        return this.request(hubId, 0);
      }
    }
  }]);
angular.module('readerApp').factory('Genre', [
    '$http', '$q', 'URLBuilder', 'APICache', 'AppError', function(
     $http,   $q,   URLBuilder,   APICache,   AppError
  )
  {
    var fetchCollection = function(params) {
      var url     = URLBuilder.build("/pages/catalit_genres/");

      var httpSuccess = function(response) {
        return response.data;
      }

      var httpFailure = function(response) {
        if (AppError.is(response)) {
          throw response;
        } else if (response.status == 0) {
          throw AppError.serverUnavailable();
        } else if (response.status) {
          throw AppError.unexpectedResponseStatus(response.status.toString());
        } else {
          throw AppError.unknown();
        }
      }

      return $http.post(url, params).then(
        httpSuccess,
        httpFailure
      );
    }

    var fetchElement = function(token) {
      return fetchCollection().then(function(tree) {
        var next = splitCollectionElemets(tree),
            element;

        for (element = next(); element; element = next()) {
          if (element.pk == token.toString()) {
            return element;
          }
        }
      });
    }

    var transformGenre = function(genre) {
      return {
        id:       genre._id,
        title:    genre._title,
        token:    genre._token,
        children: (genre.genre || []).map(transformGenre)
      };
    };

    var splitCollectionElemets = function(collectionTree) {
      var collectionList = [],
          currentIndex;

      var pushElement = function(element) {
        var pushKeys = ["_token", "_id", "_title"],
            key,
            i;

        for (i = 0; i < pushKeys.length; ++i) {
          key = pushKeys[i];

          if (element[key]) {
            var elementToPush = {},
                prop;

            for (prop in element) {
              elementToPush[prop] = element[prop];
              //if (prop != "genre") {
              //}
            }

            collectionList.push({
              pk:    elementToPush[key],
              value: elementToPush
            });
          }
        }
      }

      var treeWalk = function(level) {
        if (angular.isArray(level)) {
          for (var i = 0; i < level.length; ++i) {
            var element = level[i];

            pushElement(element);

            if (element.genre) {
              treeWalk(element.genre);
            }
          }
        } else if (angular.isObject(element.genre)) {
          pushElement(level);
        } else {
          throw new TypeError("genre tree level should be array or object");
        }
      }

      treeWalk(collectionTree['catalit-genres'].genre);

      currentIndex = 0;

      return function() {
        if (currentIndex < collectionList.length) {
          return collectionList[currentIndex++];
        }
      };
    }

    var cache = APICache.create('genre', {
      expires: 60 * 24 * 7, // week

      fetchCollection:        fetchCollection,
      fetchElement:           fetchElement,
      splitCollectionElemets: splitCollectionElemets
    });

    return {
      collection: function(params, options) {
        return cache.collection(params, options).then(function(collection) {
          return collection['catalit-genres'].genre.map(transformGenre);
        });
      },

      element: function(token, options) {
        var $promise,
          result,
          prop,
          ret;

        var fillRet = function(element) {
          result = transformGenre(element);

          for (prop in ret) {
            if (prop != '$promise') {
              delete ret[prop];
            }
          }

          for (prop in result) {
            ret[prop] = result[prop];
          }

          return result;
        }

        $promise = cache.element(token, options).then(fillRet, null, fillRet);

        ret = {}
        ret.$promise = $promise;

        return ret;
      }
    }
  }]);
angular.module('readerApp').factory('Comment', [
    '$http', '$q', '$sce', 'URLBuilder', 'APICache', 'AppError', 'User', function(
     $http,   $q,   $sce,   URLBuilder,   APICache,   AppError,   User
  )
  {
    var fetchCollection = function(params) {
      var url = URLBuilder.build("/pages/catalit_get_recenses/");

      var httpSuccess = function(response) {
        return response.data;
      }

      var httpFailure = function(response) {
        if (AppError.is(response)) {
          throw response;
        } else if (response.status == 0) {
          throw AppError.serverUnavailable();
        } else if (response.status) {
          throw AppError.unexpectedResponseStatus(response.status.toString());
        } else {
          throw AppError.unknown();
        }
      }

      return $http.post(url, params).then(
        httpSuccess,
        httpFailure
      );
    }

    var transformComment = function(comment) {
      return {
        id:         comment._id,
        created_at: Date.parse(comment._added.replace(/(\d+)-(\d+)-(\d+)/, '$2/$3/$1')),
        author: {
          id:     comment._user,
          login:  comment._login
        },

        title: comment._caption,
        body:  $sce.trustAsHtml(
          (new X2JS()).json2xml_str(comment.text_recense.hidden)
        )
      }
    };

    var cache = APICache.create('comment', {
      expires:         30, // minutes
      fetchCollection: fetchCollection,
    });

    return {
      collection: function(params, options) {
        var $promise,
            result,
            ret,
            i;

        var fillRet = function(collection) {
          collection = collection['catalit-recenses']['book-recenses'].recense || [];

          if (!angular.isArray(collection)) {
            collection = [collection];
          }

          result = collection.map(transformComment);

          ret.splice(0, ret.length);

          for (i = 0; i < result.length; ++i) {
            ret.push(result[i]);
          }

          return result;
        };

        $promise = cache.collection(params, options).then(fillRet, null, fillRet);

        ret = [];
        ret.$promise = $promise;

        return ret;
      },

      create: function(params) {
        var currentUserSuccess = function(currentUser) {
          var httpParams = angular.extend({
            sid:  currentUser.session_id,
            mail: currentUser.email
          }, params);

          return $http.post(URLBuilder.build("/pages/catalit_add_recense/"), httpParams);
        }

        var httpSuccess = function(response) {
          if (response.data["catalit-add-recenses-ok"] !== undefined) {
            if (params.rating) {
              var ratingParams = {
                art:  params.art,
                sid:  currentUser.session_id,
                mark: params.rating
              };

              delete params.rating;

              $http.post(URLBuilder.build("/pages/catalit_vote/"), ratingParams).then(
                function(result) { console.log('vote success: ', result) },
                function(result) { console.log('vote failure: ', result) }
              );
            }

            // TODO: drop comments APICache
            return "OK";
          } else if (response.data["catalit-authorization-failed"] !== undefined) {
            throw AppError.authorizationFailed();
          } else if (response.data["catalit-add-recenses-fail"] !== undefined) {
            throw AppError.validation(response.data["catalit-add-recenses-fail"]["_error"]);
          } else {
            throw AppError.unexpectedResponseData();
          }
        };

        var httpFailure = function(response) {
          if (AppError.is(response)) {
            throw response;
          } else if (response.status == 0) {
            throw AppError.serverUnavailable();
          } else if (response.status) {
            throw AppError.unexpectedResponseStatus(response.status.toString());
          } else {
            throw AppError.unknown();
          }
        };

        return User.current().$promise.then(
          currentUserSuccess
        ).then(
          httpSuccess,
          httpFailure
        );
      }
    }
  }]);
angular.module('readerApp').factory('User', [
    '$http', '$q', 'URLBuilder', 'KeyVal', 'AppError', 'deviceId', '$injector', 'Analytics', function(
     $http,   $q,   URLBuilder,   KeyVal,   AppError,   deviceId,   $injector,   Analytics
  ) {
    const DATABASE_KEY = 'current_user';
    const EMAIL_REGEXP = /^[-a-z0-9~!$%^&*_=+}{\'?]+(\.[-a-z0-9~!$%^&*_=+}{\'?]+)*@([a-z0-9_][-a-z0-9_]*(\.[-a-z0-9_]+)*\.(aero|arpa|biz|com|coop|edu|gov|info|int|mil|museum|name|net|org|pro|travel|mobi|[a-z][a-z])|([0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}))(:[0-9]{1,5})?$/i;

    var transformUser = function(user) {
      var keys = ['catalit-authorization-ok', 'catalit-unite-user-ok'],
          key,
          i;

      for(i = 0; i < keys.length; ++i) {
        if (user[keys[i]]) {
          key = keys[i];
          break;
        }
      }

      if (!key) {
        throw AppError.unexpectedResponseData();
      }

      return {
        session_id:   user[key]['_sid'],
        first_name:   user[key]['_first-name'],
        middle_name:  user[key]['_middle-name'],
        last_name:    user[key]['_last-name'],
        login:        user[key]['_login'],
        email:        user[key]['_mail'],
        phone:        user[key]['_phone'],
        www:          user[key]['_www'],
        city:         user[key]['_city'],
        can_rebill:   user[key]['_can-rebill'],
        male:         user[key]['_male'],
        now:          user[key]['_now'],
        account:      user[key]['_account'],
        account_full: user[key]['_account_full'],
        bonus:        user[key]['_bonus'],
        user_id:      user[key]['_user-id'],
        books_cnt:    user[key]['_books-cnt'],
        authors_cnt:  user[key]['_authors-cnt'],
        biblio_user:  user[key]['_biblio_user'],
        birth_day:    user[key]['_birth_day'],

        temporary:    user.temporary || false,
        now:          user.now       || new Date()
      }
    }

    var detransformUser = function(user) {
      return {
        "catalit-authorization-ok": {
          "_sid":          user.session_id,
          "_first-name":   user.first_name,
          "_middle-name":  user.middle_name,
          "_last-name":    user.last_name,
          "_login":        user.login,
          "_mail":         user.email,
          "_phone":        user.phone,
          "_www":          user.www,
          "_city":         user.city,
          "_can-rebill":   user.can_rebill,
          "_male":         user.male,
          "_now":          user.now,
          "_account":      user.account,
          "_account_full": user.account_full,
          "_bonus":        user.bonus,
          "_user-id":      user.user_id,
          "_books-cnt":    user.books_cnt,
          "_authors-cnt":  user.authors_cnt,
          "_biblio_user":  user.biblio_user,
          "_birth_day":    user.birth_day
        },

        "temporary": user.temporary,
        "now":       user.now
      }
    }

    var transformSignUpError = function(errorCode, comment) {
      switch (errorCode.toString()) {
        case '1':
          // Ошибка сервера: такой логин уже занят
          message = 'Пользователь с таким email уже существует. Если это ваш email, то воспользуйтесь восстановлением пароля';
          break;
        case '2':
          // Ошибка сервера: пустой логин
          message = 'пустой email';
          break;
        case '3':
          message = 'пустой пароль';
          break;
        case '4':
          message = 'некорректный email';
          break;
        case '5':
          message = 'временно превышен лимит регистрации с данного IP';
          break;
        case '6':
          // Ошибка сервера: такой e-mail уже занят
          message = 'Пользователь с таким email уже существует. Если это ваш email, то воспользуйтесь восстановлением пароля';
          break;
        case '7':
          message = 'повтор пароля не совпадает с паролем';
          break;
        case '8':
          message = 'некорректный номер телефона';
          break;
        case '9':
          message = 'такой номер телефона уже занят';
          break;
        case '100':
        default:
          message = 'неизвестная ошибка: ' + comment;
          break;
      }

      return new AppError.validation(message);
    }

    var transformRecoverPasswordError = function(response) {
      switch (response["catalit-pass-recover-failed"]._error.toString()) {
        case '1':
          message = 'Email не зарегистрирован';
          break;
        case '2':
          message = 'Email не указан';
          break;
        case '100':
        default:
          message = 'неизвестная ошибка: ' + response["catalit-registration-failed"]._comment.toString()
          break;
      }

      return new AppError.validation(message);
    }

    var currentUser    = window.currentUser = {};

    var setupCurrentUser = function(newCurrentUser) {
      var defered = $q.defer(),
          prop;

      clearCurrentUser()

      for (prop in newCurrentUser) {
        currentUser[prop] = newCurrentUser[prop];
      }

      currentUser.authenticated = true;
      currentUser.$promise = defered.promise;

      Analytics.set('&uid', currentUser.user_id);

      setBalance(currentUser.account, newCurrentUser.now).then(function() {
        defered.resolve(currentUser);
      });

      return newCurrentUser;
    }

    var clearCurrentUser = function(error) {
      var defered = $q.defer(),
          prop;

      for (prop in currentUser) {
        delete currentUser[prop];
      }


      currentUser.authenticated = false;
      currentUser.error         = error;

      currentUser.$promise = defered.promise;
      Analytics.set('&uid', null);
      defered.reject(error);
    }

    clearCurrentUser();

    var setSession = function(session, dontDropMyBooks) {
      // Drop MyBook cache only on sign up and sign in
      if (!dontDropMyBooks) {
        clearMyBooks();
      }

      return KeyVal.set(DATABASE_KEY, session);
    };

    var delSession = function() {
      // Drop MyBook cache on sign out
      clearMyBooks();

      return KeyVal.del(DATABASE_KEY);
    };

    var uniteUsers = function(newCurrentUser) {
      newCurrentUser.$unitePromise = deviceId().then(function(deviceId) {
        var url = URLBuilder.build('/pages/catalit_unite_user/'),
            defered = $q.defer();

        if (newCurrentUser.login == deviceId) {
          defered.resolve(newCurrentUser);
          return defered.promise;
        }

        $http.post(url, {sid: newCurrentUser.session_id, user_login: deviceId, user_passwd: deviceId}, {sidHttpInterceptorDisable: true, wifiHttpInterceptorDisable: true})
          .success(function(result, status, headers, config) {
            if (result["catalit-unite-user-ok"]) {
              defered.resolve(result);
            } else if (result["catalit-unite-user-failed"]) {
              defered.reject(AppError.authorizationFailed());
            } else {
              defered.reject(AppError.unexpectedResponseData());
            }            
          })
          .error(function(result, status, headers, config) {
            if (status == 0) {
              defered.reject(AppError.serverUnavailable());
            } else {
              defered.reject(AppError.unexpectedResponseStatus(status.toString()));
            }            
          });

        return defered.promise
          .then(setSession)
          .then(transformUser)
          .then(setupCurrentUser);
      });

      return newCurrentUser;
    }

    var clearMyBooks = function() {
      // Avoid circular dependency between MyBook service (MyBook->Book->User)
      // and User (User->MyBook): inject MyBook when it already initialized
      return $injector.get('MyBook').clear();
    }

    var getBalance = function() {
      return KeyVal.get('balance').then(function(cachedValue) {
        if (cachedValue && cachedValue.session_id == currentUser.session_id) {
          return cachedValue.balance;
        };

        return undefined;
      });
    }

    var setBalance = function(newValue, actualAt) {

      if (!newValue) {
        return;
      }

      if (actualAt) {
        return KeyVal.get('balance').then(function(cachedValue) {
          if (!cachedValue || actualAt > cachedValue.actualAt) {
            return KeyVal.set('balance', {
              session_id: currentUser.session_id,
              balance:    newValue,
              actualAt:   actualAt
            });
          }
        });
      } else {
        return KeyVal.set('balance', {
          session_id: currentUser.session_id,
          balance:    newValue,
          actualAt:   new Date()
        });
      }
    }

    return {
      signOut: function() {
        var self = this;

        var autoSignIn = function() {
          return self.autoSignIn();
        }

        currentUser.$promise = delSession()
          .then(autoSignIn);

        return currentUser.$promise;
      },

      signIn: function(login, password, isTemporary) {
        var url     = URLBuilder.build('/pages/catalit_authorise/'),
            defered = $q.defer(),
            self    = this;

        isTemporary = isTemporary || false;
        
        $http.post(url, {login: login, pwd: password, skip_ip: true}, {sidHttpInterceptorDisable: true, wifiHttpInterceptorDisable: true})
          .success(function(result, status, headers, config) {
            if (result["catalit-authorization-ok"]) {
              result.temporary = isTemporary;
              defered.resolve(result);
            } else if (result["catalit-authorization-failed"] !== undefined) {
              defered.reject(AppError.validation('Не верный email или пароль'));
            } else {
              defered.reject(AppError.unexpectedResponseData());
            }
          })
          .error(function(result, status, headers, config) {
            if (status == 0) {
              defered.reject(AppError.serverUnavailable());
            } else {
              defered.reject(AppError.unexpectedResponseStatus(status.toString()));
            }
          });

        return defered.promise
          .then(setSession)
          .then(transformUser)
          .then(setupCurrentUser)
          .then(uniteUsers);
      },

      signUp: function(login, password, params, isTemporary) {
        var signUpParams,
            url     = URLBuilder.build('/pages/catalit_register_user/'),
            defered = $q.defer();

        login = (login || "").toString();

        if (!login.match(EMAIL_REGEXP) && !isTemporary) {
          var message = 'некорректный email';
          defered.reject(AppError.validation(message));
          return defered.promise;
        }

        signUpParams = angular.extend({new_login: login, new_pwd1: password}, params || {});
        isTemporary  = isTemporary || false;



        $http.post(url, signUpParams, {sidHttpInterceptorDisable: true, wifiHttpInterceptorDisable: true})
          .success(function(result, status, headers, config) {
            if (result["catalit-authorization-ok"]) {
              result.temporary = isTemporary;
              defered.resolve(result);
            } else if (result["catalit-registration-failed"]) {
              defered.reject(transformSignUpError(
                result["catalit-registration-failed"]._error,
                result["catalit-registration-failed"]._comment
              ));
            } else {
              defered.reject(AppError.unexpectedResponseData());
            }
          })
          .error(function(result, status, headers, config) {
            if (status == 0) {
              defered.reject(AppError.serverUnavailable());
            } else {
              defered.reject(AppError.unexpectedResponseStatus(status.toString()));
            }
          });

        return defered.promise
          .then(setSession)
          .then(transformUser)
          .then(setupCurrentUser)
          .then(uniteUsers);
      },

      autoSignIn: function() {
        var self = this;

        return deviceId().then(function(deviceId) {
          return self.signIn(deviceId, deviceId, true).catch(function() {
            return self.signUp(deviceId, deviceId, {}, true).catch(function(error) {
              clearCurrentUser(error);
              throw error;
            });
          });
        });
      },

      current: function() {
        var self = this;

        if (!currentUser.session_id && currentUser.$promise.$$state.status != 0) {
          var getFailure = function() {
            return self.autoSignIn();
          }

          currentUser.$promise = KeyVal.get(DATABASE_KEY)
            .catch(getFailure)
            .then(transformUser)
            .then(setupCurrentUser);
        }

        return currentUser;
      },

      passwordRecover: function(email) {
        var url = URLBuilder.build('/pages/catalit_recover_pass/');

        var success = function(response) {
          if (response.data["catalit-pass-recover-failed"]) {
            throw transformRecoverPasswordError(response.data);
          } else {
            return 'OK';
          }
        }

        var failure = function(response) {
          if (AppError.is(response)) {
            throw response;
          } else if (response.status == 0) {
            throw AppError.serverUnavailable();
          } else if (response.status) {
            throw AppError.unexpectedResponseStatus(response.status.toString());
          } else {
            throw AppError.unknown();
          }
        }

        return $http.post(url, {mail: email}, {sidHttpInterceptorDisable: true, wifiHttpInterceptorDisable: true}).then(success, failure);
      },

      update: function(params) {
        var self = this;

        return self.current().$promise.then(function(currentUser) {
          var url        = URLBuilder.build('/pages/catalit_update_user/'),
              httpParams = angular.extend({}, params, {sid: currentUser.session_id});

          var success = function(response) {
            if (response.data["catalit-updateuser-ok"] !== undefined) {
              var prop;

              for (prop in params) {
                currentUser[prop] = params[prop];
              }

              return setSession(detransformUser(currentUser), true);
            } else if (response.data["catalit-updateuser-failed"]) {
              throw transformSignUpError(
                response.data["catalit-updateuser-failed"]._error,
                response.data["catalit-updateuser-failed"]._comment
              )
            } else {
              throw AppError.unexpectedResponseData();
            }
          }

          var failure = function(response) {
            if (AppError.is(response)) {
              throw response;
            } else if (response.status == 0) {
              throw AppError.serverUnavailable();
            } else if (response.status) {
              throw AppError.unexpectedResponseStatus(response.status.toString());
            } else {
              throw AppError.unknown();
            }
          }

          return $http.post(url, httpParams, {sidHttpInterceptorDisable: true, wifiHttpInterceptorDisable: true}).then(
            success,
            failure
          );
        });
      },

      getBalance: getBalance,
      setBalance: setBalance
    }
  }]);
angular.module('readerApp').factory('Folder', [
    '$http', '$q', 'URLBuilder', 'User', 'AppError', function(
     $http,   $q,   URLBuilder,   User,   AppError
  )
  {
    var transformPutError = function(serverError) {
      switch (serverError) {
        case "bad params":
          // переданы не все обязательные параметры
          return "не удалось переместить книгу на полку";
        case "bad folder":
          return "полка не найдена";
        case "not in the basket":
          return "книга не найдена";
        default:
          return "неизвестная ошибка (" + serverError.toString() + ")";
      }
    }

    var put = function(bookId, folderId) {
      var url = URLBuilder.build("/pages/catalit_put_book_to_folder/");

      var httpSuccess = function(response) {
        if (response.data["catalit-put-book-to-folder-ok"] !== undefined) {
          return "OK";
        } else if (response.data["catalit-put-book-to-folder-failed"] !== undefined) {
          throw AppError.application(
            transformPutError(response.data["catalit-put-book-to-folder-failed"]._comment)
          );
        } else {
          throw AppError.unexpectedResponseData();
        }
      };

      var httpFailure = function(response) {
        if (AppError.is(response)) {
          throw response;
        } else if (response.status == 0) {
          throw AppError.serverUnavailable();
        } else if (response.status) {
          throw AppError.unexpectedResponseStatus(response.status.toString());
        } else {
          throw AppError.unknown();
        }
      };

      var params = {
        art:    bookId,
        folder: folderId
      }

      return $http.post(url, params).then(
        httpSuccess,
        httpFailure
      );
    };

    return {
      put: put
    };
  }]);
angular.module('readerApp').factory('Search', [
    '$http', '$q', 'URLBuilder', 'KeyVal', 'AppError', 'deviceId', '$injector', 'Book', function(
     $http,   $q,   URLBuilder,   KeyVal,   AppError,   deviceId,   $injector,   Book
  ) {
    const DATABASE_KEY = 'current_user';

    parsePersons = function(data) {
      return data.map(function(element) {
        return {
          id: element.id,
          type: 'author',
          full_name: element.name
        }
      })
    };

    parseSeries = function(data) {
      return data.map(function(element) {
        return {
          type: 'series',
          id: element.id,
          name: element.name,
          author_name: element.bb_author
        }
      })
    };

    var prevQuery = null;

    return function(term) {
      var url     = URLBuilder.build("/pages/search_rmd2/", { json: 1, q: term, limit: 30 }),
          defered = $q.defer();


      console.log('SEARCH: ', url)

      $http.get(url)
        .success(function(response, status, headers, config) {
          var result = {};

          if(response.arts) {
            var book_ids = response.arts.map(function(element) { return element.id });
            console.log({arts: book_ids})
            var booksPromise = Book.collection({art: book_ids}).$promise;

            booksPromise.then(function(books) {
              console.log('books: ', books)
              result.books = books;

              if(!result.best && (result.books && result.books.length)) {
                result.best = angular.extend(result.books.shift(), {type: 'book'})
              }

              defered.resolve(result);
            })
          }

          if(response.authors) {
            result.persons = parsePersons(response.authors)
          }

          if(response.series) {
            result.series = parseSeries(response.series)
          }

          if((result.persons && result.persons.length)) {
            result.best = angular.extend(result.persons.shift(), {type: 'author'})
          }

          if(!response.arts) {
            defered.resolve(result);
          }
        })
        .error(function(response, status, headers, config) {
          defered.reject(response, status, headers, config);
        });

      if(prevQuery) {
        prevQuery.reject(null);
        prevQuery = defered
      }
      return defered.promise;

    }
  }]);
angular.module('readerApp')
  .service('PaymentsTrack', [
   '$http', '$q', '$injector', '$interval', '$indexedDB', 'URLBuilder', 'ngNotify', 'BookFileDownload', 'KeyVal', 'MyBook', 'MyBookStates', 'Analytics', function(
    $http,   $q,   $injector,   $interval,   $indexedDB,   URLBuilder,   ngNotify,   BookFileDownload,   KeyVal,   MyBook,   MyBookStates,   Analytics
  )
  {
    const STORE_NAME = 'payments';

    var   timer     = null,
          interval  = 5000;

    var setCheckInterval = function(_interval) {
      interval = _interval
    };

    var checkOrder = function(orderNumber) {
      var url     = URLBuilder.build("/pages/catalit_payorder_check_state/"),
          defered = $q.defer(),
          params  = {};

      params.order_id = orderNumber;

      $http.post(url, params)
        .success(function(response, status, headers, config) {
          defered.resolve(response);
        })
        .error(function(response, status, headers, config) {
          defered.reject(response, status, headers, config);
        });

      return defered.promise;
    };

    var analyticsDoneTransaction = function(trackItem) {
      var analyticsOptions = trackItem.analyticsOptions;
      
      Analytics.addTrans(
        trackItem.orderNumber,    // transaction ID
        '',                       // affiliation
        analyticsOptions.price,   // total price
        '0',                      // tax
        '0',                      // shipping
        '',                       // city
        '',                       // state
        'Russian Federation',     // country
        analyticsOptions.currency // currency
      );

      Analytics.addItem(
        trackItem.orderNumber,     // transaction ID
        analyticsOptions.bookId,   // SKU
        analyticsOptions.bookName, // name
        '',                        // category
        analyticsOptions.price,    // price
        '1'                        // quantity
      );

      // Complete transaction
      Analytics.trackTrans();
    };

    var start = function() {
      timer = $interval(function() {
        tackList().then(function(data) {
          angular.forEach(data, function(item) {
            checkOrder(item.orderNumber).then(function(response) {
              console.log('checkOrder response: ', response);
              switch(response['catalit-payorder-processing-check']['_state']) {
                case 'failed':
                  //show error message
                  ngNotify.set('При обработке платежа произошли проблемы', 'error');
                  //delete record
                  untrackOrder(item.orderNumber);
                  MyBookStates.delBuyInProcess(item.bookId);
                  break;
                case 'pending':
                  MyBookStates.addBuyInProcess(item.bookId, item.orderNumber);
                  break;
                case 'unknown':
                  //delete record
                  untrackOrder(item.orderNumber);
                  MyBookStates.delBuyInProcess(item.bookId);
                  break;
                case 'closed_ok':
                  $injector.get('Payments').purchaseBookDeafult(item.bookId);
                  analyticsDoneTransaction(item);
                  untrackOrder(item.orderNumber);
                  break;
              }

            }, function() {

            })
          })
        })
      }, interval)
    };

    var pause = function() {
      if(timer) {
        $interval.cancel(timer)
      }
    };

    var trackOrder = function (orderNumber, bookId, callbackName, analyticsOptions) {
      $indexedDB.openStore(STORE_NAME, function(store) {
        store.upsert({orderNumber: orderNumber, bookId: bookId, callbackName: callbackName, analyticsOptions: analyticsOptions});
      }, 'readwrite')
    };

    var untrackOrder = function (orderNumber) {
      $indexedDB.openStore(STORE_NAME, function(store) {
        store.delete(orderNumber);
      }, 'readwrite')
    };

    var tackList = function () {
      var defered = $q.defer()

      $indexedDB.openStore(STORE_NAME, function(store) {
        store.getAll().then(function(data) {
          defered.resolve(data);
        })
      }, 'readwrite')

      return defered.promise;
    };

    return {
      trackOrder: trackOrder,
      setCheckInterval: setCheckInterval,
      tackList: tackList,
      start: start,
      pause: pause
    };
  }])
  .factory('Payments', [
    '$http', '$q', '$injector', 'URLBuilder', 'APICache', 'PaymentsTrack', 'BookFileDownload', 'MyBook', 'ngNotify', 'KeyVal', 'MyBookStates', 'Analytics', function(
     $http,   $q,   $injector,   URLBuilder,   APICache,   PaymentsTrack,   BookFileDownload,   MyBook,   ngNotify,   KeyVal,   MyBookStates,   Analytics
  )
  {
    var currentUserCache;
    var currentUser = function() {
      return currentUserCache || (currentUserCache = $injector.get('User').current());
    };

    var creditCardInit = function(sum) {
      var url     = URLBuilder.build("/pages/catalit_credit_card_init/", true),
          defered = $q.defer(),
          params  = {};

      currentUser().$promise.then(function(asd) {
        params.sid = currentUser().session_id;
        params.sum = sum;

        $http.post(url, params)
          .success(function(response, status, headers, config) {
            if(response['catalit-paycard-processing']) {
              inner = response['catalit-paycard-processing']
              console.log(inner)
              var resp = {
                termUrl:  inner['TermUrl'],
                homepage: inner['homepage'],
                method:   inner['method'],
                name:     inner['name'],
                orderId:  inner['order-id'],
                url:      inner['url'],
                params:   inner['param'].map(function(param) {

                  return {
                    key:    param._substitute,
                    name:   param._name,
                    value:  param.__text
                  }
                })
              };

              defered.resolve(resp);
            } else {
              defered.reject(response, status, headers, config);
            }
          })
          .error(function(response, status, headers, config) {
            defered.reject(response, status, headers, config);
          });
      });

      return defered.promise;
    };

    var creditCardProcess = function(url, params, bookId, analyticsOptions) {
      var defered = $q.defer();

      currentUser().$promise.then(function(asd) {
        params.sid = currentUser().session_id;

        $http.post(url, params)
          .success(function(response, status, headers, config) {
            parsed_response = response.split("&").map(function(e) { return e.split('=') });
            response_object = {};

            angular.forEach(parsed_response, function(el) {
              response_object[el[0]] = el[1]
            });

            if(response_object.Code == '200' && response_object.Result == 'Ok') {
              PaymentsTrack.trackOrder(params['OrderID'], bookId, "mCreditCardComplete", analyticsOptions);
              MyBookStates.addBuyInProcess(bookId, params['OrderID']);
              defered.resolve(response_object);
            } else {

              // Test 2011 codes
              if(/2011/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 2100, 2101, 2110 codes
              if(/2100|2101|2110/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 2XXX codes
              if(/2[0-9]{3}/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 3XXX codes
              if(/3[0-9]{3}/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 4014 codes
              if(/4014/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'Срок действия карты истек.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 4XXX codes
              if(/4[0-9]{3}/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'Проверьте введенные данные и попробуйте еще раз.',
                  'Также Вы можете оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5205 codes
              if(/5205/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'На карте недостаточно средств.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5204, 5308 codes
              if(/5204|5308/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5301 codes
              if(/5301/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'Срок действия карты истек.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5310 codes
              if(/5310/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5411 codes
              if(/5411/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'Неверно введен CVV2/CVC2.',
                  'Попробуйте оплатить еще раз или воспользуйтесь любым другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 5XXX codes
              if(/5[0-9]{3}/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 6002, 6004 codes
              if(/6002|6004/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить еще раз или воспользуйтесь другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              // Test 6003 codes
              if(/6003/.test(response_object.Code)) {
                response_object.errorDesc = [
                  'При обработке платежа возникла ошибка.',
                  'Попробуйте оплатить еще раз или воспользуйтесь другим удобным способом.'
                ];
                return defered.reject(response_object, status, headers, config);
              }

              defered.reject(response_object, status, headers, config);
            }
          })
          .error(function(response, status, headers, config) {
            defered.reject(response, status, headers, config);
          });
      });

      return defered.promise;
    };

    var mobileCommerceInit = function(phone, sum, bookId, analyticsOptions) {
      if (sum && Number(sum) < 10) {
        sum = 10;
      }

      var url     = URLBuilder.build("/pages/catalit_mcommerce_init/"),
        defered = $q.defer(),
        params  = {};

      currentUser().$promise.then(function(user) {
        params.sid    = user.session_id;
        params.sum    = sum;
        params.phone  = phone;

        $http.post(url, params)
          .success(function(response, status, headers, config) {
            //_order_id:"876273709"
            //_price:"100"
            //_state:"success"
            console.log('mobileCommerceInit response: ', response);

            if(response['catalit-mcommerce-init']['_state'] == 'success') {
              PaymentsTrack.trackOrder(response['catalit-mcommerce-init']['_order_id'], bookId, "mCommerceComplete", analyticsOptions);
              MyBookStates.addBuyInProcess(bookId, response['catalit-mcommerce-init']['_order_id']);
              defered.resolve(response);
            } else {
              defered.reject(response, status, headers, config);
            }
          })
          .error(function(response, status, headers, config) {
            defered.reject(response, status, headers, config);
          });
      });

      return defered.promise;
    };

    var getUserAmount = function () {
      var url     = URLBuilder.build("/pages/purchase_book/"),
        defered = $q.defer(),
        params  = {};

      currentUser().$promise.then(function(asd) {
        params.sid    = currentUser().session_id;

        $http.post(url, params)
          .success(function(response, status, headers, config) {
            $injector.get('User').setBalance(['_account']);
            defered.resolve(parseFloat(response['catalit-purchase-ok']['_account']));
          })
          .error(function(response, status, headers, config) {
            defered.reject(response, status, headers, config);
          });
      });

      return defered.promise;
    };

    var purchaseBook = function (bookId) {
      var url     = URLBuilder.build("/pages/purchase_book/"),
          defered = $q.defer(),
          params  = {};

      currentUser().$promise.then(function(asd) {
        params.sid    = currentUser().session_id;
        params.art    = bookId;

        $http.post(url, params)
          .success(function(response, status, headers, config) {
            if (response['catalit-purchase-failed']) {
              defered.reject(response['catalit-purchase-failed'], status, headers, config);
            } else {
              if (response && response['catalit-purchase-ok']) {
                $injector.get('User').setBalance(response['catalit-purchase-ok']['_account']);
              };

              defered.resolve(response);
            }
          })
          .error(function(response, status, headers, config) {
            defered.reject(response, status, headers, config);
          });
      });

      return defered.promise;
    };

    var purchaseBookDeafult = function (bookId) {
      return purchaseBook(bookId).then(function(response) {
        ngNotify.set('Покупка успешно совершена.', {
          duration: 10000,
          type: 'success'
        });

        MyBook.push(bookId).then(function() {        
          KeyVal.get('settings_download_after_purchase', {reject: false}).then(function(value) {
            if (value) {
              BookFileDownload.downloadFull(bookId);
            }
          });
        });
      }, function(response) {
        if(response._error == "1") {
          ngNotify.set('На вашем счете недостаточно средств.', {
            duration: 10000,
            type: 'error'
          });
        } else {
          ngNotify.set('Произошла неизвестная ошибка, пожалуйста попробуйте позже.', {
            duration: 10000,
            type: 'error'
          });
        }

        throw response._error;
      });
    };

    return {
      creditCardInit: creditCardInit,
      mobileCommerceInit: mobileCommerceInit,
      getUserAmount: getUserAmount,
      purchaseBook: purchaseBook,
      purchaseBookDeafult: purchaseBookDeafult,
      creditCardProcess: creditCardProcess
    }
  }]);
angular.module('readerApp').directive('busy', [
   '$q', function(
    $q
  )
  {
    return {
      link: function($scope, element, attrs, ctrl) {
        // Ladda style
        element.addClass('ladda-button');
        element.attr('data-style', 'expand-left');

        var ladda = Ladda.create(element[0]);

        var disable = function() { ladda.start() };
        var enable  = function() { ladda.stop()  };

        return $scope.$watchCollection(attrs.busy, function(newVal) {
          var promise;
          
          if (!newVal) {
            enable();
            return;
          }
          
          if (angular.isArray(newVal)) {
            promise = $q.all(newVal);
          } else if (newVal.$promise) {
            promise = newVal.$promise;
          } else if (newVal.promise) {
            promise = newVal.promise;
          } else {
            promise = newVal;
          }

          disable();

          return promise["finally"](function() {
            enable();
          });
        });
      }
    };
  }
  ]);
angular.module('readerApp').directive('cachedSrc', [
   'IMGCache', function(
    IMGCache
  )
  {
    return {
      restrict: 'A',
      scope:    { cachedSrc: '@' },

      link: function($scope, element, attrs) {
        const IMAGE_PLACEHOLDERS = {
          "person_photo":       "/images/author-no-photo.png"
        };

        var placeholderUrl = IMAGE_PLACEHOLDERS[element.attr('cached-src-placeholder')];

        var getSuccess = function(url) {
          attrs.$set('src', url);
        };

        var getFailure = function() {
          attrs.$set('src', $scope.cachedSrc);
        };

        $scope.$watch('cachedSrc', function(newUrl) {
          if (newUrl && newUrl != '') {
            IMGCache.get($scope.cachedSrc).then(
              getSuccess,
              getFailure
            );
          } else if (placeholderUrl) {
            attrs.$set('src', placeholderUrl);
          } else {
            attrs.$set('src', '');
          }
        });

      }
    };
  }
])
angular.module('readerApp').directive('cachedBgUrl', [
   'IMGCache', 'KeyVal', function(
    IMGCache,   KeyVal
  )
  {
    return {
      restrict: 'A',
      scope:    { cachedBgUrl: '@' },

      link: function($scope, element, attrs) {
        const IMAGE_PLACEHOLDERS = {
          "book_cover_preview": "images/book_cover_placeholder.png",
          "book_cover_full":    "images/book_cover_placeholder_3x.png",
          "person_photo":       "images/author-no-photo.png"
        };

        var placeholderUrl = IMAGE_PLACEHOLDERS[element.attr('cached-bg-placeholder')];

        var updateClasses = function (url) {
          var image = new Image();

          image.onload = function () {
            var width  = image.width,
              height = image.height;

            if(width > height) {
              element.removeClass('port');
              element.addClass('land')
            } else {
              element.removeClass('land');
              element.addClass('port')
            }
          };

          image.src = url;
        };

        var getSuccess = function(url) {
          element.css("background-image", "url('" + url + "')");
          updateClasses(url)
        };

        var getFailure = function() {;
          if (navigator.connection == undefined) {
            element.css("background-image", "url('" + $scope.cachedBgUrl + "')");
            updateClasses($scope.cachedBgUrl);
            return;
          }

          if (navigator.connection.type == "wifi") {
            element.css("background-image", "url('" + $scope.cachedBgUrl + "')");
            updateClasses($scope.cachedBgUrl);
            return;
          }

          KeyVal.get('settings_wifi_only', {reject: false}).then(function(value) {
            if (value && placeholderUrl) {
              element.css("background-image", "url('" + placeholderUrl + "')");
              updateClasses(placeholderUrl)
            } else {
              element.css("background-image", "url('" + $scope.cachedBgUrl + "')");
              updateClasses($scope.cachedBgUrl)
            }
          });

        };

        $scope.$watch('cachedBgUrl', function(newUrl) {
          if (newUrl) {
            IMGCache.get($scope.cachedBgUrl).then(
              getSuccess,
              getFailure
            );
          } else if (placeholderUrl) {
            element.css("background-image", "url('" + placeholderUrl + "')");
            updateClasses(placeholderUrl)
          } else {
            element.css("background-image", "");
          }

        });
      }
    };
  }
]);
angular.module('readerApp')
  .directive('starRating', function () {
    return {
      restrict: 'A',
      template: '<ul class="rating"><li ng-repeat="star in stars" class="ion" ng-class="star" class="" ng-click="toggle($index)"></li><li ng-if="votesValue" class="counter">({{votesValue}})</li></ul>',
      scope: {
        ratingValue: '=',
        votesValue: '=',
        max: '=',
        readonly: '@',
        onRatingSelected: '&'
      },
      link: function (scope) {
        var updateStars = function () {
          var ratingValue = scope.ratingValue || 0;
          scope.stars = [];
          for (var i = 0; i < scope.max; i++) {
            var _i = i;
            scope.stars.push({
              'ion-ios-star': _i < ratingValue,
              'ion-ios-star-half': ratingValue % 1 > 0 && _i === Math.floor(ratingValue),
              'ion-ios-star-outline': _i >= ratingValue
            });
          }
        };
        scope.toggle = function (index) {
          if (angular.isUndefined(scope.readonly)) {
            scope.ratingValue = index + 1;
            scope.onRatingSelected({rating: index + 1});
          }
        };
        scope.$watch('ratingValue', function () {
            updateStars();
          }
        );
      }
    };
  })
;
angular.module('readerApp').directive('onLongPress', ['$timeout', function($timeout) {
  return {
    restrict: 'A',
    link: function($scope, $elm, $attrs) {
      $elm.bind('touchstart', function(evt) {
        // Locally scoped variable that will keep track of the long press

        $scope.longPress = 'start';
        $scope.touchTarget = evt.changedTouches[0]

        // We'll set a timeout for 600 ms for a long press
        $timeout(function() {
          if ($scope.longPress == 'start') {
            $scope.longPress = 'proccess';
            // If the touchend event hasn't fired,
            // apply the function given in on the element's on-long-press attribute
            $scope.$apply(function() {
              $scope.$eval($attrs.onLongPress)
            });
          }
        }, 600);
      });

      $elm.bind('touchend', function(evt) {
        dist = $scope.touchTarget.clientY - evt.changedTouches[0].clientY
        if(Math.abs(dist) > 20) {
          return $scope.longPress = 'end';
        }
        // If there is an on-touch-end function attached to this element, apply it
        if ($attrs.onTouchEnd && $scope.longPress !== 'proccess') {
          $scope.$apply(function() {
            $scope.$eval($attrs.onTouchEnd)
          });
        }

        // Prevent the onLongPress event from firing
        $scope.longPress = 'end';
      });
    }
  };
}])
angular.module('readerApp')
  .directive('expandable', function () {
    return {
      restrict: 'A',
      link: function (scope, elem, params) {
        var wrapper = angular.element(elem);

        //console.log("arguments", scope, elem, params)

        wrapper.css('height', '13rem');
        wrapper.append('<a class="expand"><span class="ion ion-ios-arrow-down"></span></a>')

        var button_wrapper = wrapper.find('a')
        var button = button_wrapper.find('span')

        button.on('click', function () {
          wrapper.css('height', 'auto');
          button_wrapper.remove()
        })


      }
    };
  })
;
angular.module('readerApp').directive('downloadProgress',
  function() {
    return {
      restrict:    'E',
      scope:       { download: '=' },
      templateUrl: 'templates/download-progress.html',

      link: function($scope, element, attrs) {
        // 70% for json, 30% for images
        const IMAGES_PERCENT = 30;

        $scope.status  = 'standby';

        $scope.percent      = 0;
        $scope.percentLeft  = 0;
        $scope.percentRight = 0;

        var onProgress = function(status) {
          $scope.status  = 'active';
          $scope.ready   = status.ready || 0;
          $scope.total   = status.total;
          $scope.images  = status.images;

          if ($scope.images) {
            if (angular.isNumber($scope.total) && $scope.total > 0) {
              $scope.percent = (IMAGES_PERCENT * $scope.ready) / $scope.total + (100 - IMAGES_PERCENT);
            } else {
              $scope.percent = 100 - IMAGES_PERCENT;
            }
          } else {
            if (angular.isNumber($scope.total) && $scope.total > 0) {
              $scope.percent = ((100 - IMAGES_PERCENT) * $scope.ready) / $scope.total;
            } else {
              $scope.percent = 0;
            }
          }

          $scope.percentRight = Math.min($scope.percent, 50);
          $scope.percentLeft  = Math.max($scope.percent - 50, 0);
        };

        var onComplete = function() {
          $scope.status = 'standby';
        };

        $scope.$watch('download', function(newDownload) {
          if (newDownload) {
            if (newDownload.stats.images) {
              onProgress({ready: newDownload.stats.imagesReady, total: newDownload.stats.imagesTotal, images: true});
            } else {
              onProgress({ready: newDownload.stats.partsReady, total: newDownload.stats.partsTotal, images: false});
            }

            newDownload.promise.finally(onComplete, onProgress);
            $scope.status = 'pending';
          } else {
            $scope.status = 'standby';
          }
        });

        $scope.stop = function() {
          if ($scope.download) {
            $scope.download.stop();
          }
        }
      }
    };
  })
angular.module('readerApp').directive('buyingProgress', [
    'ngNotify', function(
     ngNotify
  )
  {
    return {
      restrict:    'E',
      scope:       { orderId: '=' },
      templateUrl: 'templates/buying-progress.html',

      link: function($scope, element, attrs) {
        $scope.showHint = function() {
          if ($scope.orderId) {
            ngNotify.set('Ожидание поступления оплаты. Номер заказа: ' + $scope.orderId.toString() + '.', {
              duration: 10000,
              type: 'success'
            });
          }
        }
      }
    };

  }]);
angular.module('readerApp').directive('onLongPress', ['$timeout', function($timeout) {
  return {
    restrict: 'A',
    link: function($scope, $elm, $attrs) {
      $elm.bind('touchstart', function(evt) {
        // Locally scoped variable that will keep track of the long press

        $scope.longPress = 'start';
        $scope.touchTarget = evt.changedTouches[0]

        // We'll set a timeout for 600 ms for a long press
        $timeout(function() {
          if ($scope.longPress == 'start') {
            $scope.longPress = 'proccess';
            // If the touchend event hasn't fired,
            // apply the function given in on the element's on-long-press attribute
            $scope.$apply(function() {
              $scope.$eval($attrs.onLongPress)
            });
          }
        }, 600);
      });

      $elm.bind('touchend', function(evt) {
        dist = $scope.touchTarget.clientY - evt.changedTouches[0].clientY
        if(Math.abs(dist) > 20) {
          return $scope.longPress = 'end';
        }
        // If there is an on-touch-end function attached to this element, apply it
        if ($attrs.onTouchEnd && $scope.longPress !== 'proccess') {
          $scope.$apply(function() {
            $scope.$eval($attrs.onTouchEnd)
          });
        }

        // Prevent the onLongPress event from firing
        $scope.longPress = 'end';
      });
    }
  };
}])
angular.module('readerApp').directive('purchaseBtn', [
   '$ionicModal', '$state', '$compile', 'User', 'Payments', 'ngNotify', function(
    $ionicModal,   $state,   $compile,   User,   Payments,   ngNotify
  )
  {
    return {
      restrict: 'EA',
      scope:    { book: '=' },
      link: function($scope, element, attrs) {

        var menuForBook = null;
        var $modal = null;
        var disabled = false;

        var template = [
          '<a class="button" ng-click="purchaseMenu(book)">',
            attrs.text || "<span ng-bind='book.price | price'></span>",
          '</a>'
        ].join('');

        element.html(template);
        $compile(element.contents())($scope);

        $ionicModal.fromTemplateUrl('templates/purchase-menu.html', {
          scope: $scope,
          animation: 'slide-in-up'
        }).then(function(modal) {
          $modal = modal;
          window.modal = $modal;
        });

        $scope.purchaseMenu = function(book) {
          if (disabled) {
            console.log('disabled');
            return;
          }

          disabled = true;

          Payments.getUserAmount().then(function(amount) {
            if(amount > book.price) {
              Payments.purchaseBookDeafult(book.id).catch(function() {
                disabled = false;
              });
            } else {
              menuForBook = book;
              $modal.show();

              disabled = false;
            }
          }, function() {
            disabled = false;
          });
        };

        $scope.purchaseWithMobileCommerce = function(book) {
          $modal.hide();
          $state.go('ui.purchase.mobile_commerce', {bookId: menuForBook.id})
        };

        $scope.purchaseWithBankCard = function(book) {
          $modal.hide();
          $state.go('ui.purchase.bank_card', {bookId: menuForBook.id})
        };

        $scope.closeModal = function() {
          $modal.hide();
        }


      }
    };
  }
])
angular.module('readerApp').directive('pageError', [
   '$compile', '$templateCache', '$http', '$state', '$stateParams', '$ionicHistory', function(
    $compile,   $templateCache,   $http,   $state,   $stateParams,   $ionicHistory
  )
  {
    return {
      restrict: 'A',
      scope:    { pageError: '=', reloadPage: '&' },

      link: function($scope, element, attrs, ctrl) {
        var firstEnter = true;

        if ($scope.$parent) {
          $scope.$parent.$on('$ionicView.beforeEnter', function() {
            if (firstEnter) {
              firstEnter = false;
            } else if ($scope.reloadPage && $scope.pageError) {
              $scope.doReloadPage();
            }
          });
        }

        $scope.doReloadPage = function() {
          window.doReloadPageScope = $scope;
          if ($scope.reloadPage) {
            $scope.pageError = null;
            $scope.reloadPage();
          }
        }

        const TEMPLATE_NAME = 'templates/page_error.html';

        $scope.$watch('pageError', function(newVal) {
          if (newVal) {
            var templateSuccess = function(template) {
              element.children().addClass('page-error-hide');
              element.append(template.data);
              $compile(element.find('page-error-handle'))($scope);
            }

            var templateFailure = function(template) {
              alert(newVal.message);
            }

            $http.get(TEMPLATE_NAME, {cache: $templateCache}).then(
              templateSuccess,
              templateFailure
            );
          } else {
            element.children().removeClass('page-error-hide');
            element.find('page-error-handle').remove();
          }
        });
      }
    };
  }
  ]);
angular.module('readerApp').directive('onpressKlass', function() {
  return {
    restrict: 'A',

    link: function($scope, element, attrs, ctrl) {
      var klassName = element.attr('onpress-klass');
      
      element.on('touchstart', function() {
        element.addClass(klassName);
      });

      element.on('touchend', function() {
        element.removeClass(klassName);
      });
    }
  };
});
angular.module('readerApp').filter('price', function() {
  return function(price) {
    if (price === undefined) {
      return '';
    }
    
    price = price.toString() || '0.0';
    price = price.split('.').join(',');
    price += ' р';

    return price;
  }; 
})

Array.prototype.contains = function(v) {
  for(var i = 0; i < this.length; i++) {
    if(angular.equals(this[i], v)) return true;
  }
  return false;
};

Array.prototype.unique = function() {
  var arr = [];
  for(var i = 0; i < this.length; i++) {
    if(!arr.contains(this[i])) {
      arr.push(this[i]);
    }
  }
  return arr;
}

if (!String.prototype.includes) {
  String.prototype.includes = function() {'use strict';
    return String.prototype.indexOf.apply(this, arguments) !== -1;
  };
}

angular.module('readerApp')
.config([
   '$stateProvider', '$locationProvider', '$compileProvider', '$urlRouterProvider', function(
    $stateProvider,   $locationProvider,   $compileProvider,   $urlRouterProvider
    )
  {
    // Add app:// protocol to trusted protocols (used by Firefox OS)
    $compileProvider.aHrefSanitizationWhitelist(/^\s*(file|https?|ftp|mailto|app|data):/);
    $compileProvider.imgSrcSanitizationWhitelist(/^\s*(https?|local|data|app|blob):/);

    $urlRouterProvider.when('', '/editors_choice').when('/', '/editors_choice');

    // Application routes
    $stateProvider
      .state('ui', {
        url: "",
        abstract: true,
        views: {
          'sidebar': {
            controller: 'SidebarCtl',
            templateUrl: "templates/sidebar.html"
          }
        }
      })
      .state('ui.editors_choice', {
        url: "/editors_choice",
        params: {
          scope: 'editors_choice',
          title: 'Выбор редакции'
        },
        views: {
          'header@': {
            templateUrl: "templates/catalog/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/catalog/content.html",
            controller:  'CatalogCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.bookmarks', {
        url: "/bookmarks/:session_id",
        views: {
          'header@': {
            templateUrl: "templates/bookmarks/header.html"
          },
          '@': {
            templateUrl: "templates/bookmarks/content.html",
            controller:  'BookmarksCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.novelty', {
        url: "/novelty",
        params: {
          scope: 'novelty',
          title: 'Новинки'
        },
        views: {
          'header@': {
            templateUrl: "templates/catalog/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/catalog/content.html",
            controller:  'CatalogCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.most_popular', {
        url: "/most_popular",
        params: {
          scope: 'most_popular',
          title: 'Популярные'
        },
        views: {
          'header@': {
            templateUrl: "templates/catalog/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/catalog/content.html",
            controller:  'CatalogCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.sequence', {
        url: "/sequence/:sequenceId/",
        params: {
          scope: 'sequence',
          title: 'Серия'
        },
        enable_menu: false,
        views: {
          'header@': {
            templateUrl: "templates/catalog/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/catalog/content.html",
            controller:  'CatalogCtl'
          }
        }
      })
      .state('ui.person', {
        url: "/person/:personId/",
        views: {
          'header@': {
            templateUrl: "templates/person/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/person/content.html",
            controller:  'PersonCtl'
          }
        }
      })
      .state('ui.my', {
        url: "/my/:session_id",
        views: {
          'header@': {
            templateUrl: "templates/my/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/my/content.html",
            controller:  'MyCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.my.archive', {
        url: "/archive/:session_id",
        views: {
          'header@': {
            templateUrl: "templates/my/archive/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/my/archive/content.html",
            controller:  'MyCtl'
          }
        }
      })
      .state('ui.genres_index', {
        url: "/genres_list/:is_sub_page/:title",
        params: {
          genre: null
        },
        views: {
          'header@': {
            templateUrl: "templates/genres/index/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/genres/index/content.html",
            controller: 'GenresIndexCtl'
          }
        }
      })
      .state('ui.genres_show', {
        url: "/genres_show/:title",
        params: {
          genre: null
        },
        views: {
          'header@': {
            templateUrl: "templates/genres/show/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/genres/show/content.html",
            controller: 'GenresShowCtl'
          }
        }
      })
      .state('ui.book', {
        url: "/book/:book_id",
        params: {
          book: null
        },
        views: {
          'header@': {
            templateUrl: "templates/book/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/book/content.html",
            controller:  'BookCtl'
          }
        }
      })
      .state('ui.book.authors', {
        url: "/authors",
        views: {
          'header@': {
            templateUrl: "templates/book/authors/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/book/authors/content.html",
            controller:  'BookAuthorsCtl'
          }
        }
      })
      .state('ui.book.comments', {
        url: "/comments",
        params: {
          book: null
        },
        views: {
          'header@': {
            templateUrl: "templates/book/comments/header.html",
            controller: 'CommentsHeaderCtl'
          },
          '@': {
            templateUrl: "templates/book/comments/content.html",
            controller:  'BookCommentsCtl'
          }
        }
      })
      .state('ui.book.comments.new', {
        url: "/new",
        params: {
          book: null
        },
        views: {
          'header@': {
            templateUrl: "templates/book/comments/new/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/book/comments/new/content.html",
            controller:  'BookCommentsNewCtl'
          }
        }
      })
      .state('ui.reader_trial', {
        url: "/reader/trial/:book_id",
        enable_menu: false,
        disable_header: true,
        cache: false,
        params: {
          book:  null,
          is_trial: true
        },
        views: {
          '@': {
            templateUrl: "templates/reader/content.html",
            controller:  'ReaderCtl'
          }
        }
      })
      .state('ui.reader_full', {
        url: "/reader/full/:book_id",
        enable_menu: false,
        disable_header: true,
        cache: false,
        cache_clear: true,
        params: {
          book:  null,
          is_trial: false
        },
        views: {
          '@': {
            templateUrl: "templates/reader/content.html",
            controller:  'ReaderCtl'
          }
        }
      })
      .state('ui.settings', {
        url: "/settings",
        views: {
          'header@': {
            templateUrl: "templates/settings/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/settings/content.html",
            controller: 'SettingsCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.account', {
        url: "/account",
        views: {
          'header@': {
            templateUrl: "templates/account/account_header.html",
            controller:  'AccountHeaderCtl'
          },
          '@': {
            templateUrl: "templates/account/account.html",
            controller: 'AccountCtl'
          }
        },
        cache_clear: true
      })
      .state('ui.account.profile', {
        url: "/account/profile",
        cache: false,
        views: {
          'header@': {
            templateUrl: "templates/account/profile/header.html",
            controller:  'AccountHeaderCtl'
          },
          '@': {
            templateUrl: "templates/account/profile/content.html",
            controller: 'AccountProfileCtl'
          }
        }
      })
      .state('ui.account.register', {
        url: "/account/register",
        views: {
          'header@': {
            templateUrl: "templates/account/register_header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/account/register.html",
            controller: 'AccountRegisterCtl'
          }
        }
      })
      .state('ui.account.recover_password', {
        url: "/account/recover_password",
        views: {
          'header@': {
            templateUrl: "templates/account/recover_password_header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/account/recover_password.html",
            controller: 'AccountRecoverPasswordCtl'
          }
        }
      })
      .state('ui.purchase', {
        url: "/purchase",
        abstract: true
      })
      .state('ui.purchase.bank_card', {
        url: "/bank_card/:bookId",
        cache: false,
        views: {
          'header@': {
            templateUrl: "templates/purchase/bank_card/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/purchase/bank_card/content.html",
            controller: 'PurchaseWithBankCardCtl'
          }
        }
      })
      .state('ui.purchase.mobile_commerce', {
        url: "/mobile_commerce/:bookId",
        cache: false,
        views: {
          'header@': {
            templateUrl: "templates/purchase/mobile_commerce/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/purchase/mobile_commerce/content.html",
            controller: 'PurchaseWithMobileCommerceCtl'
          }
        }
      })
      .state('ui.offer', {
        url: "/static/offer",
        cache: false,
        views: {
          'header@': {
            templateUrl: "templates/static/header.html",
            controller: 'HeaderCtl'
          },
          '@': {
            templateUrl: "templates/static/content.html"
          }
        }
      });
  }
]).config([
   '$httpProvider',function(
    $httpProvider
  )
  {
    // Support of 'Only WiFi' option
    $httpProvider.interceptors.push('wifiHttpInterceptor');

    // Parse XML respones and transform it to JS Objects
    $httpProvider.interceptors.push('xmlHttpInterceptor');

    // Put session_id in request param
    $httpProvider.interceptors.push('sidHttpInterceptor');

    // Console output of all requests and responses
    // $httpProvider.interceptors.push('debugHttpInterceptor');

    // Use urlencode instead of JSON for POST params
    $httpProvider.defaults.headers.post['Content-Type'] = 'application/x-www-form-urlencoded';

    $httpProvider.defaults.transformRequest = function(params) {
      var strings = [],
          p;
      
      for(prop in params) {
        if(angular.isArray(params[prop])) {
          if (params[prop].$comma) {
            var values = [];

            angular.forEach(params[prop], function(el) {
              values.push(encodeURIComponent(el));
            });

            strings.push(encodeURIComponent(prop) + "=" + values.join(','));
          } else {
            angular.forEach(params[prop], function(el) {
              strings.push(encodeURIComponent(prop) + "=" + encodeURIComponent(el));
            });
          }
        } else {
          strings.push(encodeURIComponent(prop) + "=" + encodeURIComponent(params[prop]));
        }
      }

      $httpProvider.defaults.timeout = 30 * 1000; // In milliseconds
        
      return strings.join("&");
    };
  }
]).config([
   'AnalyticsProvider',function(
    AnalyticsProvider
  ) {
    AnalyticsProvider
      .logAllCalls(true)
      .useECommerce(true, false)
      .setPageEvent('$stateChangeSuccess')
      // .enterDebugMode(true)
      .setHybridMobileSupport(true)
      .setAccount({
        tracker: 'UA-15543008-1',

        trackEvent:     true,
        trackEcommerce: true,

        fields: {
          cookieDomain: 'none'
        }
      });
  }
]).run([
   'Analytics', '$rootScope', function(
    Analytics,   $rootScope
  ) {
    window.Analytics = Analytics
  }
]).run(['$rootScope', '$templateCache', '$ionicHistory', 'PaymentsTrack', '$state', function($rootScope, $templateCache, $ionicHistory, PaymentsTrack, $state) {
  window.tplc = $templateCache;
  var enable_menu = true;

  $rootScope.alert = alert;

  $rootScope.swipeLeft = function () {
    if(enable_menu) {
      $rootScope.drawerOpened = false
    }
  };

  $rootScope.swipeRight = function () {
    if(enable_menu) {
      $rootScope.drawerOpened = true
    }
  };

  $rootScope.toggleDrawer = function ($event) {
    $event.stopPropagation();

    if(enable_menu) {
      $rootScope.drawerOpened = !$rootScope.drawerOpened
    }
  };

  $rootScope.goBack = function($event) {
    $event.stopPropagation();

    if ($ionicHistory.backView()) {
      $ionicHistory.goBack()
    } else {
      $state.go('ui.editors_choice');
    }
  };

  $rootScope.$on('$stateChangeStart', 
    function(event, toState, toParams, fromState, fromParams){
      $rootScope.drawerOpened = false;
      $rootScope.searchOpened = false;

      if(toState.enable_menu != undefined) {
        enable_menu = toState.enable_menu
      } else {
        toState.enable_menu = true
      }

      if(toState.disable_header) {
        $rootScope.disable_header = true
      } else {
        $rootScope.disable_header = false
      }

      if(toState.cache_clear) {
        console.log('clear history cache');
        $ionicHistory.clearCache();
      }
      return true;
  });

  $rootScope.noop = angular.noop;

  window.tracker = PaymentsTrack;
  //PaymentsTrack.setCheckInterval(5000)
  PaymentsTrack.start()
}]).value('cgBusyDefaults',{
  templateUrl: 'templates/angular-busy.html'
});