angular.module('rjr')

.factory('AudioPlayer', function($http) {
	// Lecteur HTML5
	function BrowserAudio() {
		this.audio = new Audio();
		this.audio.mozAudioChannelType = 'content'; // Pour Firefox OS
	}

	BrowserAudio.prototype.isUsable = function() {
		if(typeof(Audio) != 'undefined') {
			return true;
		} else {
			return false;
		}
	}

	BrowserAudio.prototype.setPlayingCallback = function(playCallback) {
		this.audio.addEventListener('play', playCallback);
	}

	BrowserAudio.prototype.setErrorCallback = function(errCallback) {
		var that = this;
		this.audio.addEventListener('error', function(err) {
			if(that.audio.error.code == MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED
				&& that.audio.src.match(/null$/) != null) {
				// Ca veux juste dire que l'objet Audio est réglé sur aucune source.
				// Donc ignorer l'erreur et la réserver pour les vrais problèmes
				return;
			}
			console.log(that.audio.error);
			errCallback(that.audio.error.code);
		});
	}

	BrowserAudio.prototype.play = function(url) {
		this.audio.src = url;
		this.audio.load();
		this.audio.play();
	}

	BrowserAudio.prototype.stop = function() {
		this.audio.pause();
		this.audio.src = null;
	}

	BrowserAudio.prototype.setVolume = function(vol) {
		this.audio.volume = vol / 100;
	}

	// Lecteur Cordova
	function CordovaAudio() {
		this.media = null;
		this.internalVolume = 1;
		this.timeoutHandle = null;
		this.errorEvent = function() {};
		this.playingEvent = function() {};
	}

	CordovaAudio.prototype.isUsable = function() {
		if(typeof(cordova) != 'undefined' 
			&& window.Media != null
			&& device.platform != 'firefoxos') {
			return true;
		} else {
			return false;
		}
	}

	CordovaAudio.prototype.setPlayingCallback = function(playCallback) {
		this.playingEvent = playCallback;
	}

	CordovaAudio.prototype.setErrorCallback = function(errCallback) {
		this.errorEvent = errCallback;
	}

	CordovaAudio.prototype.play = function(url) {
		var that = this;

		this.stop();
		this.media = new Media(url, function() {
			console.log('Success');
		}, function(err) {
			console.log('Error !');
			console.log(JSON.stringify(err));
		}, function(status) {
			console.log('Status');
			console.log(status);
			if(status === 1) {
				// Ca charge
				this.media.setVolume(this.internalVolume);
			}
			else if(status === 2) {
				clearTimeout(that.timeoutHandle);
				that.playingEvent();
			}
			else if(status === 3) {
				clearTimeout(that.timeoutHandle);
				that.errorEvent();
			}
		});
		this.media.play();

		this.timeoutHandle = setTimeout(function(){
			var msg = "La connexion au serveur est anormalement longue.\n"+
				"Il est possible que le flux sélectionné soit incompatible avec votre appareil"
				"ou actuellement indisponible.\n"+
				"Merci de choisir un autre flux dans le menu 'Sélection du flux radio'."
			if(typeof(navigator.notification) != 'undefined') {
				navigator.notification.alert(msg, null, 'Erreur de flux');
			} else {
				alert(msg);
			}
		}, 25000);
	}

	CordovaAudio.prototype.stop = function() {
		clearTimeout(this.timeoutHandle);
		if(this.media != null) {
			this.media.stop();
			this.media.release();
		}
	}

	CordovaAudio.prototype.setVolume = function(vol) {
		this.internalVolume = vol / 100;
		if(this.media != null) {
			this.media.setVolume(this.internalVolume);
		}
	}

	// One core to bring them all
	function AudioCore() {
		this.playing = false;
		this.loading = false;
		this.initialized = false;
		this.mediaUrl = "";

		this.streams = {
			defaultIndex: null,
			selectedIndex: null,
			list: null
		}
		
		this.engine = null;
		this.onReady = function() {};

		this.availableEngines = [
			CordovaAudio,
			BrowserAudio
		];

		var that = this;
		if(typeof(cordova) != 'undefined') {
			document.addEventListener('deviceready', function() {
				that.asyncInit();
			}, false);
		} else {
			this.asyncInit();
		}

		// Bricolage dégueu si l'évènement deviceready
		// se déclenche pas
		setTimeout(function() {
			that.asyncInit();
		}, 3000);
	}

	AudioCore.prototype.asyncInit = function() {
		if(this.initialized) {
			return;
		}

		var that = this;
		this.availableEngines.some(function(currentValue) {
				candidate = new currentValue();
				console.log('Test de ' + candidate.constructor.name);

				if(candidate.isUsable() && that.engine === null) {
					that.engine = candidate;
					console.log('Moteur chargé : ' + that.engine.constructor.name);
					return true;
				}
		});

		if(this.engine === null) {
			console.log('Aucun moteur compatible');
		}

		this.engine.setErrorCallback(function(errCode) {
			that.stop();

			if(errCode === MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED) {
				console.log('Seems like a stream error');
				// Signaler que le format n'est pas supporté
				var msg = "Nous sommes désolés, le flux sélectionné n'est pas compatible avec votre appareil"+
				" ou n'est pas disponible actuellement.\n"+
				"Merci de choisir un autre flux dans le menu 'Sélection du flux radio'."
				if(typeof(navigator.notification) != 'undefined') {
					navigator.notification.alert(msg, null, 'Erreur de flux');
				} else {
					alert(msg);
				}
			}
			else if(errCode != MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED) {
				// Seulement si l'erreur ne concerne pas un pb de format
				// Sinon ça va boucler pour rien...
				setTimeout(function() {
					that.play();
				}, 1000);
			}
		});

		this.engine.setPlayingCallback(function() {
			that.playing = true;
			that.loading = false;
		});

		this.fetchStreamList(function() {
			that.autoSelectStream(function(streamIndex) {
				that.setStream(streamIndex);
				
				that.initialized = true;
				that.onReady();
			});
		});
	}

	AudioCore.prototype.fetchStreamList = function(dataCallback) {
		var that = this;
		$http.get("http://rjrradio.fr/radio-data/streams.json")
		.success(function(data) {
			that.streams.defaultIndex = data.defaultIndex || 0;
			that.streams.list = data.streams;
			dataCallback();
		});
	}

	AudioCore.prototype.autoSelectStream = function(choiceCallback) {
		var choice = this.streams.defaultIndex || 0;
		choiceCallback(choice);

		if(this.isMobileInternet()) {
			var msg = "Utiliser de manière prolongée l'application sur un réseau 4G ou 3G est susceptible d'épuiser le crédit Internet inclus dans votre forfait.\n\n"+
				'Nous vous conseillons vivement de choisir un flux adapté à ce type de réseaux, identifié par la mention "Bas débit", dans le menu "Sélection du flux radio"';	
			
			if(typeof(navigator.notification) != 'undefined') {
				navigator.notification.alert(msg, null, 'Internet mobile détecté');
			} else {
				// Timeout de 1 ms pour async l'appel à alert()
				setTimeout(function() {
					alert(msg);
				}, 1);
			}
		}
	}

	AudioCore.prototype.isMobileInternet = function() {
		if(typeof(navigator.connection) === 'undefined') {
			return false;
		}

		switch(navigator.connection.type) {
			case Connection.CELL:
			case Connection.CELL_2G:
			case Connection.CELL_3G:
			case Connection.CELL_4G:
				return true;
			default:
				return false;
		}
	}

	AudioCore.prototype.setCallback = function(type, callback) {
		switch(type) {
			case 'ready':
				this.onReady = callback;
				if(this.initialized) {
					this.onReady();
				}
			default:
				// Nuthin'
				break;
		}

	}

	AudioCore.prototype.setURL = function(url) {
		this.streams.selectedIndex = null;
		this.mediaUrl = url;
	}

	AudioCore.prototype.setStream = function(index) {
		if(index >= 0 && index < this.streams.list.length) {
			this.streams.selectedIndex = index;
			this.mediaUrl = this.streams.list[index].url;
		} else {
			throw 'Unknown stream';
		}
	}

	AudioCore.prototype.getSelectedStream = function() {
		return this.streams.selectedIndex;
	}

	AudioCore.prototype.getStreamList = function() {
		return this.streams.list;
	}

	AudioCore.prototype.isPlaying = function() {
		var that = this;
		return function() {
			return that.playing;
		};
	}

	AudioCore.prototype.isLoading = function() {
		var that = this;
		return function() {
			return that.loading;
		};
	}

	AudioCore.prototype.play = function() {
		console.log('Playing '+this.mediaUrl);
		this.engine.play(this.mediaUrl);
		this.loading = true;
	}

	AudioCore.prototype.stop = function() {
		this.engine.stop();
		this.playing = false;
		this.loading = false;
	}

	AudioCore.prototype.toggle = function() {
		if(!this.playing && !this.loading) {
			this.play(this.mediaUrl);
		} else {
			this.stop();
		}
	}

	AudioCore.prototype.setVolume = function(vol) {
		this.engine.setVolume(vol);
	}

	return new AudioCore();
});