/**
 * @class
 * @description Класс брендированного прелодера компании PlayToMax
 */
var TTLoader =
{
	endCallback: null,
	loadedData: null,
	landscapeMode: false,
	skipPlayButton: false,

	/**
	 * Создание прелодера
	 * @param {Function} callback функция, которая будет вызвана по окончанию процесса загрузки
	 * @param {Boolean} landscape режим: false - портретный, true - лэндскейпный
	 * @param {Boolean} skipButton нужно ли пропустить обязательное нажание кнопки Play
	 * @param {Boolean} disableLogoLink нужно ли отключить ссылку на логотипе
	 * @example
	 * TTLoader.create(loadEnd, true, false);
	 * var preloader = new ImagesPreloader();
	 * preloader.load(images, TTLoader.loadComplete, TTLoader.showLoadProgress);
	 */
	create: function(callback, landscape, skipButton, disableLogoLink)
	{
		TTLoader.endCallback = callback;
		TTLoader.landscapeMode = landscape;
		TTLoader.skipPlayButton = skipButton;

		document.getElementById("progress_container").style.background = "#fff";

		document.getElementById("progress_container").style.zIndex = "1000";

		var c = document.getElementById("progress");
		c.setAttribute("valign", "top");
		c.style.verticalAlign = "top";
		c.style.background = "#fff";

		var d = document.createElement("div");

		var a = document.createElement("a");
		a.setAttribute("id", "tt_load_logo_c");

		var link = (window.ExternalAPI && ExternalAPI.getPreloaderURL) ? ExternalAPI.getPreloaderURL() : "http://playtomax.com/?utm_source=ff_marketplace&utm_medium=preloader&utm_campaign=save_planet";
		var target = "_blank";

		if(disableLogoLink)
		{
		    link = "javascript:void(0)";
		    target = "";
        }

		a.setAttribute("href", link);
		a.setAttribute("target", target);

		var logo = new Image();
		logo.setAttribute("id", "tt_load_logo");
		logo.setAttribute("border", "");
		logo.src = TTLoader.logoSrc;
		logo.style.cursor = "pointer";

		a.appendChild(logo);
		d.appendChild(a);
		c.appendChild(d);

		var d = document.createElement("div");
		d.setAttribute("id", "tt_load_progress_cont");
		d.setAttribute("align", "left");
		d.setAttribute("style", "padding: 1px; border: 2px solid #e44d26; background: #fff");

		var d2 = document.createElement("div");
		d2.setAttribute("id", "tt_load_progress");
		d2.setAttribute("style", "width: 0px; background: #e44d26;");
		d2.innerHTML = "&nbsp;";
		d.appendChild(d2);
		c.appendChild(d);

		var d = document.createElement("div");
		d.setAttribute("id", "tt_load_play");
		var button = new Image();
		button.setAttribute("id", "tt_load_button");
		button.src = TTLoader.buttonDisabledSrc;
		//button.style.display = "none";
		d.appendChild(button);
		c.appendChild(d);

		Utils.addEventListener("fitlayout", TTLoader.setSizes);

		TTLoader.setSizes();
	},

	/** @ignore */
	setSizes: function()
	{
		document.getElementById("progress_container").style.width = window.innerWidth + "px";
		document.getElementById("progress_container").style.height = Utils.ignoreMobileHeightCorrection ? window.innerHeight + "px" : "6144px";

		var c = Utils.globalScale*Utils.globalPixelScale;

		if(!TTLoader.landscapeMode) document.getElementById("progress").style.paddingTop = Math.floor(c*80)+"px";
		document.getElementById("tt_load_progress_cont").style.width = Math.floor(c*200)+"px";
		document.getElementById("tt_load_progress").style.height = Math.floor(c*12)+"px";
		document.getElementById('tt_load_progress').style.width = (c*TTLoader.progressVal*2)+"px";

		document.getElementById("tt_load_logo").style.marginTop = Math.floor(c*80)+"px";
		document.getElementById("tt_load_logo").setAttribute("width", Math.floor(c*300)+"px");
		document.getElementById("tt_load_logo").setAttribute("height", Math.floor(c*65)+"px");

		document.getElementById("tt_load_button").setAttribute("height", Math.floor(c*29)+"px");
		document.getElementById("tt_load_button").style.marginTop = Math.floor(c*30)+"px";
	},

	progressVal: 0,

	/**
	 * Изменение прогресса загрузки
	 * @param {Number} val значение в процентах от 0 до 100
	 */
	showLoadProgress: function(val)
	{
		if(val < 0) val = 0;
		if(val > 100) val = 100;

		TTLoader.progressVal = val;
		TTLoader.setSizes();
	},

	/**
	 * Завершение процесса загрузки
	 * @param {Object} загруженные данные
	 */
	loadComplete: function(data)
	{
		TTLoader.showLoadProgress(100);
		TTLoader.loadedData = data;
		var b = document.getElementById("tt_load_button");

		if(Utils.touchScreen)
		{
			//document.body.removeEventListener("touchstart", Utils.preventEvent);
			b.ontouchstart = TTLoader.close;
		}
		else b.onclick = TTLoader.close;

		b.style.cursor = "pointer";
		b.src = TTLoader.buttonSrc;

		b = document.getElementById("tt_load_progress");
		b.style.background = "transparent";

		b = document.getElementById("tt_load_progress_cont");
		b.style.border = "2px solid transparent";
		b.style.background = "transparent";

		document.getElementById("tt_load_button").style.display = "block";

		if(TTLoader.skipPlayButton) TTLoader.close();
	},

	/** @ignore */
	close: function()
	{
		clearTimeout(TTLoader.animateTimeout);
		TTLoader.endCallback(TTLoader.loadedData);
		if(Utils.touchScreen && !navigator.userAgent.match(/(iPad|iPhone|iPod).*CPU.*OS 7_\d/i)) document.body.addEventListener("touchstart", Utils.preventEvent, false);
	},

	logoSrc: "data:image/jpeg;base64,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",
	buttonDisabledSrc: "data:image/jpeg;base64,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",
	buttonSrc: "data:image/jpeg;base64,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"
};