/* unit and functional tests for FileSelectionService.js for Serene Notes */
/* Copyright © 2016 Hominid Software */
/*jsl:import ../source/FileSelectionService.js */

var ServiceTestJig = enyo.kind({
	name: "FileSelectionServiceTestJig",
	serviceFileSelectResponse: null,
	components: [
		{
			kind: "FileSelectionService",
			onFileSelect: "serviceFileSelect"
		}
	],
	serviceFileSelect: function (inSender, inResponse) {
		this.log(inResponse);
		this.serviceFileSelectResponse = inResponse;
	}
});


describe('FileSelectionService', function () {
	var div, serviceTestJig;
	beforeEach( function () {
		div = document.createElement("div");
		document.body.appendChild(div);
		serviceTestJig = new ServiceTestJig();
		serviceTestJig.renderInto(div);
	});
	
	afterEach(function () {
		serviceTestJig.destroy();
		document.body.removeChild(div);
	});


	it('should refuse to import image files',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		var file = {name: 'IMG0001.JPG', type: 'image/jpeg'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.argsFor(0)[1]).toEqual({message: "Not a text, Markdown nor HTML file"});
		expect(serviceTestJig.serviceFileSelectResponse).toEqual(null);
		expect(signalsSpy.calls.count()).toEqual(1);
	});

	it('should refuse to import office documents',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		var file = {name: 'report.docx', type: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.argsFor(0)[1]).toEqual({message: "Not a text, Markdown nor HTML file"});
		expect(serviceTestJig.serviceFileSelectResponse).toEqual(null);
		expect(signalsSpy.calls.count()).toEqual(1);
	});

	/** The only difference between this scenario and the next is the action */
	it('should restore files as Markdown with no coda, regardless of type',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		serviceTestJig.$.fileSelectionService.action = 'RESTORE';
		var file = {name: 'some-file.html', type: 'text/html'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: 'MARKDOWN'});
	});

	it('should import HTML files as HTML with a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'some-file.html', type: 'text/html'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: 'HTML', coda: file.name});
	});

	it('should import Markdown files as Markdown with a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'some-file.md', type: 'text/markdown'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: 'MARKDOWN', coda: file.name});
	});

	it('should import text files, that the user says are plain, as plain text, with a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');
		spyOn(serviceTestJig.$.fileSelectionService.$.markupDlg, 'show').and.callFake(function () {
			serviceTestJig.$.fileSelectionService.$.plainBtn.bubble('ontap');
		});

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'some-file.txt', type: 'text/plain'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: undefined, coda: file.name});
	});

	it('should import text files, that the user says are Markdown, as Markdown, with a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');
		spyOn(serviceTestJig.$.fileSelectionService.$.markupDlg, 'show').and.callFake(function () {
			serviceTestJig.$.fileSelectionService.$.markdownBtn.bubble('ontap');
		});

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'some-file.txt', type: 'text/plain'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: 'MARKDOWN', coda: file.name});
	});

	it('should import files, named like backup files, as Markdown without a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');
		var markdownDlgSpy = spyOn(serviceTestJig.$.fileSelectionService.$.markupDlg, 'show');

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'serene-notes-2015.txt', type: 'text/plain'};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(markdownDlgSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: 'MARKDOWN'});
	});

	it('should import files with no type as plain text with a coda',function () {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		serviceTestJig.$.fileSelectionService.action = 'IMPORT';
		var file = {name: 'some-file.log', type: ''};
		serviceTestJig.$.fileSelectionService.determineOptions(file);

		expect(signalsSpy.calls.count()).toEqual(0);
		expect(serviceTestJig.serviceFileSelectResponse.file).toEqual(file);
		expect(serviceTestJig.serviceFileSelectResponse.options).toEqual({markup: undefined, coda: file.name});
	});
});

