// Jasmine 2.2 tests of NoteService in Serene Notes 
// Copyright © 2015-2016 Hominid Software

/*jsl:import ../source/NoteService.js*/

var dbName = 'TestNoteDatabase';


var TestJig = enyo.kind({
	name: "TestJig",
	noteNotesResponse: null,
	components: [
		{	kind: "NoteService", 
			dbName: dbName,
			onNotes: "noteNotes"
		}
	],
	noteNotes: function (inSender, inResponse) {
		this.log(inResponse.items);
		this.noteNotesResponse = inResponse;
	}
});


describe("NoteService (synchronous)", function () {
	var noteService;
	beforeAll(function () {
		noteService = new NoteService({dbName: dbName}); 
	});
	afterAll(function() {
		noteService.destroy();		
	});


	it('can be instantiated', function () {
		expect(noteService).toBeTruthy();
		expect(noteService.timeout).toEqual(0);
	});


	it('normalizes words', function () {
		var normalizeWord = noteService.normalizeWord;
		expect(normalizeWord('editor-in-chief')).toEqual('EDITORINCHIEF');
		expect(normalizeWord('foo_bar')).toEqual('FOOBAR');
		expect(normalizeWord('_underlined_')).toEqual('UNDERLINED');
		expect(normalizeWord('614-555-1212')).toEqual('6145551212');
	});
	
	
	it('parses text into words', function () {
		expect(noteService.parseWords('')).toEqual([]);
		expect(noteService.parseWords('================')).toEqual([]);
	});
			
			
	it('denormalizes items for indexing', function () {		
		var dbItem1 = noteService.toDbItem({subject: 'To the ends of the Earth.'});
		expect(dbItem1.subject).toEqual('To the ends of the Earth.');
		expect(dbItem1.wordArr).toEqual(['TO', 'THE', 'ENDS', 'OF', 'EARTH']);
		expect(dbItem1.sortKey).toEqual(jasmine.any(Number));
		expect(dbItem1.sortKey + Date.now()).toBeLessThan(9000);
		expect(dbItem1.sortKey + Date.now()).not.toBeLessThan(0);
	});

	it('distingushes words with the same incipit', function () {		
		var dbItem6 = noteService.toDbItem({subject: 'to Togo'});
		expect(dbItem6.wordArr).toEqual(['TO', 'TOGO']);		
	});
	
	it("allows internal, but not surrounding, apostrophes", function () {
		expect(noteService.parseWords("I'll fix 'John's & F'lar's' boat")).toEqual(["I'LL", "FIX", "JOHN'S", "F'LAR'S", "BOAT"]); 
	});

	it('forms words using periods, then drops them', function () {
		var parsed = noteService.parseWords("C.A.T. scan Mr. Brown 45.2 end.");
		expect(parsed.length).toEqual(6);
		expect(parsed).toContain("CAT");
		expect(parsed).toContain("SCAN");
		expect(parsed).toContain("MR");
		expect(parsed).toContain("BROWN");
		expect(parsed).toContain("452");
		expect(parsed).toContain("END");
	});
			
	it('forms words using underscores, then drops them', function () {		
		var dbItem2 = noteService.toDbItem({subject: '__FILE_FORMAT__\r\n________'});
		expect(dbItem2.subject).toEqual('__FILE_FORMAT__\r\n________');
		expect(dbItem2.wordArr).toEqual(['FILEFORMAT']);
	});
		
	it('forms words using dashes, then drops them', function () {		
		var dbItem3 = noteService.toDbItem({subject: '~~~Uranium-238~~~'});
		expect(dbItem3.subject).toEqual('~~~Uranium-238~~~');
		expect(dbItem3.wordArr).toEqual(['URANIUM238']);
	});
		
	it('forms words using digits and dashes, then drops the dashes', function () {		
		var dbItem4 = noteService.toDbItem({subject: 'Phone Bob at 614-555-1212.'});
		expect(dbItem4.subject).toEqual('Phone Bob at 614-555-1212.');
		expect(dbItem4.wordArr).toEqual(['PHONE', 'BOB', 'AT', '6145551212']);		
	});
		
//	it('recognizes North American phone numbers', function () {		
//		var dbItem5 = noteService.toDbItem({subject: 'Phone Alice at (614) 555-1212.'});
//		expect(dbItem5.wordArr).toEqual(['PHONE', 'ALICE', 'AT', '6145551212']);		
//	});

	it('strips out HTML entities when parsing', function () {
		expect(noteService.parseWords(
			'&nbsp; &amp; &middot; &lt;foo&gt; &sect; &copy; &ordf;&laquo; &para; &#38; &#x27;')).
			toEqual(['FOO']);
	});

	it('strips out HTML tags when parsing', function () {
		expect(noteService.parseWords(
			'normal <b>bold</b> normal<br />line with <a href="http://g.co">link</a><!-- and comment -->')).
			toEqual(['NORMAL', 'BOLD', 'LINE', 'WITH', 'LINK']);
	});

	it('tokenizes "Það á sér langan aðdraganda." to "THATH", "A", "SER", "LANGAN", "ATHDRAGANDA"', function () {
		expect(noteService.parseWords('Það á sér langan aðdraganda.')).toEqual(['THATH', 'A', 'SER', 'LANGAN', "ATHDRAGANDA"]);
	});
	
	switch ((navigator.language || navigator.userLanguage).slice(0,2).toLowerCase()) {
		case 'de':
			it('in your German locale, tokenizes "café Peña słychać grüßen Åland Ælfred" to "CAFE", "PENA", "SLYCHAC", "GRUESSEN", "ALAND", "AELFRED"', function () {
				expect(noteService.parseWords('café Peña słychać grüßen Åland Ælfred')).toEqual(['CAFE', 'PENA', 'SLYCHAC', 'GRUESSEN', 'ALAND', 'AELFRED']);
			});
			
			it('in your German locale, tokenizes "Öffnen beider Türen?" to "OEFFNEN", "BEIDER", "TUEREN"', function () {
				expect(noteService.parseWords('Öffnen beider Türen?')).toEqual(['OEFFNEN', 'BEIDER', 'TUEREN']);
			});
	
			it('in your German locale, tokenizes ABCDEFG_ÀÁÂÃÄÅÆÇ to ABCDEFGAAAAAEAAEC', function () {
				expect(noteService.parseWords('ABCDEFG_ÀÁÂÃÄÅÆÇ')).toEqual(['ABCDEFGAAAAAEAAEC']);
			});
	
			it('in your German locale, tokenizes "ÒÓÔÕÖ×ØÙÚÛÜÝÞß" to "OOOOOE", "OUUUUEYTHSS"', function () {
				expect(noteService.parseWords('ÒÓÔÕÖ×ØÙÚÛÜÝÞß')).toEqual(['OOOOOE', 'OUUUUEYTHSS']);
			});
	
			it('in your German locale, tokenizes "àáâãäåæ_ç_èéêë_ìíîï" to "AAAAAEAAECEEEEIIII"', function () {
				expect(noteService.parseWords('àáâãäåæ_ç_èéêë_ìíîï')).toEqual(['AAAAAEAAECEEEEIIII']);
			});
	
			it('in your German locale, tokenizes "ð_ñ_òóôõö_÷_ø_ùúûü_ý_þ_ÿ" to "THNOOOOOE", "OUUUUEYTHY"', function () {
				expect(noteService.parseWords('ð_ñ_òóôõö_÷_ø_ùúûü_ý_þ_ÿ')).toEqual(['THNOOOOOE', 'OUUUUEYTHY']);
			});
			break;
			
		case 'es':
			it('in your Spanish locale, tokenizes "ÁÉÍÓÚÜÑ¿¡N" to "AEIOUUÑ", "N"', function () {
				expect(noteService.parseWords('ÁÉÍÓÚÜÑ¿¡N')).toEqual(['AEIOUUÑ', 'N']);
			});
			it('in your Spanish locale, tokenizes "áéíóúünñ¿¡" to "AEIOUUNÑ"', function () {
				expect(noteService.parseWords('áéíóúünñ¿¡')).toEqual(['AEIOUUNÑ']);
			});
			it('in your Spanish locale, tokenizes "2.º 3.ª" to "2O", "3A"', function () {
				expect(noteService.parseWords("2.º 3.ª")).toEqual(["2O", "3A"]);
			});
			break;
			
		default:
			it('in your locale, tokenizes "café Peña słychać grüßen Åland Ælfred" to "CAFE", "PENA", "SLYCHAC", "GRUSSEN", "ALAND", "AELFRED"', function () {
				expect(noteService.parseWords('café Peña słychać grüßen Åland Ælfred')).toEqual(['CAFE', 'PENA', 'SLYCHAC', 'GRUSSEN', 'ALAND', 'AELFRED']);
			});
			
			it('in your locale, tokenizes "Öffnen beider Türen?" to "OFFNEN", "BEIDER", "TUREN"', function () {
				expect(noteService.parseWords('Öffnen beider Türen?')).toEqual(['OFFNEN', 'BEIDER', 'TUREN']);
			});
	
			it('in your locale, tokenizes ABCDEFG_ÀÁÂÃÄÅÆÇ to ABCDEFGAAAAAAAEC', function () {
				expect(noteService.parseWords('ABCDEFG_ÀÁÂÃÄÅÆÇ')).toEqual(['ABCDEFGAAAAAAAEC']);
			});
	
			it('in your locale, tokenizes "ÈÉÊËÌÍÎÏÐÑ" to "EEEEIIIITHN"', function () {
				expect(noteService.parseWords('ÈÉÊËÌÍÎÏÐÑ')).toEqual(['EEEEIIIITHN']);
			});
	
			it('in your locale, tokenizes "ÒÓÔÕÖ×ØÙÚÛÜÝÞß" to "OOOOO", "OUUUUYTHSS"', function () {
				expect(noteService.parseWords('ÒÓÔÕÖ×ØÙÚÛÜÝÞß')).toEqual(['OOOOO', 'OUUUUYTHSS']);
			});
	
			it('in your locale, tokenizes "àáâãäåæ_ç_èéêë_ìíîï" to "AAAAAAAECEEEEIIII"', function () {
				expect(noteService.parseWords('àáâãäåæ_ç_èéêë_ìíîï')).toEqual(['AAAAAAAECEEEEIIII']);
			});
	
			it('in your locale, tokenizes "ð_ñ_òóôõö_÷_ø_ùúûü_ý_þ_ÿ" to "THNOOOOO", "OUUUUYTHY"', function () {
				expect(noteService.parseWords('ð_ñ_òóôõö_÷_ø_ùúûü_ý_þ_ÿ')).toEqual(['THNOOOOO', 'OUUUUYTHY']);
			});
	}
	
	it('tokenizes "ĀāĂăĄą ĆćĈĉĊċČč Ďď" to "AAAAAA", "CCCCCCCC", "DD"', function () {
		expect(noteService.parseWords('ĀāĂăĄą ĆćĈĉĊċČč Ďď')).toEqual(['AAAAAA', 'CCCCCCCC', 'DD']);
	});
	
	it('tokenizes "Đđ ĒēĔĕĖėĘęĚě ĜĝĞğ" to "DD", "EEEEEEEEEE", "GGGG"', function () {
		expect(noteService.parseWords('Đđ ĒēĔĕĖėĘęĚě ĜĝĞğ')).toEqual(['DD', 'EEEEEEEEEE', 'GGGG']);
	});
	
	it('tokenizes "ĠġĢģ ĤĥĦħ ĨĩĪīĬĭĮį" to "GGGG", "HHHH", "IIIIIIII"', function () {
		expect(noteService.parseWords('ĠġĢģ ĤĥĦħ ĨĩĪīĬĭĮį')).toEqual(['GGGG', 'HHHH', 'IIIIIIII']);
	});
	
	it('tokenizes "İı Ĳĳ Ĵĵ Ķķĸ ĹĺĻļĽľĿ" to "II", "IJIJ", "JJ", "KKK", "LLLLLLL"', function () {
		expect(noteService.parseWords('İı Ĳĳ Ĵĵ Ķķĸ ĹĺĻļĽľĿ')).toEqual(['II', 'IJIJ', 'JJ', 'KKK', 'LLLLLLL']);
	});
	
	it('tokenizes "ŀŁł ŃńŅņŇňŉŊŋ ŌōŎŏ" to "LLL", "NNNNNNNNGNG", "OOOO"', function () {
		expect(noteService.parseWords('ŀŁł ŃńŅņŇňŉŊŋ ŌōŎŏ')).toEqual(['LLL', 'NNNNNNNNGNG', 'OOOO']);
	});
	
	it('tokenizes "Őő Œœ ŔŕŖŗŘř ŚśŜŝŞş" to "OO", "OEOE", "RRRRRR", "SSSSSS"', function () {
		expect(noteService.parseWords('Őő Œœ ŔŕŖŗŘř ŚśŜŝŞş')).toEqual(['OO', 'OEOE', 'RRRRRR', 'SSSSSS']);
	});
	
	it('tokenizes "Šš ŢţŤťŦŧ ŨũŪūŬŭŮů" to "SS", "TTTTTT", "UUUUUUUU"', function () {
		expect(noteService.parseWords('Šš ŢţŤťŦŧ ŨũŪūŬŭŮů')).toEqual(['SS', 'TTTTTT', 'UUUUUUUU']);
	});
	
	it('tokenizes "ŰűŲų Ŵŵ ŶŷŸ ŹźŻżŽž ſ" to "UUUU", "WW", "YYY", "ZZZZZZ", "S"', function () {
		expect(noteService.parseWords('ŰűŲų Ŵŵ ŶŷŸ ŹźŻżŽž ſ')).toEqual(['UUUU', 'WW', 'YYY', 'ZZZZZZ', 'S']);
	});
	
	it('tokenizes fullwidth digits like normal digits', function () {
		expect(noteService.parseWords('０１２３４５６７８９')).toEqual(['0123456789']);
	});
	
	it('tokenizes fullwidth capital letters like normal letters', function () {
		expect(noteService.parseWords('ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ')).toEqual(['ABCDEFGHIJKLMNOPQRSTUVWXYZ']);
	});
	
	it('tokenizes fullwidth small letters like normal letters', function () {
		expect(noteService.parseWords('ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ')).toEqual(['ABCDEFGHIJKLMNOPQRSTUVWXYZ']);
	});
	
	it('tokenizes superscript digits like normal digits', function () {
		expect(noteService.parseWords('⁰¹²³⁴⁵⁶⁷⁸⁹ⁱⁿ')).toEqual(['0123456789IN']);
	});
	
	it('tokenizes subscript digits like normal digits', function () {
		expect(noteService.parseWords('₀₁₂₃₄₅₆₇₈₉ₐₑₒₓ')).toEqual(['0123456789AEOX']);
	});
	
	it('tokenizes circled numbers like normal digits', function () {
		expect(noteService.parseWords('①⑨⑩⑪⑯⑰⑳')).toEqual(['191011161720']);
	});
	
	it('tokenizes parenthesized numbers like normal digits', function () {
		expect(noteService.parseWords('⑴⑼⑽⑿⒀⒇')).toEqual(['1910121320']);
	});
});


describe("NoteService (empty)", function () {
	var originalTimeout, testJig;	
	
	beforeEach(function (done) {
		originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
	    jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;
	    
		var deleteAllRequest, startTime = Date.now();
		console.log("beforeEach | deleting " + dbName);
		try {
			deleteAllRequest = window.indexedDB.deleteDatabase(dbName);
		} catch (error) {
			console.error('caught: ' + error);
		}
		
		setTimeout(checkDeleteDone, 1000);
		
		function checkDeleteDone() {
			if (deleteAllRequest.readyState === 'done') {
				testJig = new TestJig();
				done();
			} else if (Date.now() - startTime > 7000) {
				done(new Error('deleteDatabase timed out')); 
			} else {
				setTimeout(checkDeleteDone, 1000);
			}
		}
	});

	afterEach(function () {
		testJig.destroy();
		
		jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
	});	

	it('when an invalid key is passed to getNote, fails the async', function (done) {
		var async = testJig.$.noteService.getNote(null);
		async.response(function (inSender, inResponse) {
			throw new Error('getNote with junk key should have failed');
		});
		async.error(checkFailure);

		function checkFailure(inSender, inError) {
			expect(inError.name).toEqual('DataError');
			
			done();
		}
	});
	
	it('when an unused valid key is passed to getNote, returns undefined', function (done) {
		var async = testJig.$.noteService.getNote(999);
		async.response(function (inSender, inResponse) {
			expect(inResponse.item).toBeUndefined();
			done();
		});
	});
	
	it('creates items using putNote & retrieves them using getNote', function (done) {
		var primaryKey1, primaryKey2, primaryKey3;
				
		var async1 = testJig.$.noteService.putNote({item: {subject: "ear early earth"}});
		async1.response(checkPut1);
		function checkPut1(inSender, inResponse) {
			primaryKey1 = inResponse.key;
			expect(primaryKey1).toEqual(1);
				
			var async2 = testJig.$.noteService.putNote({item: {subject: "leap learn loop"}});
			async2.response(checkPut2);
		}
		function checkPut2(inSender, inResponse) {
			primaryKey2 = inResponse.key;
			expect(primaryKey2).toEqual(2);
				
			var async3 = testJig.$.noteService.putNote({item: {subject: "Phone Dan at 555-1212."}});
			async3.response(checkPut3);
		}
		function checkPut3(inSender, inResponse) {
			primaryKey3 = inResponse.key;
			expect(primaryKey3).toEqual(3);

			var async = testJig.$.noteService.getNote(primaryKey3);
			async.response(checkGet);
			async.error(function (inSender, inError) {
				throw inError;
			});
		}
		function checkGet(inSender, inResponse) {
			expect(inResponse.item.key).toEqual(3);
			expect(inResponse.item.subject).toEqual("Phone Dan at 555-1212.");
			expect(inResponse.item.wordArr.length).toEqual(4);
			expect(inResponse.item.wordArr).toContain('PHONE');
			expect(inResponse.item.wordArr).toContain('DAN');
			expect(inResponse.item.wordArr).toContain('AT');
			expect(inResponse.item.wordArr).toContain('5551212');
			expect(inResponse.item.sortKey).toEqual(jasmine.any(Number));
			expect(inResponse.item.sortKey + Date.now()).toBeGreaterThan(0);
			expect(inResponse.item.sortKey + Date.now()).toBeLessThan(20000);

			done();
		}
	});

	
	it('updates items using putNote', function (done) {
		var primaryKey1, sortKey1, primaryKey2, primaryKey3;
				
		var async1 = testJig.$.noteService.putNote({item: {subject: "louver Lou lower"}});
		async1.response(checkPut1);
		function checkPut1(inSender, inResponse) {
			primaryKey1 = inResponse.key;
			expect(primaryKey1).toEqual(1);
				
			var async = testJig.$.noteService.getNote(primaryKey1);
			async.response(checkGet1);
		}
		function checkGet1(inSender, inResponse) {
			expect(inResponse.item.key).toEqual(1);
			expect(inResponse.item.subject).toEqual("louver Lou lower");
			expect(inResponse.item.wordArr.length).toEqual(3);
			expect(inResponse.item.wordArr).toContain('LOUVER');
			expect(inResponse.item.wordArr).toContain('LOU');
			expect(inResponse.item.wordArr).toContain('LOWER');
			expect(inResponse.item.sortKey + Date.now()).toBeGreaterThan(0);
			expect(inResponse.item.sortKey + Date.now()).toBeLessThan(20000);
			sortKey1 = inResponse.item.sortKey;
			
			// modifies the item subject
			inResponse.item.subject += ' modified';
			var async3 = testJig.$.noteService.putNote({item: inResponse.item, oldProps: {subject: 'louver Lou lower'}});
			async3.response(checkPut2);
		}
		function checkPut2(inSender, inResponse) {
			primaryKey2 = inResponse.key;
			expect(primaryKey2).toEqual(primaryKey1);
			
			// the modified item should still be accessible under the primaryKey
			var async = testJig.$.noteService.getNote(primaryKey2);
			async.response(checkGet2);
		}
		function checkGet2(inSender, inResponse) {
			expect(inResponse.item.key).toEqual(1);
			expect(inResponse.item.subject).toEqual("louver Lou lower modified");
			expect(inResponse.item.wordArr.length).toEqual(4);
			expect(inResponse.item.wordArr).toContain('LOUVER');
			expect(inResponse.item.wordArr).toContain('LOU');
			expect(inResponse.item.wordArr).toContain('LOWER');
			expect(inResponse.item.wordArr).toContain('MODIFIED');
			expect(inResponse.item.sortKey).toEqual(sortKey1);
			
			// modifies the item sortKey
			inResponse.item.sortKey -= 1;
			var async5 = testJig.$.noteService.putNote({item: inResponse.item, oldProps: {sortKey: sortKey1}});
			async5.response(checkPut3);
		}
		function checkPut3(inSender, inResponse) {
			primaryKey3 = inResponse.key;
			expect(primaryKey3).toEqual(primaryKey1);
			
			// the modified item should be accessible under the old primaryKey
			var async = testJig.$.noteService.getNote(primaryKey3);
			async.response(checkGet3);
		}
		function checkGet3(inSender, inResponse) {
			expect(inResponse.item.subject).toEqual("louver Lou lower modified");
			expect(inResponse.item.wordArr.length).toEqual(4);
			expect(inResponse.item.wordArr).toContain('LOUVER');
			expect(inResponse.item.wordArr).toContain('LOU');
			expect(inResponse.item.wordArr).toContain('LOWER');
			expect(inResponse.item.wordArr).toContain('MODIFIED');
			expect(inResponse.item.sortKey).toEqual(sortKey1 - 1);
						
			done();
		}
	});

	
	/* Yeah, this was a bug. */
	it("doesn't delete items on putNote, when the old sortKey equals the key", function (done) {
		var key = -Date.now();
				
		var async1 = testJig.$.noteService.putNote({item: {key: key, sortKey: -102, subject: "Oregon: Pacific Wonderland"}, oldProps: {sortKey: key}});
		async1.response(checkPut1);
		function checkPut1(inSender, inResponse) {
			expect(inResponse.key).toEqual(key);
				
			var async = testJig.$.noteService.getNote(key);
			async.response(checkGet1);
		}
		function checkGet1(inSender, inResponse) {
			expect(inResponse.item.key).toEqual(key);
			expect(inResponse.item.subject).toEqual("Oregon: Pacific Wonderland");
			expect(inResponse.item.sortKey).toEqual(-102);

			done();
		}
	});
	
	it("when putNote is passed a native object, fails the async", function (done) {
		var nativeObject = document.documentElement;
		
		var async = testJig.$.noteService.putNote({item: nativeObject});
		async.response(function (inSender, inResponse) {
			done(new Error('should have failed cloning native'));
		});
		async.error(checkExpectedFail);
		
		function checkExpectedFail(inSender, inError) {
			expect(inError).toEqual(jasmine.any(DOMException));
			expect(inError.name).toEqual('DataCloneError');
			
			done();
		}
	});

	
	it('deletes items using removeNote', function (done) {
		var primaryKey;
				
		var async1 = testJig.$.noteService.putNote({item: {subject: "zebra yellow exit wallow"}});
		async1.response(checkPut);
		async1.error(putFailed);
		function checkPut(inSender, inResponse) {
			primaryKey = inResponse.key;
			expect(primaryKey).toEqual(1);
				
			var async2 = testJig.$.noteService.removeNote(primaryKey);
			async2.response(checkRemove);
			async2.error(removeFailed);
		}
		function putFailed(inSender, inResponse) {
			done(new Error("put failed: " + inResponse));
		}
		
		function checkRemove(inSender, inResponse) {
			expect(inResponse.hasOwnProperty('result')).toEqual(true);
			expect(inResponse.result).toEqual(undefined);
				
			var async = testJig.$.noteService.getNote(primaryKey);
			async.response(checkGet);
			async.error(getFailed);
		}
		function removeFailed(inSender, inResponse) {
			done(new Error("remove failed: " + inResponse));
		}

		function checkGet(inSender, inResponse) {
			expect(inResponse.hasOwnProperty('item')).toEqual(true);
			expect(inResponse.item).toEqual(undefined);
						
			done();
		}
		function getFailed(inSender, inResponse) {
			done(new Error("get failed: " + inResponse));
		}
	});
	
	it('when deleting an unused key, succeeds', function (done) {
		var async = testJig.$.noteService.removeNote(9000000);
		async.response(checkRemove);
		function checkRemove(inSender, inResponse) {
			expect(inResponse).toEqual(jasmine.any(Object));
			done();
		}	
	});
	
	it('when deleting an invalid key, fails the async', function (done) {
		var async = testJig.$.noteService.removeNote({});
		async.error(checkExpectedFail);
		function checkExpectedFail(inSender, inError) {
			expect(inError.name).toEqual('DataError');
			done();
		}
	});
});


describe("NoteService (populated)", function () {
	var originalTimeout, testJig;	
	var items = [
		{subject: '6 I have your cheese.'},
		{subject: '5 That there, with cheese!'},
		{subject: '4 To Have And Have Not'},
		{subject: '3 I have that for you!'},
		{subject: '2 Was I young?'},
		{subject: '1 With you, I have all.'}
	];
	var i;
		
	beforeEach(function (done) {
		var deleteAllRequest, startTime = Date.now();
		originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
	    jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;
	    
		console.log("beforeEach | deleting " + dbName);
		try {
			deleteAllRequest = window.indexedDB.deleteDatabase(dbName);
		} catch (error) {
			console.error('caught: ' + error);
		}
		
		setTimeout(createAfterDelete, 1000);
		
		function createAfterDelete() {
			if (deleteAllRequest.readyState === 'done') {
				testJig = new TestJig();
				
				i=0;
				putNext();
			} else if (Date.now() - startTime > 5000) {
				done(new Error('deleteDatabase timed out')); 
			} else {
				setTimeout(createAfterDelete, 1000);
			}
		}
		
		function putNext() {
			if (i<items.length) {
				var async = testJig.$.noteService.putNote({item: items[i]});
				async.response(putNext);
				++i;
			} else {   // complete
				console.log('populate complete');
				done();
			}
		}		
	});

	afterEach(function () {
		testJig.destroy();

		jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
	});	
	
	
	it('retrieves all notes when searched on 0 words', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(false);
			expect(inResponse.items.length).toEqual(6);
			expect(inResponse.items[0].subject).toEqual('1 With you, I have all.');
			expect(inResponse.items[1].subject).toEqual('2 Was I young?');
			expect(inResponse.items[4].subject).toEqual('5 That there, with cheese!');
			expect(inResponse.items[5].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves four exact matches for "have"', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('have');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(4);
			expect(inResponse.items[0].subject).toEqual('1 With you, I have all.');
			expect(inResponse.items[1].subject).toEqual('3 I have that for you!');
			expect(inResponse.items[2].subject).toEqual('4 To Have And Have Not');
			expect(inResponse.items[3].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves four stem matches for "you", in sortKey order', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('you');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(4);
			expect(inResponse.items[0].subject).toEqual('1 With you, I have all.');
			expect(inResponse.items[1].subject).toEqual('2 Was I young?');
			expect(inResponse.items[2].subject).toEqual('3 I have that for you!');
			expect(inResponse.items[3].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves three matches for "have you", in sortKey order', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('have you');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(3);
			expect(inResponse.items[0].subject).toEqual('1 With you, I have all.');
			expect(inResponse.items[1].subject).toEqual('3 I have that for you!');
			expect(inResponse.items[2].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves three matches for "you have", in sortKey order', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('you have');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(3);
			expect(inResponse.items[0].subject).toEqual('1 With you, I have all.');
			expect(inResponse.items[1].subject).toEqual('3 I have that for you!');
			expect(inResponse.items[2].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves one match for "have you ch"', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('have you ch');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(1);
			expect(inResponse.items[0].subject).toEqual('6 I have your cheese.');
			
			done();
		}
	});
	
	it('retrieves no matches for "have you z"', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('have you z');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(0);
			
			done();
		}
	});
	
	it('doesn\'t return the same note twice', function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('th');
		function checkFetch(inSender, inResponse) {
			expect(inResponse.isSearch).toEqual(true);
			expect(inResponse.items.length).toEqual(2);
			expect(inResponse.items[0].subject).toEqual('3 I have that for you!');
			expect(inResponse.items[1].subject).toEqual('5 That there, with cheese!');
			
			done();
		}
	});	
	
	it("when fetchNotes is called, aborts any unfinished previous calls to fetchNotes", function (done) {
		testJig.noteNotes = checkFetch;
		testJig.$.noteService.fetchNotes('t');     // 3 matches
		testJig.$.noteService.fetchNotes('to');    // 1 match
		testJig.$.noteService.fetchNotes('too');   // 0 matches
		var numResponses = 0;
		function checkFetch(inSender, inResponse) {
			switch (++numResponses) {
				case 1:
					expect(inResponse.items.length).toEqual(0);

					setTimeout(done, 1000);   // allows time for another call to finish
					break;

				default:
					expect().fail("unexpected # calls to checkFetch: " + numResponses);
			}
		}
	});
});


describe("NoteService (prefs functionality)", function () {
	var originalTimeout, noteService;	
	
	beforeEach(function (done) {
		var deleteAllRequest, startTime = Date.now();

		originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
	    jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;

	    console.log("prefs beforeEach | deleting " + dbName);
		try {
			deleteAllRequest = window.indexedDB.deleteDatabase(dbName);
		} catch (error) {
			console.error('caught: ' + error);
			throw error;
		}
		
		setTimeout(checkDeleteDone, 1000);
		
		function checkDeleteDone() {
			if (deleteAllRequest.readyState === 'done') {
				noteService = new NoteService({dbName: dbName});
				done();
			} else if (Date.now() - startTime > 7000) {
				done(new Error('deleteDatabase timed out')); 
			} else {
				setTimeout(checkDeleteDone, 1000);
			}
		}
	});

	afterEach(function () {
		noteService.destroy();
		
		jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
	});	

	it('initially returns undefined for a pref', function (done) {
		var getPrefAsync1 = noteService.getPref('test');
		expect(getPrefAsync1).toEqual(jasmine.any(enyo.Async));
		getPrefAsync1.response(checkGet1);
		function checkGet1(inSender, inResponse) {
			expect(inResponse).toEqual(undefined);
			
			done();
		}
	});

	it('allows putting & getting a pref', function (done) {	
		var prefValue = Math.random();   // different than previous runs
		var putPrefAsync = noteService.putPref('test', prefValue);
		expect(putPrefAsync).toEqual(jasmine.any(enyo.Async));
		putPrefAsync.response(checkPut);
		function checkPut(inSender, inResponse) {
			expect(inResponse).toEqual('test');
			
			noteService.getPref('test').response(checkGet2);
		}
		function checkGet2(inSender, inResponse) {
			expect(inResponse).toEqual(prefValue);

			done();
		}
	});
	
	it('initially lists no prefs', function (done) {
		var listPrefsAsync = noteService.listPrefs();
		expect(listPrefsAsync).toEqual(jasmine.any(enyo.Async));
		listPrefsAsync.response(checkList);
		function checkList(inSender, inResponse) {
			expect(inResponse).toEqual({});
			
			done();
		}
	});
	
	it('lists all prefs', function (done) {
		var prefValue1 = Math.random();   // different than previous runs
		var prefValue2 = String.fromCharCode('A'.charCodeAt(0) + Math.random() * 26);
		noteService.putPref('test1', prefValue1).response(put2);
		function put2(inSender, inResponse) {
			expect(inResponse).toEqual('test1');
			noteService.putPref('test2', prefValue2).response(listAll);
		}
		function listAll(inSender, inResponse) {
			expect(inResponse).toEqual('test2');

			var listPrefsAsync = noteService.listPrefs();
			expect(listPrefsAsync).toEqual(jasmine.any(enyo.Async));
			listPrefsAsync.response(checkList);
		}
		function checkList(inSender, inResponse) {
			expect(inResponse).toEqual({test1: prefValue1, test2: prefValue2});
			
			done();
		}
	});
});


describe("NoteService (backups)", function () {
	var originalTimeout, noteService;
	var reader = new commonmark.Parser({smart: false});
	var writer = new commonmark.HtmlRenderer({safe: false});

	beforeAll(function (done) {
		var deleteAllRequest, startTime = Date.now();

		originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
		jasmine.DEFAULT_TIMEOUT_INTERVAL = 10000;

		console.log("backups beforeAll | deleting " + dbName);
		try {
			deleteAllRequest = window.indexedDB.deleteDatabase(dbName);
		} catch (error) {
			console.error('caught: ' + error);
			throw error;
		}

		setTimeout(checkDeleteDone, 100);

		function checkDeleteDone() {
			if (deleteAllRequest.readyState === 'done') {
				noteService = new NoteService({dbName: dbName});
				done();
			} else if (Date.now() - startTime > 7000) {
				fail(new Error('deleteDatabase timed out'));
				done();
			} else {
				setTimeout(checkDeleteDone, 100);
			}
		}
	});

	var fileSet = {};
	afterAll(function(done) {
		console.log("files to delete:", fileSet);
		var numRemainingFiles = Object.keys(fileSet).length;
		if (numRemainingFiles === 0) {
			finish();
		}

		if ('getDeviceStorage' in navigator) {   // FxOS
			var sdcard = navigator.getDeviceStorage('sdcard');
			for (filePath in fileSet) {
				var request = sdcard.delete(filePath);
				request.onsuccess = function () {
					console.log("File deleted:", this.result);
					if (--numRemainingFiles === 0) {
						finish();
					}
				};
				request.onerror = function () {
					console.error("Unable to delete the file:", this.error);
					if (--numRemainingFiles === 0) {
						finish();
					}
				};
			}
		} else if ('chrome' in window && 'fileSystem' in chrome) {   // Chrome app
			noteService.getPref('backupDirId').response(gotBackupDirPref);

			function gotBackupDirPref(inSender, backupDirId) {
				console.log("gotBackupDirPref", backupDirId, " [test cleanup]");
				if (backupDirId) {
					chrome.fileSystem.restoreEntry(backupDirId, function (restoredDirEntry){
						console.log("restoredDirEntry:", restoredDirEntry, " [test cleanup]");
						if (restoredDirEntry) {
							gotDirectory(restoredDirEntry);
						} else {
							console.error("restoring backup dir entry failed:", chrome.runtime.lastError, " [test cleanup]");
						}
					});
				} else {
					console.error("no backup dir ID; can't delete files. [test cleanup]");
				}
			}
		} else if ('webkitRequestFileSystem' in window) {   // Chrome browser
			window.webkitRequestFileSystem(window.PERSISTENT, 0, gotFileSystem, fileSystemFail);
			function gotFileSystem(fs) {
				fs.root.getDirectory('backups', {create: true}, gotDirectory, fileSystemFail);
			}
		}

		function gotDirectory(dirEntry) {
			for (filePath in fileSet) {
				dirEntry.getFile(filePath, {create: false, exclusive: false}, gotFile, fileSystemFail);
			}
		}
		function gotFile(fileEntry) {
			fileEntry.remove(logRemoval, fileSystemFail);
			function logRemoval() {
				console.log("removed:", fileEntry.name);
				if (--numRemainingFiles === 0) {
					finish();
				}
			}
		}
		function fileSystemFail(err) {
			console.error("FileSystem error:", err, " [test cleanup]");
			if (! ('name' in err) && ! ('message' in err) && 'target' in err && 'error' in err.target) {
				err = err.target.error;
				console.log("err appears to be ProgressEvent", err);
			}
			if (--numRemainingFiles === 0) {
				finish();
			}
		}

		function finish() {
			console.log("remove attempted on all files");
			noteService.destroy();
			jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
			done();
		}
	});

	it('converts HTML bold, italic & underlining to Markdown', function () {
		var html1 = '<b>bold</b> or <strong>strong</strong>';
		expect(toMarkdown(html1)).toEqual('**bold** or **strong**');
		var html2 = '<i>italic</i> or <em>emphasized</em>';
		expect(toMarkdown(html2)).toEqual('_italic_ or _emphasized_');
		var html3 = '<u>underlined</u><br >second line';
		expect(toMarkdown(html3)).toEqual('<u>underlined</u>  \nsecond line');
	});

	it('converts core HTML named character entities to Unicode characters', function () {
		expect(toMarkdown('&amp;&nbsp;&copy;&trade;&reg;&mdash;&ndash;&hellip;&laquo;&raquo;')).toEqual('&\xa0©™®—–…«»');
		expect(toMarkdown('&lsquo;&rsquo;&ldquo;&rdquo;&bull;&deg;')).toEqual('‘’“”•°');
		expect(toMarkdown('&times;&divide;')).toEqual('×÷');
	});

	it('converts HTML numeric character entities to Unicode characters', function () {
		expect(toMarkdown('&#247;&#x20AC;&#8203;')).toEqual('\xf7\u20AC\u200B');
	});

	it("converts HTML headers to Markdown", function () {
		var html1 = 'foo<h1>document title</h1>';
		expect(toMarkdown(html1)).toMatch(/foo\n\n# document title\s*/);
		var html2 = 'preface<h2>major section title</h2>';
		expect(toMarkdown(html2)).toMatch(/preface\n\n## major section title\s*/);
		var html2b = '<h2>Installing Ubuntu Touch on Nexus 4<div> </div></h2>'
		expect(toMarkdown(cleanHtml(html2b))).toMatch(/\n?## Installing Ubuntu Touch on Nexus 4\s*/);
		var html3 = '<h3>minor section title</h3>';
		expect(toMarkdown(html3)).toMatch(/\n?### minor section title\s*/);
		var html4 = '<h4>subsection title</h4>';
		expect(toMarkdown(html4)).toMatch(/\n?#### subsection title\s*/);
		var html5 = '<h5>paragraph title</h5>';
		expect(toMarkdown(html5)).toMatch(/\n?##### paragraph title\s*/);
		var html6 = '<h6>sub-paragraph title</h6>';
		expect(toMarkdown(html6)).toMatch(/\n?###### sub-paragraph title\s*/);
	});

	it('converts notes to Markdown text w/ newlines & date', function () {
		var date = new Date(Date.UTC(2014, 4, 20, 21, 48, 59, 999));
		var note = {key: 42, sortKey: -date.valueOf(), subject: 'one &amp; another<br>99 &gt; 42 &quot;&lt;&apos;<br data="geotag"><br>Idaho'};
		expect(noteService.noteToText(note)).toMatch(/one & another  \n99 > 42 "<\'  \n *\nIdaho\n2014-05-20T21:48:59.999Z\n\n\n\n/);
	});

	it('converts divs to newlines when converting notes to Markdown', function () {
		var date = new Date(Date.UTC(2014, 4, 20, 21, 48, 59, 999));
		var note = {key: 103, sortKey: -date.valueOf(),
			subject: '<div>first line<br></div><div>second line</div><div><br></div><div><br></div><div>after two line gap</div><br>Oregon<br>'};
		expect(noteService.noteToText(note)).toMatch(
			/  \nfirst line  \nsecond line  \n( *\n)+after two line gap  \n *\nOregon *\n2014-05-20T21:48:59.999Z\n\n\n\n/);
	});

	it('converts p tags to double newlines when converting notes to Markdown', function () {
		var date = new Date(Date.UTC(2012, 0, 30, 21, 48, 59, 999));
		var note = {key: 106, sortKey: -date.valueOf(),
			subject: 'typed<div><p>first</p>\n<p>second\n</p><p>third</p>non-par<br>non-par<p>fourth</p></div>non-par'};
		expect(noteService.noteToText(note)).toMatch(
			/typed  \n\nfirst\n\nsecond *\n\nthird\n\nnon-par  \nnon-par\n\nfourth\n\nnon-par\n2012-01-30T21:48:59.999Z\n\n\n\n/);
	});

	it('collapses multiple blank lines when converting notes to Markdown text', function () {
		var date = new Date(Date.UTC(2015, 3, 6, 22, 44, 55, 666));
		var note = {key: 104, sortKey: -date.valueOf(),
			subject: 'first line<br>second line<br><br><br><br>sixth line<br data="geotag"><br>Iowa'};
		expect(noteService.noteToText(note)).toMatch(
			/first line  \nsecond line  \n( *\n){1,2}sixth line  \n( *\n){1,2}Iowa\n2015-04-06T22:44:55.666Z\n\n\n\n/);
		var note2 = {key: 108, sortKey: -date.valueOf(),
			subject: 'first line<br><br><br>\u200B<br><br><br>seventh line'};
		expect(noteService.noteToText(note2)).toMatch(
			/first line  \n([\s\u200b]*\n){1,2}seventh line\n2015-04-06T22:44:55.666Z\n\n\n\n/);
		var note3 = {key: 108, sortKey: -date.valueOf(),
			subject: 'first line<br><br><br>\u200B<br>\u200B<br>\t\xa0<br>seventh line'};
		expect(noteService.noteToText(note3)).toMatch(
			/first line  \n([\s\u200b]*\n){1,2}seventh line\n2015-04-06T22:44:55.666Z\n\n\n\n/);
	});

	/* stripping tags is handled by cleanHtml(); no need to test all tags here */
	it('strips visual formatting tags when converting notes to Markdown text', function () {
		var date = new Date(Date.UTC(2016, 3, 10, 23, 51, 58, 333));
		var note = {key: 105, sortKey: -date.valueOf(), subject:
			'unstyled <span style="color: blue">blue</span> unstyled <font color="red" size="-2">small red</font> unstyled'};
		expect(noteService.noteToText(note)).toEqual(
			'unstyled blue unstyled small red unstyled\n2016-04-10T23:51:58.333Z\n\n\n\n');
	});

	it("allows text that looks like an ordered list to be a Markdown ordered list", function () {
		var date = new Date(Date.UTC(2016, 3, 16, 23, 51, 58, 333));
		var note = {key: 107, sortKey: -date.valueOf(), subject: '1. first<br>2. second'};
		expect(noteService.noteToText(note)).toEqual('1. first  \n2. second\n2016-04-16T23:51:58.333Z\n\n\n\n');
	});

	var fakeDates = [new Date(Date.UTC(2009, 0, 7, 11, 0)),
			new Date(Date.UTC(2009, 5, 6, 9, 0)),
			new Date(Date.UTC(2011, 7, 17, 13, 30)),
			new Date(Date.UTC(2013, 6, 23, 19, 45))];
	var fakeDateIdx = 0;
	function fakeDate() {
		return fakeDates[fakeDateIdx++];
	}

	if ('getDeviceStorage' in navigator ||   // FxOS
				'chrome' in window && 'fileSystem' in chrome ||   // Chrome app
				'webkitRequestFileSystem' in window) {   // Chrome browser
		it("writes backup, as FxOS app, as Chrome app or in Chrome browser", function (done) {
			fakeDateIdx = 0;
			spyOn(window, 'Date').and.callFake(fakeDate);
			var backupAsync = noteService.backupToFile(true);
			expect(backupAsync).toEqual(jasmine.any(enyo.Async));
			backupAsync.response(function (inSender, inResponse) {
				fileSet[inResponse.filePath] = true;
				expect(inResponse.isNew).toBeTruthy();
				expect(inResponse.filePath).toMatch(/\/serene-notes-.*\.txt/);
				done();
			});
			backupAsync.error(checkBackupFail);
		});

		it("doesn't write duplicate backups", function (done) {
			var key = -Date.UTC(2009, 6, 25, 13, 15);
			var item = {subject: "anything", key: key, sortKey: key};
			fakeDateIdx = 1;
			spyOn(window, 'Date').and.callFake(fakeDate);
			noteService.putNote({item: item}).response(function (inSender, inResponse) {
				expect(inResponse.key).toEqual(jasmine.any(Number));

				var backupAsync = noteService.backupToFile(true);
				expect(backupAsync).toEqual(jasmine.any(enyo.Async));
				backupAsync.response(function (inSender, inResponse) {
					fileSet[inResponse.filePath] = true;
					expect(inResponse.isNew).toEqual(true);
					expect(inResponse.filePath).toMatch(/\/serene-notes-2011-08-17T1330\.txt$/);

					var backupAsync2 = noteService.backupToFile(true);
					backupAsync2.response(function (inSender, inResponse) {
						fileSet[inResponse.filePath] = true;
						expect(inResponse.isNew).toEqual(false);
						expect(inResponse.filePath).toMatch(/\bserene-notes-2011-08-17T1330\.txt$/);
						done();
					});
					backupAsync2.error(checkBackupFail);
				});
				backupAsync.error(checkBackupFail);
			}).error(function (inSender, inErr) {
				fail(inErr);
				done();
			});
		});

		function checkBackupFail(inSender, inError) {
			if (/\(Mobile; rv:[^)]*\) Gecko/.test(navigator.userAgent)) {   // FxOS browser
				expect(inError.name).toEqual('TypeError');   // running in browser isn't privileged
				// SecurityError: thrown under earlier of FxOS browser (not privileged)
				// Unknown [Keon]: Default Media Location set to Internal rather than SD Card
				// NoModificationAllowedError: typically means a backup was done within the last minute
				expect(inError.message).toEqual("sdcard is null");
				//fail("running in FxOS browser isn't privileged");   // pending doesn't work here
			} else {
				fail(inError);
			}
			done();
		}
	} else {
		xit("can't write backup w/o DeviceStorage, chrome.fileSystem nor sandboxed FileSystem API");
	}


	it('restores Markdown emphasis into HTML', function () {
		var markdown1 = "one star *  \nbut _two kinds_ of *emphasis* and snake_case_symbol  \n*A*";
		expect(writer.render(reader.parse(markdown1))).toMatch(/<p>one star \*<br \/?>\n?but <em>two kinds<\/em> of <em>emphasis<\/em> and snake_case_symbol<br \/?>\n?<em>A<\/em><\/p>/);
		var markdown2 = "two stars ** ranking  \nbut __strong emphasis__ or **another way**  \n**A**";
		expect(writer.render(reader.parse(markdown2))).toMatch(/<p>two stars \*\* ranking<br \/?>\n?but <strong>strong emphasis<\/strong> or <strong>another way<\/strong><br \/*>\n?<strong>A<\/strong><\/p>/);
		var markdown4 = '<table class="pretty">\n    <tr>\n        <td>Foo</td>\n    </tr>\n</table>';
		expect(writer.render(reader.parse(markdown4))).toMatch(
			/<table class="pretty">\s+<tr>\s+<td>Foo<\/td>\s+<\/tr>\s+<\/table>/);
	});

	it("restores Markdown headers as HTML", function () {
		var markdown1 = "foo\n# Header Level 1\n## Header Level 2 #\n### Header Level 3\r\n###### Header Level 6";
		expect(writer.render(reader.parse(markdown1))).toMatch(
			/<p>foo<\/p>\s+<h1>Header Level 1<\/h1>\s+<h2>Header Level 2<\/h2>\s+<h3>Header Level 3\s*<\/h3>\s+<h6>Header Level 6<\/h6>/
		);
	});

	it("restores Markdown ordered lists as HTML", function () {
		var markdown1 = "prelude\n2. erste\n2. zwitte\n1. dritte\n\npostlude";
		expect(writer.render(reader.parse(markdown1))).toMatch(
			/(<p>)?prelude(<\/p>)?\n?<ol start="2">\n?<li>erste<\/li>\n?<li>zwitte<\/li>\n?<li>dritte<\/li>\n?<\/ol>\n?(<p>)?postlude(<\/p>)?\n?/
		);
	});

	it('round-trips some HTML to Markdown and back', function () {
		var html1 = '<p><em>emphasized text</em><br />\n<strong>strong emphasis</strong></p>\n';
		var markdown1 = toMarkdown(html1);
		expect(writer.render(reader.parse(markdown1))).toEqual(html1);
	});

	it("round-trips headers from HTML to Markdown and back", function () {
		var html = '<h1>document title</h1><h2>section title</h2><h2>another section</h2><h6>sub-paragraph</h6>';
		var markdown = toMarkdown(html);
		expect(writer.render(reader.parse(markdown))).toMatch(
			/\s*<h1>document title<\/h1>\s*<h2>section title<\/h2>\s*<h2>another section<\/h2>\s*<h6>sub-paragraph<\/h6>\s*/);
	});
	
	it('restores backup Markdown lines as note', function () {
		var backupLines = [
			'Walmart is not the _hellzone_ I expect.',
			'',
			'Britton Parkway, Franklin County, Ohio ',
			'2014-05-24T17:46:44.123Z ',
			' ',
			' ',
			' '
		];
		var note = noteService.linesToNote(backupLines, {markup: 'MARKDOWN'});
		expect(note.subject).toMatch(
			/(<p>)?Walmart is not the <em>hellzone<\/em> I expect.(<\/p>\s*<p>|<br><br>)Britton Parkway, Franklin County, Ohio(<\/p>)?/
		);
		expect(note.sortKey).toEqual(-Date.parse('2014-05-24T17:46:44.123Z'));
		expect(note.key).toBeLessThan(0);
	});

	it("imports plain text lines as note", function () {
		var backupLines = [
			"The *dao* that is seen",
			"Is not the **true dao**, until",
			"You bring fresh toner",
			"",
			" ",
			""
		];
		var note = noteService.linesToNote(backupLines);
		expect(note.subject).toEqual("The *dao* that is seen<br>Is not the **true dao**, until<br>You bring fresh toner");
		expect(note.sortKey).toBeLessThan(0);
		expect(note.key).toBeLessThan(0);
	});

	it('restores Markdown unordered lists as note', function () {
		var backupLines = [
			'before',
			'* alpha',
			'* beta',
			'* gamma',
			'',
			'after'
		];
		var note = noteService.linesToNote(backupLines, {markup: 'MARKDOWN'});
		expect(note.subject).toMatch(
			/(<p>)?before(<\/p>)?\s?<ul>\s?<li>alpha<\/li>\s?<li>beta<\/li>\s?<li>gamma<\/li>\s?<\/ul>\s?(<p>)?after(<\/p>)?\s?/
		);
		expect(note.sortKey).toBeLessThan(0);
		expect(note.key).toBeLessThan(0);
	});

	it('restores backup text as notes', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 42}); }, 0);
			return putAsync;
		});
		var backupText = 'Tadka Indian Cuisine  \nnice atmosphere  \nnever tried their curry, either\n\nW Dublin Granville Rd, Columbus, Franklin County, Ohio\n2014-05-22T07:34:34.085Z\n\n\n\n' +
		'The Matrix (1999)  \nwouldn\'t cows be a better heat source?\nWithout the ability to stage a revolt and all, you know  \naction, adventure, science-fiction\n2014-05-22T04:19:06.697Z\n\n\n\n' +
		'CbusJS d3  \n  \nNye County, Nevada\n2014-05-22T01:45:17.987Z\n\n\n\n';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(3);
			expect(putNoteStub.calls.count()).toEqual(3);
			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toMatch(
				/(<p>)?Tadka Indian Cuisine<br>\snice atmosphere<br>\snever tried their curry, either(<\/p>\s*)?|<br>\s?<br>\s?W Dublin Granville Rd, Columbus, Franklin County, Ohio\s*/
			);
			expect(putNoteStub.calls.argsFor(0)[0].item.sortKey).toEqual(-Date.parse('2014-05-22T07:34:34.085Z'));
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toMatch(
				/(<p>)?The Matrix \(1999\)<br>\s?wouldn't cows be a better heat source\? Without the ability to stage a revolt and all, you know<br>\s?action, adventure, science-fiction(<\/p>)?\s*/
			);
			expect(putNoteStub.calls.argsFor(1)[0].item.sortKey).toEqual(-Date.parse('2014-05-22T04:19:06.697Z'));
			expect(putNoteStub.calls.argsFor(2)[0].item.subject).toMatch(
				/(<p>)?CbusJS d3(<\/p>\s<p>|<br>\s?<br>\s?)Nye County, Nevada(<\/p>)?\s*/
			);
			expect(putNoteStub.calls.argsFor(2)[0].item.sortKey).toEqual(-Date.parse('2014-05-22T01:45:17.987Z'));
			expect(putNoteStub.calls.argsFor(2)[0].item.key).toBeLessThan(0);
			
			done();
		}
	});

	it('imports plain text as notes', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 43}); }, 0);
			return putAsync;
		});
		var importText = "There's rain today, but don't complain\nnor waste your tears of sorrow.\nIf you can wait, we'll have a great\nbig hurricane tomorrow.\n\n \n\n" +
			"Row, row, row your boat\ngently *down the stream*.  \nMerrily, merrily, merrily, merrily\nLife is but a dream.\n\n\n\n \n" +
			"One, two,    \n  buckle my shoe.  \n Three, four\n open the door  \nJune 6, 2009\n\n";
		var modDate = new Date("2005-11-25T23:55:15")
		var splitPromise = noteService.importText(importText, {lastModifiedDate: modDate});
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(3);
			expect(putNoteStub.calls.count()).toBe(3);

			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toEqual("There's rain today, but don't complain<br>nor waste your tears of sorrow.<br>If you can wait, we'll have a great<br>big hurricane tomorrow.");
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(modDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(modDate.getMonth());
			expect(noteDate.getDate()).toEqual(modDate.getDate());
			expect(noteDate.getHours()).toEqual(modDate.getHours());
			expect(noteDate.getMinutes()).toEqual(modDate.getMinutes());
			expect(noteDate.getSeconds()).toEqual(modDate.getSeconds());
			expect(noteDate.getMilliseconds()).toEqual(modDate.getMilliseconds());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toEqual("Row, row, row your boat<br>gently *down the stream*.  <br>Merrily, merrily, merrily, merrily<br>Life is but a dream.");
			noteDate = new Date(-putNoteStub.calls.argsFor(1)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(modDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(modDate.getMonth());
			expect(noteDate.getDate()).toEqual(modDate.getDate());
			expect(noteDate.getHours()).toEqual(modDate.getHours());
			expect(noteDate.getMinutes()).toEqual(modDate.getMinutes());
			expect(noteDate.getSeconds()).toEqual((modDate.getSeconds()+59) % 60);
			expect(noteDate.getMilliseconds()).toEqual((modDate.getMilliseconds()+999) % 1000);
			expect(putNoteStub.calls.argsFor(1)[0].item.key).toEqual(putNoteStub.calls.argsFor(1)[0].item.sortKey);

			expect(putNoteStub.calls.argsFor(2)[0].item.subject).toEqual("One, two,    <br>  buckle my shoe.  <br> Three, four<br> open the door  ");
			noteDate = new Date(-putNoteStub.calls.argsFor(2)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(2009);
			expect(noteDate.getMonth()).toEqual(5);
			expect(noteDate.getDate()).toEqual(6);
			expect(putNoteStub.calls.argsFor(2)[0].item.key).toEqual(putNoteStub.calls.argsFor(2)[0].item.sortKey);

			done();
		}
	});

	it('appends coda to notes from text files', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 101}); }, 0);
			return putAsync;
		});
		var importText = "Mary had a little lamb\nIts fleece was white as snow.\n2012-01-07T06:52:16.000Z\n\n\n\nMary, Mary\nquite contrary";
		var embeddedDate = new Date("2012-01-07T06:52:16.000Z");
		var modDate = new Date("2013-02-08T07:53:17.000Z");
		var splitPromise = noteService.importText(importText, {coda: 'Mary.txt', lastModifiedDate: modDate});
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(2);
			expect(putNoteStub.calls.count()).toBe(2);

			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toEqual("Mary had a little lamb<br>Its fleece was white as snow.<br><br>Mary.txt");
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(embeddedDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(embeddedDate.getMonth());
			expect(noteDate.getDate()).toEqual(embeddedDate.getDate());
			expect(noteDate.getHours()).toEqual(embeddedDate.getHours());
			expect(noteDate.getMinutes()).toEqual(embeddedDate.getMinutes());
			expect(noteDate.getSeconds()).toEqual(embeddedDate.getSeconds());
			expect(noteDate.getMilliseconds()).toEqual(embeddedDate.getMilliseconds());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toEqual("Mary, Mary<br>quite contrary<br><br>Mary.txt");
			noteDate = new Date(-putNoteStub.calls.argsFor(1)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(modDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(modDate.getMonth());
			expect(noteDate.getDate()).toEqual(modDate.getDate());
			expect(noteDate.getHours()).toEqual(modDate.getHours());
			expect(noteDate.getMinutes()).toEqual(modDate.getMinutes());
			expect(putNoteStub.calls.argsFor(1)[0].item.key).toEqual(putNoteStub.calls.argsFor(1)[0].item.sortKey);

			done();
		}
	});

	it('handles line breaks while importing plain text as notes', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 44}); }, 0);
			return putAsync;
		});
		var importText = "Cloud Develop Aug 2012\n" +
				"\n" +
				"In-Memory Big Data\n" +
				"Jon @jwbstr #gridgain @gridgain\n" +
				"\n" +
				"?impact as big as web & cloud\n";
		var splitPromise = noteService.importText(importText);
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(1);
			expect(putNoteStub.calls.count()).toBe(1);

			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toEqual("Cloud Develop Aug 2012<br><br>In-Memory Big Data<br>Jon @jwbstr #gridgain @gridgain<br><br>?impact as big as web & cloud");
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			var today = new Date();
			expect(noteDate.getFullYear()).toEqual(today.getFullYear());
			expect(noteDate.getMonth()).toEqual(today.getMonth());
			expect(noteDate.getDate()).toEqual(today.getDate());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			done();
		}
	});

	it('handles cr-lf same as lf while importing', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 45}); }, 0);
			return putAsync;
		});
		var importText = 'The first sentence.\r\nThe second sentence.\r\n\r\n' +
			'The second paragraph\r\n' +
			'\r\n\r\n\r\nAfter three blank lines, a third paragraph\r\n';
		var splitPromise = noteService.importText(importText);
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(2);
			expect(putNoteStub.calls.count()).toBe(2);

			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toEqual('The first sentence.<br>The second sentence.' +
					'<br><br>The second paragraph');
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			var today = new Date();
			expect(noteDate.getFullYear()).toEqual(today.getFullYear());
			expect(noteDate.getMonth()).toEqual(today.getMonth());
			expect(noteDate.getDate()).toEqual(today.getDate());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toEqual('After three blank lines, a third paragraph');
			expect(putNoteStub.calls.argsFor(1)[0].item.sortKey).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(1)[0].item.key).toEqual(putNoteStub.calls.argsFor(1)[0].item.sortKey);

			done();
		}
	});

	it('continues restoring when saving a note fails', function (done) {
		var numPutCalls = 0;
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () {
				if (++numPutCalls % 2 === 1) {
					putAsync.go({key: 46});
				} else {
					putAsync.fail(new Error('probe'));
				}
			}, 0);
			return putAsync;
		});
		var backupText = 'foo\n2014-03-01T07:34:34.085Z\n\n\n\n' +
			'bar\n2014-03-02T01:45:17.987Z\n\n\n\n' +
			'spam\n2014-03-03T01:45:17.987Z\n\n\n\n';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.keys.length).toEqual(2);
			expect(inResponse.errors.length).toEqual(1);
			expect(inResponse.errors[0].message).toEqual('probe');
			expect(putNoteStub.calls.count()).toEqual(3);
			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toMatch(/(<p>)?foo(<\/p>)?/);
			expect(putNoteStub.calls.argsFor(0)[0].item.sortKey).toEqual(-Date.parse('2014-03-01T07:34:34.085Z'));
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toMatch(/(<p>)?bar(<\/p>)?/);
			expect(putNoteStub.calls.argsFor(1)[0].item.sortKey).toEqual(-Date.parse('2014-03-02T01:45:17.987Z'));
			expect(putNoteStub.calls.argsFor(1)[0].item.key).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(2)[0].item.subject).toMatch(/(<p>)?spam(<\/p>)?/);
			expect(putNoteStub.calls.argsFor(2)[0].item.sortKey).toEqual(-Date.parse('2014-03-03T01:45:17.987Z'));
			expect(putNoteStub.calls.argsFor(2)[0].item.key).toBeLessThan(0);
			
			done();
		}
	});
	
	it('uses only contiguous blank lines to separate backup text into notes', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 47}); }, 0);
			return putAsync;
		});
		var backupText = 'alpha  \n  \nbeta  \n  \ngamma  \n  \ndelta  \n  \nepsilon\n2014-02-28T07:00:00.000Z\n\n\n\n';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toMatch(
				/(<p>)?alpha(<\/p>\s?<p>|<br>\s?<br>\s?)beta(<\/p>\s?<p>|<br>\s?<br>\s?)gamma(<\/p>\s?<p>|<br>\s?<br>\s?)delta(<\/p>\s?<p>|<br>\s?<br>\s?)epsilon(<\/p>)?\s*/
			);
			expect(putNoteStub.calls.argsFor(0)[0].item.sortKey).toEqual(-Date.parse('2014-02-28T07:00:00.000Z'));
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toBeLessThan(0);
			expect(putNoteStub.calls.count()).toEqual(1);
			expect(inResponse.keys.length).toEqual(1);
			expect(inResponse.errors.length).toEqual(0);
			
			done();
		}
	});
	
	it('uses p tags for Markdown paragraphs', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 48}); }, 0);
			return putAsync;
		});
		var backupText = 'first\n\nthird\n\nfifth\nsixth\n\n\nninth\n2014-01-15T07:00:00.000Z\n\n\n\npostlog\n\n\nendword';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toMatch(
				/<p>first<\/p>\s<p>third<\/p>\s<p>fifth +sixth<\/p>\s<p>ninth<\/p>\s*/
			);
			expect(putNoteStub.calls.argsFor(0)[0].item.sortKey).toEqual(-Date.parse('2014-01-15T07:00:00.000Z'));
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toBeLessThan(0);
			expect(putNoteStub.calls.count()).toEqual(2);
			expect(inResponse.keys.length).toEqual(2);
			expect(inResponse.errors.length).toEqual(0);
			expect(putNoteStub.calls.argsFor(1)[0].item.subject).toMatch(
				/<p>postlog<\/p>\s*<p>endword<\/p>\s*/
			);
			expect(putNoteStub.calls.argsFor(1)[0].item.sortKey).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(1)[0].item.key).toBeLessThan(0);
			
			done();
		}
	});

	it('restores empty backup text as 0 notes', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 49}); }, 0);
			return putAsync;
		});
		var backupText = '';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(0);
			expect(putNoteStub.calls.count()).toEqual(0);
			
			done();
		}
	});
	
	it('restores file of blank lines as zero or one note(s)', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 50}); }, 0);
			return putAsync;
		});
		var backupText = '\n\n\n\n\n\n\n\n\n';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		splitPromise.then(checkCounts);
		function checkCounts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).not.toBeLessThan(0);
			expect(inResponse.keys.length).not.toBeGreaterThan(1);
			expect(putNoteStub.calls.count()).not.toBeLessThan(0);
			expect(putNoteStub.calls.count()).not.toBeGreaterThan(1);
			
			done();
		}
	});
	
	it('restores files that don\'t end with newline', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 51}); }, 0);
			return putAsync;
		});
		var backupText = '  foo\n  bar  \nspam';
		var splitPromise = noteService.importText(backupText, {markup: 'MARKDOWN'});
		expect(splitPromise.then).toBeDefined();
		splitPromise.then(checkPuts);
		function checkPuts(inResponse) {
			expect(inResponse.errors.length).toEqual(0);
			expect(inResponse.keys.length).toEqual(1);
			expect(putNoteStub.calls.count()).toEqual(1);
			expect(typeof putNoteStub.calls.argsFor(0)[0].item.subject).toEqual('string');
			expect(putNoteStub.calls.argsFor(0)[0].item.subject).toMatch(/(<p>)?\s*foo\s+bar<br>\s?spam(<\/p>)?/);
			expect(putNoteStub.calls.argsFor(0)[0].item.sortKey).toBeLessThan(0);
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toBeLessThan(0);
			
			done();
		}
	});

	it('refuses to save notes larger than 100,000 chars', function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 102}); }, 0);
			return putAsync;
		});
		var backupText = 'Feb 16 00:17:00 frodo Java Updater[24847]: Untrusted apps are not allowed to connect to Window Server before login.\n' +
			'Feb 16 00:15:30 frodo spindump[24839]: Removing excessive log: file:///Library/Logs/DiagnosticReports/powerstats_2016-02-07-001552_frodo.diag\n'
		while (backupText.length <= 100000) {
			backupText += backupText;
		}
		noteService.importText(backupText, {markup: undefined}).then(checkResult);
		function checkResult(inResponse) {
			expect(inResponse.errors.length).toEqual(1);
			expect(inResponse.errors[0].name).toEqual('Error');
			expect(inResponse.errors[0].message).toEqual('Divide the text manually before importing.');

			expect(inResponse.keys.length).toEqual(0);
			expect(putNoteStub.calls.count()).toEqual(0);

			done();
		}
	});

	it("imports an HTML file as one note", function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 52}); }, 0);
			return putAsync;
		});
		var importText = '<html>\r\n<head>\r\n<title>Some Document</title>\r\n</head>\r\n<h1>The Title</h1>\r\n' +
			'<p>The first sentence.\r\nThe second sentence.</p>\r\n' +
			'<ol><li>The first item\r\n</li><li>The second item</li></ol>\r\n' +
			'\r\n\r\n\r\n<p>After three blank lines, another paragraph</p>\r\n';
		var modDate = new Date("1997-10-31T19:30");
		var importPromise = noteService.importText(importText, {markup: 'HTML', lastModifiedDate: modDate});
		expect(importPromise.then).toBeDefined();
		importPromise.then(function (result) {
			expect(result).toBeDefined();
			expect(result.keys.length).toEqual(1);
			expect(result.errors).toBeDefined();
			expect(result.errors.length).toEqual(0);

			expect(putNoteStub.calls.count()).toBe(1);
			expect(putNoteStub.calls.argsFor(0)[0].item.subject.replace(/[\s\u200B]+/g, " ")).toEqual(
				' <h1>The Title</h1> ' +
				'<p>The first sentence. The second sentence.</p> ' +
				'<ol><li>The first item </li><li>The second item</li></ol> ' +
				'<p>After three blank lines, another paragraph</p> ');
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(modDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(modDate.getMonth());
			expect(noteDate.getDate()).toEqual(modDate.getDate());
			expect(noteDate.getHours()).toEqual(modDate.getHours());
			expect(noteDate.getMinutes()).toEqual(modDate.getMinutes());
			expect(noteDate.getSeconds()).toEqual(modDate.getSeconds());
			expect(noteDate.getMilliseconds()).toEqual(modDate.getMilliseconds());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			done();
		})
	})

	it("appends coda to the note from an HTML file", function (done) {
		var putNoteStub = spyOn(noteService, "putNote").and.callFake(function() {
			var putAsync = new enyo.Async();
			setTimeout(function () { putAsync.go({key: 53}); }, 0);
			return putAsync;
		});
		var importText = '<html>\r\n<head>\r\n<title>Another Document</title>\r\n</head>\r\n<h1>Its Title</h1>\r\n' +
			'<p>First paragraph.</p>\r\n<p>Second paragraph!</p>';
		var modDate = new Date("1998-11-32T20:31");
		var importPromise = noteService.importText(importText, {markup: 'HTML', coda: 'boring stuff.html', lastModifiedDate: modDate});
		expect(importPromise.then).toBeDefined();
		importPromise.then(function (result) {
			expect(result).toBeDefined();
			expect(result.keys.length).toEqual(1);
			expect(result.errors).toBeDefined();
			expect(result.errors.length).toEqual(0);

			expect(putNoteStub.calls.count()).toBe(1);
			expect(putNoteStub.calls.argsFor(0)[0].item.subject.replace(/[\s\u200B]+/g, " ")).toEqual(
				' <h1>Its Title</h1> ' +
				'<p>First paragraph.</p> <p>Second paragraph!</p>' +
				'<br><br>boring stuff.html');
			var noteDate = new Date(-putNoteStub.calls.argsFor(0)[0].item.sortKey);
			expect(noteDate.getFullYear()).toEqual(modDate.getFullYear());
			expect(noteDate.getMonth()).toEqual(modDate.getMonth());
			expect(noteDate.getDate()).toEqual(modDate.getDate());
			expect(noteDate.getHours()).toEqual(modDate.getHours());
			expect(noteDate.getMinutes()).toEqual(modDate.getMinutes());
			expect(noteDate.getSeconds()).toEqual(modDate.getSeconds());
			expect(noteDate.getMilliseconds()).toEqual(modDate.getMilliseconds());
			expect(putNoteStub.calls.argsFor(0)[0].item.key).toEqual(putNoteStub.calls.argsFor(0)[0].item.sortKey);

			done();
		})
	})
});
