/* unit and functional tests for App.js (Serene Notes) */
/*jsl:import ../source/App.js */

describe('Serene Notes App', function () {
	var originalTimeout, div, app, noteKey = 60000;

	beforeEach( function () {
		try {
			originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
		    jasmine.DEFAULT_TIMEOUT_INTERVAL = 20000;

		    div = document.createElement("div");
			document.body.appendChild(div);
			app = new App({dbName: 'TestAppNoteDatabase'});
		} catch (err) {
			console.error('beforeEach:', err);
			throw err;
		}
	});
	
	afterEach(function (done) {
		try {	
			app.$.noteService.removeNote(noteKey).response(function finishCleanup() {
				try {
					app.destroy();
					document.body.removeChild(div);
					setTimeout(done, 1000);
				} catch (err) {
					console.error('afterEach app.destroy():', err);
					throw err;
				}
			});
		} catch (err) {
			console.error('afterEach:', err);
			throw err;
		} finally {
			jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
		}
	});
	
	// This checks that the second pane is correctly displayed, for the current window width.
	// Testing both cases requires testing with two window widths.
	it('should display ' + (window.innerWidth >= 640 ? '2 panes' : '1 pane') + ' at this window width', function (done) {
		var noteNotesStub = spyOn(app, "noteNotes").and.callFake(checkPanels);
		
		expect(app.$.listPane).toBeTruthy();
		expect(app.$.listPane.$).toEqual(jasmine.any(Object));
		expect(app.$.noteService).toBeTruthy();
		expect(app.$.noteService.$).toEqual(jasmine.any(Object));

		app.renderInto(div);

		var appNode = div.firstChild;
		expect(app.hasNode()).toEqual(appNode);
				
		function checkPanels (inSender, inEvent) {			
			var bounds = app.getBounds();
			expect(bounds.top).toEqual(jasmine.any(Number));   // value depends on runner HTML
			expect(bounds.left).toEqual(jasmine.any(Number));   // value depends on runner HTML
			expect(app.$.twoPanels.getIndex()).toEqual(window.innerWidth < 640 ? 0 : 1);
			
			app.$.listPane.doSetDetailsItem({item: {key: -101, subject: 'display details'}});
			expect(app.$.twoPanels.getIndex()).toEqual(1);

			done();
		}
	});
	
	it('should synchronously call fetchNotes on render', function (done) {
		var noteNotesStub = spyOn(app, "noteNotes").and.callFake(cleanup);
		var fetchNotesSpy = spyOn(app, 'fetchNotes').and.callThrough();
		app.fetchNotes = fetchNotesSpy;
		expect(fetchNotesSpy.calls.count()).toEqual(0);
		
		app.renderInto(div);		
		expect(fetchNotesSpy.calls.count()).toEqual(1);

		// clean up goes smoothly only if noteService is fully set up
		function cleanup(inSender, inEvent) {
			done();
		}
	});
	
	it('should asynchronously call populate after render', function (done) {
		var populateStub = spyOn(app.$.listPane, 'populate').and.callFake(checkList);
		app.renderInto(div);				

		function checkList(result) {
			expect(result.items.length).toBeGreaterThan(-1);   // initially, all items

			done();
		}
	});
	

	it('should update detail panel, list & DB when updateDbItem is called', function (done) {
		var signalsSpy = spyOn(enyo.Signals, 'send').and.callThrough();
		
		var setItemSpy = spyOn(app.$.detailPanel, 'setItemIfMatch');
		var listSpy = spyOn(app.$.listPane, 'modifyListItem');
		
		var specItem = {key: noteKey, subject: "Agriculture is power. -Jared Diamond"};
		var updateAsync = app.updateDbItem(null, {key: noteKey, item: specItem});
		
		expect(setItemSpy).toHaveBeenCalledWith(specItem, true);

		expect(listSpy.calls.count()).toEqual(1);
		expect(listSpy.calls.argsFor(0)[0].key).toEqual(noteKey);
		expect(listSpy.calls.argsFor(0)[0].item.subject).toEqual(specItem.subject);

		expect(updateAsync).toEqual(jasmine.any(enyo.Async));
		updateAsync.response(checkUpdate);
		
		function checkUpdate(inSender, inResponse) {
			app.$.noteService.getNote(noteKey).response(checkGet);
		}
		
		function checkGet(inSender, inResponse) {
			expect(inResponse.item).toBeDefined();
			expect(inResponse.item.key).toEqual(noteKey);
			expect(inResponse.item.subject).toEqual(specItem.subject);

//			expect(signalsSpy.calls.count()).toBe(0);
			done();
		}
	});

	it('when list bubbles onUpdateItem, should update detailPanel but not list', function (done) {
		var listSpy = spyOn(app.$.listPane, 'modifyListItem');
		var detailPanelSpy = spyOn(app.$.detailPanel, 'setItemIfMatch');
		
		app.$.listPane.bubble('onUpdateItem', {key: noteKey, item: {key: noteKey, subject: 'list new'}, oldProps: {subject: 'list old'}});
	
		expect(detailPanelSpy.calls.count()).toEqual(1);
		expect(listSpy.calls.count()).toEqual(0);

		// clean up goes smoothly only if noteService is fully set up
		setTimeout(done, 1000);
	});

	it('when detailPanel bubbles onUpdateItem, should update list but not detailPanel', function (done) {
		var listSpy = spyOn(app.$.listPane, 'modifyListItem');
		var detailPanelSpy = spyOn(app.$.detailPanel, 'setItemIfMatch');

		app.$.detailPanel.bubble('onUpdateItem', {key: noteKey, item: {key: noteKey, subject: 'detail new'}, oldProps: {subject: 'detail old'}});
		
		expect(detailPanelSpy.calls.count()).toEqual(0);
		expect(listSpy.calls.count()).toEqual(1);

		// clean up goes smoothly only if noteService is fully set up
		setTimeout(done, 1000);
	});


	it('should show the PrefHelpPanel when the list pane says to', function (done) {
		var noteNotesStub = spyOn(app, 'noteNotes').and.callFake(cleanup);

		app.renderInto(div);		
		expect(app.$.secondPane.getIndex()).toEqual(0);
		
		app.$.listPane.bubble('onPrefHelp');

		expect(app.$.secondPane.getIndex()).toEqual(2);
		
		// clean up goes smoothly only if noteService is fully set up
		function cleanup(inSender, inEvent) {
			done();
		}
	});
	
	it('should set geotagOnCreate pref UI from the DB when the PrefHelpPanel is shown', function (done) {
		app.$.noteService.putPref('geotagOnCreate', true).response(openPrefHelp);
		function openPrefHelp(inSender, inResponse) {
			spyOn(app.$.prefHelpPanel.$.geotagOnCreateTgl, 'setValue').and.callFake(checkSetValue);
			app.$.listPane.bubble('onPrefHelp');
		}
		function checkSetValue(value) {
			expect(value).toEqual(true);
			
			done();
		}
	});
	
	it('should set backupNightly pref UI from the DB when PrefHelpPanel is shown', function (done) {
		app.$.noteService.putPref('backupNightly', true).response(openPrefHelp);
		function openPrefHelp(inSender, inResponse) {
			spyOn(app.$.prefHelpPanel.$.backupNightlyTgl, 'setValue').and.callFake(checkSetValue);
			app.$.listPane.bubble('onPrefHelp');
		}
		function checkSetValue(value) {
			expect(value).toEqual(true);
			
			done();
		}
	});
	
	it('should update DB & call geolocation when pref UI is toggled', function (done) {
		var test = this.test, putPrefSpy, geolocationStub;
		app.$.noteService.putPref('geotagOnCreate', false).response(simulateUiChange);
		function simulateUiChange(inSender, inResponse) {
			putPrefSpy = spyOn(app.$.noteService, 'putPref').and.callThrough();
			geolocationStub = spyOn(navigator.geolocation, 'getCurrentPosition');
			
			app.$.prefHelpPanel.$.geotagOnCreateTgl.bubble('onChange', {value: true});
			setTimeout(waitForPut, 100);
		}
		function waitForPut() {
			if (putPrefSpy.calls.count()) {
				expect(putPrefSpy.calls.argsFor(0)).toEqual(['geotagOnCreate', true]);
				app.$.noteService.getPref('geotagOnCreate').response(checkPref);
			} else if (true /*! test.timedOut*/) {
				setTimeout(waitForPut, 100);
			}
		}
		function checkPref(inSender, inResponse) {
			expect(inResponse).toEqual(true);
			
			expect(geolocationStub.calls.count()).toEqual(1);
			expect(geolocationStub.calls.argsFor(0)[0]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[1]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[2].timeout).toBeGreaterThan(-1);
			expect(Boolean(geolocationStub.calls.argsFor(0)[2].enableHighAccuracy)).toEqual(false);
			expect(geolocationStub.calls.argsFor(0)[2].maximumAge).toBeGreaterThan(1000);
			
			done();
		}
	});

	
	it('should call geolocation when geotagBtn tapped', function (done) {
		var noteNotesStub = spyOn(app, 'noteNotes').and.callFake(manualGeotag);
		var geolocationStub = spyOn(navigator.geolocation, 'getCurrentPosition').and.callFake( function (success, error, options) {
			success({coords: {latitude: 101.13, longitude: 102.17}});
		});
		var openStreetMapStub = spyOn(app.$.openStreetMap, 'send');
		var signalsSpy = spyOn(enyo.Signals, 'send').and.callThrough();
		
		app.renderInto(div);		
		function manualGeotag() {
			expect(app.keysToGeotag).toEqual({});
			expect(app.$.detailPanel.getGeotagging()).toEqual(false);
			
			app.$.detailPanel.setItem({key: 69, sortKey: -Date.now(), subject: ''});
			app.$.detailPanel.$.geotagBtn.bubble('ontap');
			app.$.detailPanel.$.geotagBtn.bubble('ontap');		
			
			expect(69 in app.keysToGeotag).toBeTruthy();
		
			expect(geolocationStub.calls.count()).toEqual(1);
			expect(geolocationStub.calls.argsFor(0)[0]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[1]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[2].timeout).toBeGreaterThan(-1);
			expect(geolocationStub.calls.argsFor(0)[2].enableHighAccuracy).toEqual(true);
			expect(geolocationStub.calls.argsFor(0)[2].maximumAge).toBeGreaterThan(1000);
	
			expect(openStreetMapStub.calls.count()).toEqual(1);
			expect(openStreetMapStub.calls.argsFor(0)[0]).toEqual({lat: 101.13, lon: 102.17, zoom: 18, format: 'json', addressdetails:1, email:'sales@hominidsoftware.com'});
	
			expect(app.$.detailPanel.getGeotagging()).toEqual('REVERSE_GEOCODING');
	
//			expect(signalsSpy.calls.count()).toEqual(0);
	
			done();
		}
	});
	
	it('should display a message & not call OpenStreetMap when geolocation times out', function (done) {
		var noteNotesStub = spyOn(app, 'noteNotes').and.callFake(manualGeotag);
		var geolocationStub = spyOn(navigator.geolocation, 'getCurrentPosition').and.callFake( function (success, error, options) {
			error({code: 3});
		});
		var openStreetMapStub = spyOn(app.$.openStreetMap, 'send');
		var signalsSpy = spyOn(enyo.Signals, 'send').and.callThrough();
		
		app.renderInto(div);		
		function manualGeotag() {
			expect(app.keysToGeotag).toEqual({});
			expect(app.$.detailPanel.getGeotagging()).toEqual(false);
			
			app.$.detailPanel.setItem({key: 86, sortKey: -Date.now(), subject: ''});
			app.$.detailPanel.$.geotagBtn.bubble('ontap');
						
			expect(geolocationStub.calls.count()).toEqual(1);
			expect(geolocationStub.calls.argsFor(0)[0]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[1]).toEqual(jasmine.any(Function));
			expect(geolocationStub.calls.argsFor(0)[2].timeout).toBeGreaterThan(-1);
			expect(geolocationStub.calls.argsFor(0)[2].enableHighAccuracy).toEqual(true);
	
			expect(app.keysToGeotag).toEqual({});
			expect(app.$.detailPanel.getGeotagging()).toEqual(false);
	
			expect(openStreetMapStub.calls.count()).toEqual(0);
			
			expect(signalsSpy).toHaveBeenCalledWith('onShowMessage', jasmine.objectContaining({message: 'Position timeout.<br>Type in your location by hand.'}));
	
			done();
		}
	});

	
	it('should show a message when OpenStreetMap returns error', function (done) {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		app.keysToGeotag[noteKey] = "";
		app.$.detailPanel.setGeotagging('REVERSE_GEOCODING');
		
		app.$.openStreetMap.bubble('onResponse', {data: {error: 'foo'}});

		expect(app.keysToGeotag).toEqual({});
		expect(app.$.detailPanel.getGeotagging()).toEqual(false);

		expect(signalsSpy.calls.count()).toEqual(1);
		expect(signalsSpy.calls.argsFor(0)[1]).toEqual({message: 'OpenStreetMap says "foo"'});

		// clean up goes smoothly only if noteService is fully set up
		setTimeout(done, 1000);
	});
	
	it('should show a message when OpenStreetMap fails', function (done) {
		var signalsSpy = spyOn(enyo.Signals, 'send');

		app.keysToGeotag[noteKey] = "";
		app.$.detailPanel.setGeotagging('REVERSE_GEOCODING');
		
		app.$.openStreetMap.bubble('onError', {data: new Error('probe OpenStreetMap fail')});

		expect(app.keysToGeotag).toEqual({});
		expect(app.$.detailPanel.getGeotagging()).toEqual(false);

		expect(signalsSpy.calls.count()).toEqual(1);
		expect(signalsSpy.calls.argsFor(0)[1]).toEqual({message: 'Check your network connection'});

		// clean up goes smoothly only if noteService is fully set up
		setTimeout(done, 1000);
	});

	it('should append geotags when OpenStreetMap succeeds', function (done) {
		var updateDbItemSpy = spyOn(app, 'updateDbItem').and.callFake(checkUpdate);

		app.$.noteService.putNote({item: {key: noteKey, subject: 'HWÆT, WE GAR-DEna in geardagum'}}).
				response(function injectLocation() { 
			app.keysToGeotag[noteKey] = "";
			app.$.detailPanel.setGeotagging('REVERSE_GEOCODING');
			
			app.$.openStreetMap.bubble('onResponse', {data: {address:{"restaurant":"Pizza Rustica","house_number":"17",
					"road":"High St","city":"Columbus","county":"Franklin County","state":"Ohio","postcode":"43215",
					"country":"United States of America","country_code":"us"}}});
	
			expect(app.$.detailPanel.getGeotagging()).toEqual(false);
		});
		
		function checkUpdate(inSender, inEvent) {
			expect(inEvent.key).toEqual(noteKey);
			expect(inEvent.item.key).toEqual(noteKey);
			expect(inEvent.item.subject).toEqual('HWÆT, WE GAR-DEna in geardagum<br data="geotag"><br>Pizza Rustica, High St, Columbus, Franklin County, Ohio');
			expect(inEvent.item.sortKey).toBeLessThan(0);   // automatically assigned

			expect(Object.keys(app.keysToGeotag).length).toEqual(0);

			done();
		}
	});

	it('should create a record if neccessary, when OpenStreetMap succeeds', function (done) {
		var updateDbItemSpy = spyOn(app, 'updateDbItem').and.callFake(checkUpdate);

		app.keysToGeotag[noteKey] = "";
		app.$.detailPanel.setGeotagging('REVERSE_GEOCODING');
		
		app.$.openStreetMap.bubble('onResponse', {data: {"address":{"optician":"Miraflores","road":"Av. Urdaneta",
				"city":"Parroquia Catedral","county":"Municipio Libertador","state_district":"Caracas",
				"state":"Distrito Capital","postcode":"1010","country":"Venezuela","country_code":"ve"}}});

		expect(app.$.detailPanel.getGeotagging()).toEqual(false);
		
		function checkUpdate(inSender, inEvent) {
			expect(inEvent.key).toEqual(noteKey);
			expect(inEvent.item.key).toEqual(noteKey);
			expect(inEvent.item.subject).toEqual('<br data="geotag"><br>Miraflores, Av. Urdaneta, Parroquia Catedral, Municipio Libertador, Caracas, Distrito Capital');
			expect(inEvent.item.sortKey).toBe(noteKey);   // created by tagNote

			expect(Object.keys(app.keysToGeotag).length).toEqual(0);

			done();
		}
	});
	
	
	it('should update DB & call backupToFile when pref UI is toggled on', function (done) {
		var test = this.test, putPrefSpy, backupToFileStub;
		putPrefSpy = spyOn(app.$.noteService, 'putPref').and.callThrough();
		app.$.noteService.putPref('backupNightly', false).response(simulateUiChange);
		function simulateUiChange(inSender, inResponse) {
			backupToFileStub = spyOn(app.$.noteService, 'backupToFile').and.returnValue(new enyo.Async());
			
			app.$.prefHelpPanel.$.backupNightlyTgl.bubble('onChange', {value: true});
			setTimeout(waitForPut, 100);
		}
		function waitForPut() {
			if (putPrefSpy.calls.count() > 1) {
				expect(putPrefSpy).toHaveBeenCalledWith('backupNightly', true);
				app.$.noteService.getPref('backupNightly').response(checkPref);
			} else if (true /*! test.timedOut*/) {
				setTimeout(waitForPut, 100);
			}
		}
		function checkPref(inSender, inResponse) {
			expect(inResponse).toEqual(true);
			
			expect(backupToFileStub.calls.count()).toEqual(1);
			expect(backupToFileStub.calls.argsFor(0).length).toBe(0);

			done();
		}
	});


	it('should show banner when showMessage is sent to Signals', function (done) {
		var noteNotesStub = spyOn(app, 'noteNotes').and.callFake(cleanup);

		app.renderInto(div);		
		expect(app.$.banner.getShowing()).toEqual(false);		
		enyo.Signals.send('onShowMessage', {message: 'Foo is the <strong>new bar</strong>.'});
		if ("PalmSystem" in window) {
			pending("check for banner: Foo is the new bar.");
		} else {
			expect(app.$.banner.$.paragraph.getContent()).toEqual('Foo is the <strong>new bar</strong>.');
		}
		
		// clean up goes smoothly only if noteService is fully set up
		function cleanup(inSender, inEvent) {
			done();
		}
	});
	
	it('should set up DetailPanel for a new note when the add button is tapped', function (done) {
		app.$.noteService.putPref('geotagOnCreate', false).response(mainTest);

		app.renderInto(div);

		var setDetailsItemSpy = spyOn(app, 'setDetailsItem').and.callThrough();
		var setItemSpy = spyOn(app.$.detailPanel, 'setItem');
		var putNoteSpy = spyOn(app.$.noteService, 'putNote').and.returnValue(new enyo.Async());

		function mainTest(inSender, inEvent) {
			app.$.listPane.$.addBtn.bubble('ontap');
			spyOn(app.$.listPane, 'insert').and.callFake(checkCalls);
		}

		function checkCalls() {
			expect(setDetailsItemSpy.calls.count()).toEqual(1);
			expect(setItemSpy.calls.count()).toBe(1);
			expect(setItemSpy.calls.argsFor(0)[0].subject).toEqual('');
			expect(setItemSpy.calls.argsFor(0)[0].key).toBeTruthy();
			expect(setItemSpy.calls.argsFor(0)[1]).toEqual(true);

			expect(putNoteSpy.calls.count()).toEqual(0);

			done();
		}
	});

	it('should create new notes with the current search text', function (done) {
		app.$.noteService.putPref('geotagOnCreate', false).response(mainTest);

		app.renderInto(div);

		var setDetailsItemSpy = spyOn(app, 'setDetailsItem').and.callThrough();
		var setItemSpy = spyOn(app.$.detailPanel, 'setItem');
		var putNoteSpy = spyOn(app.$.noteService, 'putNote').and.returnValue(new enyo.Async());

		function mainTest(inSender, inEvent) {
			app.$.listPane.$.input.set('value', "asparagus");
			app.$.listPane.$.addBtn.bubble('ontap');
			spyOn(app.$.listPane, 'insert').and.callFake(checkCalls);
		}

		function checkCalls() {
			expect(setDetailsItemSpy.calls.count()).toEqual(1);
			expect(setItemSpy.calls.count()).toBe(1);
			expect(setItemSpy.calls.argsFor(0)[0].subject).toEqual('<br><br>asparagus');
			expect(setItemSpy.calls.argsFor(0)[0].key).toBeTruthy();
			expect(setItemSpy.calls.argsFor(0)[1]).toEqual(true);

			expect(putNoteSpy.calls.count()).toEqual(1);

			done();
		}
	});

	it('should create new notes with geotags, when enabled', function (done) {
		app.renderInto(div);
		var geolocationStub = spyOn(navigator.geolocation, 'getCurrentPosition').and.callFake( function (success, error, options) {
			success({coords: {latitude: 34.56, longitude: -89.01}});
		});
		var openStreetMapStub = spyOn(app.$.openStreetMap, 'send').and.callFake(function () {
			setTimeout(function () {
				app.$.openStreetMap.bubble('onResponse', {
					data: {
						"address": { "road": "County Road 82", "hamlet": "Concord", "county": "Union County",
							"state": "Mississippi", "postcode": "38652",
							"country": "United States of America", "country_code": "us"
						}
					}
				});
			}, 0);
		});
		var setDetailsItemSpy = spyOn(app, 'setDetailsItem').and.callThrough();
		var setItemSpy = spyOn(app.$.detailPanel, 'setItem');
		var putNoteSpy = spyOn(app.$.noteService, 'putNote').and.returnValue(new enyo.Async());

		app.$.noteService.putPref('geotagOnCreate', true).response(mainTest);

		function mainTest(inSender, inEvent) {
			app.$.listPane.$.input.set('value', "");

			app.$.listPane.$.addBtn.bubble('ontap');

			spyOn(app, 'updateDbItem').and.callFake(checkUpdate);
		}

		function checkUpdate(inSender, inEvent) {
			expect(setDetailsItemSpy.calls.count()).toEqual(1);
			expect(setItemSpy.calls.count()).toBe(1);
			expect(setItemSpy.calls.argsFor(0)[0].subject).toEqual("");
			expect(setItemSpy.calls.argsFor(0)[0].key).toBeTruthy();
			expect(setItemSpy.calls.argsFor(0)[1]).toEqual(true);

			expect(putNoteSpy.calls.count()).toEqual(0);
			expect(inEvent.key).toBeTruthy();
			expect(inEvent.item.key).toBeTruthy();
			expect(inEvent.item.subject).toEqual('<br data="geotag"><br>County Road 82, Concord, Union County, Mississippi');
			expect(inEvent.item.sortKey).toBeLessThan(0);   // automatically assigned

			expect(Object.keys(app.keysToGeotag).length).toEqual(0);

			done();
		}
	});

	it('should create new notes with the current search text & geotags', function (done) {
		app.renderInto(div);
		var geolocationStub = spyOn(navigator.geolocation, 'getCurrentPosition').and.callFake( function (success, error, options) {
			success({coords: {latitude: 34.56, longitude: -89.01}});
		});
		var openStreetMapStub = spyOn(app.$.openStreetMap, 'send').and.callFake(function () {
			setTimeout(function () {
				app.$.openStreetMap.bubble('onResponse', {
					data: {
						"address": { "road": "County Road 82", "hamlet": "Concord", "county": "Union County",
							"state": "Mississippi", "postcode": "38652",
							"country": "United States of America", "country_code": "us"
						}
					}
				});
			}, 0);
		});
		var setDetailsItemSpy = spyOn(app, 'setDetailsItem').and.callThrough();
		var setItemSpy = spyOn(app.$.detailPanel, 'setItem');
		var putNoteSpy = spyOn(app.$.noteService, 'putNote').and.returnValue(new enyo.Async());

		app.$.noteService.putPref('geotagOnCreate', true).response(mainTest);

		function mainTest(inSender, inEvent) {
			app.$.listPane.$.input.set('value', "broccoli");

			app.$.listPane.$.addBtn.bubble('ontap');

			spyOn(app, 'updateDbItem').and.callFake(checkUpdate);
		}

		function checkUpdate(inSender, inEvent) {
			expect(setDetailsItemSpy.calls.count()).toEqual(1);
			expect(setItemSpy.calls.count()).toBe(1);
			expect(setItemSpy.calls.argsFor(0)[0].subject).toEqual('');
			expect(setItemSpy.calls.argsFor(0)[0].key).toBeTruthy();
			expect(setItemSpy.calls.argsFor(0)[1]).toEqual(true);

			expect(putNoteSpy.calls.count()).toEqual(0);

			expect(inEvent.key).toBeTruthy();
			expect(inEvent.item.key).toBeTruthy();
			expect(inEvent.item.subject).toEqual('<br data="geotag"><br>broccoli<br><br>County Road 82, Concord, Union County, Mississippi');
			expect(inEvent.item.sortKey).toBeLessThan(0);   // automatically assigned

			expect(Object.keys(app.keysToGeotag).length).toEqual(0);

			done();
		}
	});


	it("should display the first panel when left-arrow is typed outside text fields", function () {
		spyOn(enyo.dom, 'getWindowWidth').and.returnValue(320);
		app.renderInto(div);
		app.$.twoPanels.setIndex(1);
		var indexSpy = spyOn(app.$.twoPanels, 'setIndex');

		enyo.Signals.send('onkeydown', {keyCode: 37, target: {tagName: 'BODY', contentEditable: 'undefined'}});
		expect(indexSpy).toHaveBeenCalledWith(0);
	});

	it("should not change panels when left-arrow is typed in text fields", function () {
		spyOn(enyo.dom, 'getWindowWidth').and.returnValue(320);
		app.renderInto(div);
		app.$.twoPanels.setIndex(1);
		var indexSpy = spyOn(app.$.twoPanels, 'setIndex');

		enyo.Signals.send('onkeydown', {keyCode: 37, target: {tagName: 'INPUT', contentEditable: 'undefined'}});
		expect(indexSpy).not.toHaveBeenCalled();
	});

	it("should display the second panel when right-arrow is typed outside text fields", function () {
		spyOn(enyo.dom, 'getWindowWidth').and.returnValue(320);
		app.renderInto(div);
		app.$.twoPanels.setIndex(0);
		var indexSpy = spyOn(app.$.twoPanels, 'setIndex');

		enyo.Signals.send('onkeydown', {keyCode: 39, target: {tagName: 'BODY', contentEditable: 'undefined'}});
		expect(indexSpy).toHaveBeenCalledWith(1);
	});

	it("should not change panels when right-arrow is typed in text fields", function () {
		spyOn(enyo.dom, 'getWindowWidth').and.returnValue(320);
		app.renderInto(div);
		app.$.twoPanels.setIndex(0);
		var indexSpy = spyOn(app.$.twoPanels, 'setIndex');

		enyo.Signals.send('onkeydown', {keyCode: 39, target: {tagName: 'INPUT', contentEditable: 'undefined'}});
		expect(indexSpy).not.toHaveBeenCalled();
	});

	it('should call createNewNote once when four returns in a row are typed', function () {
		var markCheckpointSpy = spyOn(app.$.detailPanel, 'markCheckpoint');
		var revertToCheckpointSpy = spyOn(app.$.detailPanel, 'revertToCheckpoint');
		var newNoteSpy = spyOn(app, 'createNewNote');

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);
		expect(newNoteSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);
		expect(newNoteSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);
		expect(newNoteSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(1);
		expect(newNoteSpy.calls.count()).toEqual(1);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(2);
		expect(revertToCheckpointSpy.calls.count()).toEqual(1);
		expect(newNoteSpy.calls.count()).toEqual(1);
	});
	it('should not call createNewNote when two returns, a space and two more returns are typed', function () {
		var markCheckpointSpy = spyOn(app.$.detailPanel, 'markCheckpoint');
		var revertToCheckpointSpy = spyOn(app.$.detailPanel, 'revertToCheckpoint');
		var newNoteSpy = spyOn(app, 'createNewNote');

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 32, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(2);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(2);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);
		expect(newNoteSpy.calls.count()).toEqual(0);
	});
	it('should not call createNewNote when two returns, a non-keyboard change and two more returns are typed', function () {
		var markCheckpointSpy = spyOn(app.$.detailPanel, 'markCheckpoint');
		var revertToCheckpointSpy = spyOn(app.$.detailPanel, 'revertToCheckpoint');
		var newNoteSpy = spyOn(app, 'createNewNote');

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		app.$.detailPanel.bubble('onNonkeyboardActivity', {});
		expect(markCheckpointSpy.calls.count()).toEqual(1);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(2);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);

		enyo.Signals.send('onkeydown', {keyCode: 13, target: {tagName: 'DIV', contentEditable: 'true'}, preventDefault: enyo.nop});
		expect(markCheckpointSpy.calls.count()).toEqual(2);
		expect(revertToCheckpointSpy.calls.count()).toEqual(0);
		expect(newNoteSpy.calls.count()).toEqual(0);
	});
});

describe("clock-dependent functionality", function () {
	var div, app, timerCallback;

	beforeEach(function() {
		try {
			app = new App({dbName: 'TestAppNoteDatabase'});

			jasmine.clock().install();
		} catch (err) {
			console.error('clock-dependent beforeEach:', err);
			throw err;
		}
	});

	afterEach(function(done) {
		try {
			jasmine.clock().uninstall();
			app.destroy();
			setTimeout(done, 1000);
		} catch (err) {
			console.error('clock-dependent afterEach:', err);
			throw err;
		}
	});

	it("should not update the list when it's the same day", function (/*done*/) {
		var fetchNotesSpy = spyOn(app.$.noteService, 'fetchNotes');
		spyOn(app.$.listPane, 'populate');
		var startDate = new Date(2016, 7, 19, 12, 0);
		jasmine.clock().mockDate(startDate);
		enyo.hidden = false;

		app.noteNotes(null, {});

		expect(app.lastListRefresh).toEqual(startDate);

		jasmine.clock().tick(12*60*60*1000-1);
		app.checkNewDay();

		expect(fetchNotesSpy.calls.count()).toEqual(0);
	});

	it("should update the list when it's the next day", function (/*done*/) {
		var fetchNotesSpy = spyOn(app.$.noteService, 'fetchNotes');
		spyOn(app.$.listPane, 'populate');
		var startDate = new Date(2016, 7, 19, 12, 0);
		jasmine.clock().mockDate(startDate);
		enyo.hidden = false;

		app.noteNotes(null, {});
		
		expect(app.lastListRefresh).toEqual(startDate);

		jasmine.clock().tick(12*60*60*1000+1);
		app.checkNewDay();

		expect(fetchNotesSpy.calls.count()).toEqual(1);
	});

});
