// Jasmine 2.2 tests to verify that IndexedDB or a shim exists and supports the expected API
// Copyright © 2015 Hominid Software

window.indexedDB = window.indexedDB || window.webkitIndexedDB || window.msIndexedDB;
window.IDBTransaction = window.IDBTransaction || window.webkitIDBTransaction || window.msIDBTransaction;
window.IDBKeyRange = window.IDBKeyRange || window.webkitIDBKeyRange || window.msIDBKeyRange;


describe("IndexedDB", function () {
	var originalTimeout;

	beforeEach(function () {
		originalTimeout = jasmine.DEFAULT_TIMEOUT_INTERVAL;
	    jasmine.DEFAULT_TIMEOUT_INTERVAL = 20000;
	    
		window.indexedDB.deleteDatabase("MyTestDatabase");   // API ensures this completes before other ops
	});
	
	afterEach(function () {
		jasmine.DEFAULT_TIMEOUT_INTERVAL = originalTimeout;
	});
		
	
	it("should support .onupgradeneeded", function (done) {
		var openRequest = window.indexedDB.open("MyTestDatabase", 1);
		openRequest.onerror = function(event) { console.error(event); };

		var upgradeNeededCalled = false;
		openRequest.onupgradeneeded = function (event) {
			upgradeNeededCalled = true;
		};

		var db;
		openRequest.onsuccess = function(event) {
			db = openRequest.result;
			db.onerror = function(event) {
				// Generic error handler for all errors targeted at this database's requests!
				window.alert("Database error: " + event.target.errorCode);
			};

			expect(upgradeNeededCalled).toBeTruthy();   // we have to assert something
			
			expect(db.close).toBeTruthy();			
			db.close();   // no callback, alas
			setTimeout(done, 1000);
		};
	});
	
	
	it("should allow date keys on objectStores", function (done) {
		var openRequest = window.indexedDB.open("MyTestDatabase", 1);
		openRequest.onerror = function(event) { console.error(event); };

		openRequest.onupgradeneeded = function (event) {
			var db = event.target.result;
			var objectStore = db.createObjectStore("note", {keyPath: "dateModified"});
		};

		var errors = [];
		openRequest.onsuccess = function(event) {
			var db = openRequest.result;
			db.onerror = function(event) {   // Generic error handler for all errors targeted at this database's requests!
				console.error(event.target);
				errors.push(event.target.webkitErrorMessage || event.target.error.name || event.target.error || event.target.errorCode);
				event.preventDefault();
			};

			var transaction = db.transaction('note', "readwrite");
			transaction.onerror = function(event) {
				console.error(event.target.error);
				errors.push(event.target.webkitErrorMessage || event.target.error.name || event.target.error || event.target.errorCode);
				db.close();
			};
 
			var objectStore = transaction.objectStore("note");

			var today = new Date();
			var tomorrow = new Date(today.valueOf()+86400000);
			var yesterday = new Date(today.valueOf()-86400000);
			var noteData = [
				{dateModified: today, subject: "some kind of first note"},
				{dateModified: tomorrow, subject: "note modified later"},
				{dateModified: yesterday, subject: "note modified earlier"}
			];
			noteData.forEach( function (note) {
				objectStore.add(note);
			});
			
			var j=0;
			objectStore.openCursor().onsuccess = function(event) {
				var cursor = event.target.result;
				if (cursor) {
					switch (j++) {
						case 0: 
							expect(cursor.key).toEqual(yesterday);
							break;
						case 1: 
							expect(cursor.key).toEqual(today);
							break;
						case 2: 
							expect(cursor.key).toEqual(tomorrow);
							break;
					}
					cursor.advance(1);
				}
			};
						
			transaction.oncomplete = function(event) {			
				db.close();   // no callback, alas
				expect(errors).toEqual([]);
				
				setTimeout(done, 1000);
			};
		};		
	});
	
	
	it("should allow indexing on a single word stem", function (done) {
		var openRequest = window.indexedDB.open("MyTestDatabase", 1);
		openRequest.onerror = function(event) { console.error(event); };

		openRequest.onupgradeneeded = function (event) {
			var db = event.target.result;
			var objectStore = db.createObjectStore("note", {autoIncrement: true});
			objectStore.createIndex("words", 'wordArr', {unique: false, multiEntry: true});
		};

		var errors = [];
		openRequest.onsuccess = function(event) {
			var db = openRequest.result;
			db.onerror = function(event) {   // Generic error handler for all errors targeted at this database's requests!
				console.error(event.target);
				errors.push(event.target.webkitErrorMessage || event.target.error.name || event.target.error || event.target.errorCode);
				event.preventDefault();
			};

			var transaction = db.transaction('note', "readwrite");
			transaction.onerror = function(event) {
				console.error(event.target.error);
				errors.push(event.target.webkitErrorMessage || event.target.error.name || event.target.error || event.target.errorCode);
				db.close();
			};
 
			var objectStore = transaction.objectStore("note");

			var noteData = [
				{subject: "runs with scissors", wordArr: ['RUNS', 'WITH', 'SCISSORS'] },
				{subject: "Scissors, scissors, everywhere", wordArr: ['SCISSORS', 'EVERYWHERE']},
				{subject: "everywhere I go", wordArr: ['EVERYWHERE', 'I', 'GO']},
				{subject: "go run", wordArr: ['GO', 'RUN']}
			];
			noteData.forEach( function (note) {
				objectStore.add(note);
			});
						
			var j=0;
			var index = objectStore.index('words');
			expect(index).toBeTruthy();
			index.openCursor().onsuccess = function(event) {
				var cursor = event.target.result;
				if (cursor) {
					switch (j++) {
						case 0: 
							expect(cursor.key).toEqual('EVERYWHERE');
							expect(cursor.primaryKey).toEqual(2);
							break;
						case 1: 
							expect(cursor.key).toEqual('EVERYWHERE');
							expect(cursor.primaryKey).toEqual(3);
							break;
						case 2: 
							expect(cursor.key).toEqual('GO');
							expect(cursor.primaryKey).toEqual(3);
							break;
					}
					cursor.advance(1);
				} else {
					expect(j).toEqual(10);
				}
			};
						
			var k=0;
			index = objectStore.index('words');
			expect(index).toBeTruthy();
			index.openCursor('RUN').onsuccess = function(event) {
				var cursor = event.target.result;
				if (cursor) {
					switch (k++) {
						case 0: 
							expect(cursor.key).toEqual('RUN');
							expect(cursor.primaryKey).toEqual(4);
							expect(cursor.value.subject).toEqual('go run');
							break;
					}
					cursor.advance(1);
				} else {
					expect(k).toEqual(1);
				}
			};
						
			var m=0;
			index = objectStore.index('words');
			expect(index).toBeTruthy();
			var range = window.IDBKeyRange.bound("RUN", "RUO", false, true);
			index.openCursor(range).onsuccess = function(event) {
				var cursor = event.target.result;
				if (cursor) {
					switch (m++) {
						case 0: 
							expect(cursor.key).toEqual('RUN');
							expect(cursor.primaryKey).toEqual(4);
							expect(cursor.value.subject).toEqual('go run');
							break;
						case 1: 
							expect(cursor.key).toEqual('RUNS');
							expect(cursor.primaryKey).toEqual(1);
							expect(cursor.value.subject).toEqual('runs with scissors');
							break;
					}
					cursor.advance(1);
				} else {
					expect(m).toEqual(2);
				}
			};
			
			transaction.oncomplete = function(event) {			
				db.close();
				expect(errors).toEqual([]);

				setTimeout(done, 1000);
			};
		};
	});
	
	
//	it("should allow indexing on two word stems", function (done) {
//		var openRequest = window.indexedDB.open("MyTestDatabase", 1);
//		openRequest.onerror = function(event) { console.error(event); };
//
//		openRequest.onupgradeneeded = function (event) {
//			var db = event.target.result;
//			var objectStore = db.createObjectStore("note", {autoIncrement: true});
//			objectStore.createIndex("wordPairs", ['wordArr', 'wordArr'], {unique: false, multiEntry: true});
//		};
//
//	});
});
