/* unit tests for Serene Notes util.js */
/*jsl:import ../source/util.js */

describe("errorToMessage function", function () {
	it("should return the message field, if it exists", function () {
		var error = new Error("foo");
		var message = errorToMessage(error, "fallback");
		expect(message).toEqual("foo");
	});

	it("should return the name field, if it exists and message doesn't", function () {
		var error = new URIError();
		var message = errorToMessage(error, "fallback");
		expect(message).toEqual('URIError');
	});

	it("should return the name of the error code, if the message & name fields don't exist", function () {
		var error = {code: 5, NOT_FOUND_ERR: 1, ABORT_ERR: 3, ENCODING_ERR: 5, TYPE_MISMATCH_ERR: 11};
		var message = errorToMessage(error, "fallback");
		expect(message).toEqual('ENCODING_ERR');
	});

	it("should return the passed-in string, if message, name & code fields aren't present", function () {
		var error = {};
		var message = errorToMessage(error, "error while fooing the bar");
		expect(message).toEqual("error while fooing the bar");
	});

	it("should return a string, if message, name & code fields aren't present, and no fallback string is passed", function () {
		var error = {};
		var message = errorToMessage(error);
		expect(typeof message).toEqual(('string'));
		expect(message.length).toBeGreaterThan(0);
	});

	it("should return the passed-in string, if non-object is passed in", function () {
		var error = null;
		var message = errorToMessage(error, "error while barring the foo");
		expect(message).toEqual("error while barring the foo");

		error = undefined;
		message = errorToMessage(error, "error while spamming the bar");
		expect(message).toEqual("error while spamming the bar");
	});

	it("should return a string, if a falsy non-object is passed in", function () {
		var error = null;
		var message = errorToMessage(error);
		expect(typeof message).toEqual(('string'));
		expect(message.length).toBeGreaterThan(0);

		error = undefined;
		message = errorToMessage(error);
		expect(typeof message).toBe(('string'));
		expect(message.length).toBeGreaterThan(1);
	});

	it("should return the string, if a string is passed in", function () {
		var error = "Hey!";
		var message = errorToMessage(error);
		expect(message).toEqual(error);
	});
});


// The global array daysOfWeek is defined by util.js

describe('humanDate utility function', function () {
	var languageCode = (navigator.language || navigator.userLanguage).slice(0,2).toLowerCase();
	
	/* test bug in ilib 5.0 */
	it("should calculate day-of-week in local time zone", function () {
        var dayOfWeekFmt = new ilib.DateFmt({template: "EEEE Z", timezone: 'America/New_York'});
		var date = new Date('2014-05-13T03:17:48.674Z');
		switch (languageCode) {
			case 'en':
				expect(dayOfWeekFmt.format(date)).toEqual("Monday UTC-0400");
				break;
		}
	});

	it("should convert 12 a.m. today into 'Today (<day-of-week>)'", function () {
		var today = new Date();
		today.setHours(0, 0, 0, 0);
		switch (languageCode) {
			case 'en':
				expect(humanDate(today)).toEqual('Today (' + daysOfWeek[today.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(today)).toEqual('Heute (' + daysOfWeek[today.getDay()] + ')');
				break;
		}
	});
	
	it("should convert 11:59 p.m. today into 'Today (<day-of-week>)'", function () {
		var today = new Date();
		today.setHours(23, 59, 59, 999);
		switch (languageCode) {
			case 'en':
				expect(humanDate(today)).toEqual('Today (' + daysOfWeek[today.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(today)).toEqual('Heute (' + daysOfWeek[today.getDay()] + ')');
				break;
		}
	});


	it("should convert 11:59 p.m. yesterday into 'Yesterday (<day-of-week>)'", function () {
		var yesterday = new Date(Date.now() - 86400000);
		yesterday.setHours(23, 59, 59, 999);
		switch (languageCode) {
			case 'en':
				expect(humanDate(yesterday)).toEqual('Yesterday (' + daysOfWeek[yesterday.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(yesterday)).toEqual('Gestern (' + daysOfWeek[yesterday.getDay()] + ')');
				break;
		}
	});
	
	it('should format 12am yesterday as "Yesterday (<day-of-week>)"', function () {
		var yesterday = new Date(Date.now() - 86400000);
		yesterday.setHours(0, 0, 0, 0);
		switch (languageCode) {
			case 'en':
				expect(humanDate(yesterday)).toEqual('Yesterday (' + daysOfWeek[yesterday.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(yesterday)).toEqual('Gestern (' + daysOfWeek[yesterday.getDay()] + ')');
				break;
		}
	});


	it('should format 11:59 p.m. two days ago as "Day before yesterday (<day-of-week>)"', function () {
		var twoDaysAgo = new Date(Date.now() - 2*86400000);
		twoDaysAgo.setHours(23, 59, 59, 999);
		var twoDaysAgoDayOfWeek = daysOfWeek[twoDaysAgo.getDay()];
		switch (languageCode) {
			case 'en':
				expect(humanDate(twoDaysAgo)).toEqual('Day before yesterday (' + twoDaysAgoDayOfWeek + ')');
				break;
			case 'de':
				expect(humanDate(twoDaysAgo)).toEqual('Vorgestern (' + twoDaysAgoDayOfWeek + ')');
				break;
		}
	});

	it('should format 12am two days ago as "Day before yesterday (<day-of-week>)"', function () {
		var twoDaysAgo = new Date(Date.now() - 2*86400000);
		twoDaysAgo.setHours(0, 0, 0, 0);
		var twoDaysAgoDayOfWeek = daysOfWeek[twoDaysAgo.getDay()];
		switch (languageCode) {
			case 'en':
				expect(humanDate(twoDaysAgo)).toEqual('Day before yesterday (' + twoDaysAgoDayOfWeek + ')');
				break;
			case 'de':
				expect(humanDate(twoDaysAgo)).toEqual('Vorgestern (' + twoDaysAgoDayOfWeek + ')');
				break;
		}
	});


	it('should format three days ago as a day-of-week', function () {
		var threeDaysAgo = new Date(Date.now() - 3*86400000);
		threeDaysAgo.setHours(23, 59, 59, 999);
		var threeDaysAgoDayOfWeek = daysOfWeek[threeDaysAgo.getDay()];
		expect(humanDate(threeDaysAgo)).toEqual(threeDaysAgoDayOfWeek);
	});

	it('should format six days ago as a day-of-week', function () {
		var sixDaysAgo = new Date(Date.now() - 6*86400000);
		sixDaysAgo.setHours(0, 0, 0, 0);   // midnight beginning
		var sixDaysAgoDayOfWeek = daysOfWeek[sixDaysAgo.getDay()];
		expect(humanDate(sixDaysAgo)).toEqual(sixDaysAgoDayOfWeek);
	});


	it('should format a week ago as "Last week"', function () {
		var weekAgo = new Date(Date.now() - 7*86400000);
		weekAgo.setHours(23, 59, 59, 999);
		var dateStr = humanDate(weekAgo);
		expect(dateStr).toEqual($L('Last week'));
	});


	var monthStart = new Date();
	monthStart.setDate(1);
	monthStart.setHours(0, 0, 0, 0);

	var twoWeeksAgo = new Date(Date.now() - 14*86400000);
	twoWeeksAgo.setHours(23, 59, 59, 999);

	if (twoWeeksAgo < monthStart) {
		it('should format two weeks ago as month and year', function () {
			var dateStr = humanDate(twoWeeksAgo);
			switch (languageCode) {
				case 'en':
					expect(dateStr).toMatch(new RegExp($L('^(January|February|March|April|May|June|July|August|September|October|November|December)\\s\\d\\d\\d\\d$')));
					break;
				default:   // begins or ends with year
					expect(dateStr).toMatch(/\s\d\d\d\d$|^\d\d\d\d\s/);
			}
		});
	} else {
		it('should format two weeks ago as "Earlier this month"', function () {
			var dateStr = humanDate(twoWeeksAgo);
			expect(dateStr).toEqual($L('Earlier this month'));
		});
	}


	it('should format June 6, 2009 as "June 2009"', function () {
		var dateStr = humanDate(new Date('June 6, 2009'));
		switch (languageCode) {
			case 'en':
				expect(dateStr).toEqual($L('June 2009'));
				break;
			case 'de':
				expect(dateStr).toEqual($L('Juni 2009'));
				break;
			default:   // begins or ends with year
				expect(dateStr).toMatch(/\s2009$|^2009\s/);
		}
	});


	it("should convert 12 a.m. tomorrow into 'Tomorrow (<day-of-week>)'", function () {
		var tomorrow = new Date(Date.now() + 86400000);
		tomorrow.setHours(0, 0, 0, 0);
		switch (languageCode) {
			case 'en':
				expect(humanDate(tomorrow)).toEqual('Tomorrow (' + daysOfWeek[tomorrow.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(tomorrow)).toEqual('Morgen (' + daysOfWeek[tomorrow.getDay()] + ')');
				break;
		}
	});

	it("should convert 11:59 p.m. tomorrow into 'Tomorrow (<day-of-week>)'", function () {
		var tomorrow = new Date(Date.now() + 86400000);
		tomorrow.setHours(23, 59, 59, 999);
		switch (languageCode) {
			case 'en':
				expect(humanDate(tomorrow)).toEqual('Tomorrow (' + daysOfWeek[tomorrow.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(tomorrow)).toEqual('Morgen (' + daysOfWeek[tomorrow.getDay()] + ')');
				break;
		}
	});


	it('should format two days ahead as "Day after tomorrow (<day-of-week>)"', function () {
		var twoDaysAhead = new Date(Date.now() + 2*86400000);
		twoDaysAhead.setHours(0, 0, 0, 0);   // midnight beginning
		switch (languageCode) {
			case 'en':
				expect(humanDate(twoDaysAhead)).toEqual('Day after tomorrow (' + daysOfWeek[twoDaysAhead.getDay()] + ')');
				break;
			case 'de':
				expect(humanDate(twoDaysAhead)).toEqual('Übermorgen (' + daysOfWeek[twoDaysAhead.getDay()] + ')');
				break;
		}
	});
	

	it('should format three days ahead as "This <day of week>"', function () {
		var threeDaysAhead = new Date(Date.now() + 3*86400000);
		threeDaysAhead.setHours(0, 0, 0, 0);   // midnight beginning
		switch (languageCode) {
			case 'en':
				expect(humanDate(threeDaysAhead)).toEqual('This ' + daysOfWeek[threeDaysAhead.getDay()]);
				break;
			case 'de':
				expect(humanDate(threeDaysAhead)).toEqual('Nächsten ' + daysOfWeek[threeDaysAhead.getDay()]);
				break;
		}
	});

	it('should format six days ahead as as "This <day of week>"', function () {
		var sixDaysAhead = new Date(Date.now() + 6*86400000);
		sixDaysAhead.setHours(23, 59, 59, 999);
		switch (languageCode) {
			case 'en':
				expect(humanDate(sixDaysAhead)).toEqual('This ' + daysOfWeek[sixDaysAhead.getDay()]);
				break;
			case 'de':
				expect(humanDate(sixDaysAhead)).toEqual('Nächsten ' + daysOfWeek[sixDaysAhead.getDay()]);
				break;
		}
	});


	it('should format a week ahead as month and year', function () {
		var weekAhead = new Date(Date.now() + 8*86400000);
		weekAhead.setHours(0, 0, 0, 0);
		var dateStr = humanDate(weekAhead);
		switch (languageCode) {
			case 'en':
				expect(dateStr).toMatch(new RegExp($L('^(January|February|March|April|May|June|July|August|September|October|November|December)\\s\\d\\d\\d\\d$')));
				break;
			default:   // begins or ends with year
				expect(dateStr).toMatch(/\s\d\d\d\d$|^\d\d\d\d\s/);
		}
	});
});


describe("cleanHtml utility function", function () {
	/* TODO: remove test sensitivity to whitespacing */

	it("should preserve less-than and greater-than signs & ampersands", function () {
		expect(cleanHtml('a few < many & 4 > 3')).toEqual('a few < many & 4 > 3');
	});

	it("should replace XML declaration with whitespace", function () {
		var clean = cleanHtml('<?xml version="1.0" encoding="UTF-8" standalone="no" ?>stuff', true);
		expect(clean).toMatch(/[\s\u200B]+stuff/);
	});

	it("should replace DOCTYPE with whitespace", function () {
		var clean = cleanHtml('garbage<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN"\r\n "http://www.w3.org/TR/html4/strict.dtd">stuff',true);
		expect(clean).toMatch(/garbage[\s\u200B]+stuff/);
	});

	it("should replace comments with whitespace", function () {
		var clean = cleanHtml('foo<!--comment<b>bold?</b>-->bar <!-- another --> spam',true);
		expect(clean).toMatch(/foo[\s\u200B]+bar[\s\u200B]+spam/);
	});

	it("should replace head tags & their content with whitespace", function () {
		var clean = cleanHtml('<head><title>Document title</title><script src="javascript.js"></script><link rel="stylesheet" href="B" type="text/css"></head>things',true);
		expect(clean).toMatch(/[\s\u200B]+things/);
	});

	it("should replace script tags & their content with whitespace", function () {
		var clean = cleanHtml('foo <script type="javascript"> badstuff </script> bar',true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

	it("should replace style tags & their content with whitespace", function () {
		var clean = cleanHtml('foo <style type="text/css scoped"> \nbody {color: #eee;} </style> bar',true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

	it("should replace object tags & their content with whitespace", function () {
		var clean = cleanHtml('foo <object data="move.swf" type="application/x-shockwave-flash">\r\n<param name="foo" value="bar">\n</object> bar',true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

	it("should replace applet tags & their content with whitespace", function () {
		var clean = cleanHtml('foo <applet code="game.class" archive="game.zip" >\n<param name="difficulty" value="easy">\n<b>You need Java to play this game.</b>\n</applet> bar',true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

	it("should replace nav tags & their content with whitespace", function () {
		var clean = cleanHtml('foo <nav>\r\n<ul>\n<li><a href="#">Home</a></li>\n</ul>\n</nav> bar', true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

 	it("should replace form tags & their content with whitespace", function () {
		var clean = cleanHtml('foo<form>estimate:<input type="number"></form>bar', true);
		expect(clean).toMatch(/foo[\s\u200B]+bar/);
	});

	it("should replace all other tags with whitespace", function () {
		var clean = cleanHtml('<html>foo <iframe> bar </iframe> spam <input type="file"></html>', true);
		expect(clean).toMatch(/[\s\u200B]+foo[\s\u200B]+bar[\s\u200B]+spam[\s\u200B]+/);
	});

	it("should preserve line break tags", function () {
		expect(cleanHtml('foo <br > bar   <br><br />')).toEqual('foo <br> bar   <br><br>');
	});

	it("should remove all attributes from hr tags", function () {
		expect(cleanHtml('first section<hr>second section')).
			toEqual('first section<hr>second section');
	});

	it("should remove all attributes from p tags", function () {
		expect(cleanHtml('foo <p > bar<span>different</span>   </p>', true)).
			toMatch(/foo <p> bar[\s\u200B]different[\s\u200B]   <\/p>/);
	});

	it("should remove all attributes from div tags", function () {
		expect(cleanHtml('foo<div style="position: absolute;" onclick="badthing">outer <div>inner</div> outer2 </div></div>spam <DIV data-foo > frotz </DIV> frabble')).
			toEqual('foo<div>outer <div>inner</div> outer2 </div></div>spam <DIV> frotz </DIV> frabble');
	});

	it("should remove all attributes from h* tags", function () {
		expect(cleanHtml('foo<h1>title</h1> <h2 class="smaller">subtitle</h2><h3 style="font-size:larger">section</h3> bar<h4>paragraph</h4>')).
			toEqual('foo<h1>title</h1> <h2>subtitle</h2><h3>section</h3> bar<h4>paragraph</h4>');
	});

	it("should remove all attributes from i, b, em & strong tags", function () {
		expect(cleanHtml('foo<i style="font-weight: 900;">italic</i> and <b >bold</b > but <em>emphasized</em> or <strong>strongly emphasized</strong> bar')).
			toEqual('foo<i>italic</i> and <b>bold</b> but <em>emphasized</em> or <strong>strongly emphasized</strong> bar');
	});

	it("should remove all attributes from ol, ul and li tags", function () {
		expect(cleanHtml('foo<ol start="3"><li>first</li><li>second<ul reversed><li>erste</li><li>zwitte</li></ul></li><li>third</li></ol>')).
			toEqual('foo<ol><li>first</li><li>second<ul><li>erste</li><li>zwitte</li></ul></li><li>third</li></ol>');
	});

	it("should remove all attributes from dl, dt & dd tags", function () {
		expect(cleanHtml('<dl compact><dt>tall</dt><dt>altitudinous</dt><dd>not short</dd></dl>')).
			toEqual('<dl><dt>tall</dt><dt>altitudinous</dt><dd>not short</dd></dl>');
	});

	it("should remove all attributes from nl & label tags", function () {
		expect(cleanHtml('<nl><label>Contents </label><li href="#introduction">Introduction</li>' +
			'<li><nl><label>Terms</label><li href="#may">May</li><li href="#must">Must</li><li href="#should">Should</li></nl></li>' +
			'<li href="#conformance">Conformance</li><li href="#references">References</li></nl>')).
			toEqual('<nl><label>Contents </label><li>Introduction</li>' +
			'<li><nl><label>Terms</label><li>May</li><li>Must</li><li>Should</li></nl></li>' +
			'<li>Conformance</li><li>References</li></nl>');
	});

	it("should remove all attributes from table, caption, colgroup, thead, tfoot, tbody, tr, th & td tags", function () {
		var clean = cleanHtml('<table class="foo"><thead><tr><th>column 1</th><th>column 2</th></tr></thead>\n' +
			'<tbody><tr><td>foo</td><td>bar</td></tr></tbody></table>');
		expect(clean).toEqual('<table><thead><tr><th>column 1</th><th>column 2</th></tr></thead> ' +
			'<tbody><tr><td>foo</td><td>bar</td></tr></tbody></table>');
	});

	it("should remove all attributes from pre & code tags", function () {
		var clean = cleanHtml('foo<pre style="white-space: pre-wrap;"><code>// in the browser\'s developer console\n' +
			'console.log(&quot;Hello World!&quot;);</code></pre>bar');
		expect(clean).toEqual('foo<pre><code>// in the browser\'s developer console ' +
			'console.log(&quot;Hello World!&quot;);</code></pre>bar');
	});

	it("should remove all attributes from blockquote tags", function () {
		var clean = cleanHtml('<blockquote cite="http://developer.mozilla.org">\n' +
			'<p>This is a quotation taken from the Mozilla Developer Center.</p>\n</blockquote>');
		expect(clean).toEqual('<blockquote> ' +
			'<p>This is a quotation taken from the Mozilla Developer Center.</p> </blockquote>');
	});
	it("should remove all attributes from q tags", function () {
		expect(cleanHtml('The line is <q style="font-family: \'Trebuchet\'">To be, or not to be?</q>')).
			toEqual('The line is <q>To be, or not to be?</q>');
	});
	it("should remove all attributes from cite tags", function () {
		expect(cleanHtml('More information can be found in <cite dire="ltr">[ISO-0000]</cite>')).
			toEqual('More information can be found in <cite>[ISO-0000]</cite>');
	});
	
	it("should remove all attributes from kbd tags", function () {
		expect(cleanHtml('Type the following in the Run dialog: <kbd class="windows">cmd</kbd>')).
			toEqual('Type the following in the Run dialog: <kbd>cmd</kbd>');
	});

	it("should replace img tags with whitespace", function() {
		expect(cleanHtml('foo<img src="javascript:evil();" onload="evil();" />bar', true)).
			toMatch(/foo[\s\u200B]bar/);
	});
	//it("should remove unapproved attributes from image tags", function() {
	//	expect(cleanHtml('foo<img src="clock-demo-thumb-200.png" onclick="badstuff"\nalt="Clock"\n srcset="clock-demo-thumb-200.png 200w, clock-demo-thumb-400.png 400w" \nsizes="(min-width: 600px) 200px, 50vw">bar')).
	//		toEqual('foo<img src="clock-demo-thumb-200.png" \u200Balt="Clock"\u200B srcset="clock-demo-thumb-200.png 200w, clock-demo-thumb-400.png 400w" \u200Bsizes="(min-width: 600px) 200px, 50vw">bar');
	//})


	it("should not insert whitespace when 'makeSafe' is false", function () {
		expect(cleanHtml('foo <p > bar<span>different</span>spam</p>', false)).
			toEqual('foo <p> bardifferentspam</p>');
	});
});
