﻿// DelMar IT Javascript code library.
var ditScreenWidth;             // Available window width
var ditScreenHeight;            // Available window height
var ditScrollbarWidth;          // Size of scrollbars on platform
var ditDevicePixelRatio;        // For 'retina' displays, this should be 2. Otherwise 1.
var ditRunningFromHomeScreen;   // Indicates if mobile web app was launched from icon added to home screen.
var ditStorage;                 // Alias for storage location that could be switched in future
var ditCssTransitions;          // Detects support for CSS transitions
var ditUsingCordova;            // Detects if we are using Cordova (PhoneGap)
var ditResourceBase;            // Configure the base resource level

// Page transition variables.
var gPageHideTimeout;
var gPageHideAttributeKey = 'data-future-hide';

// Wait page variables.
var gWaitPageVisible;
var gWaitAnimateCounter;
var gWaitAnimateDegrees;
var gWaitAnimateColors;

function ditInit() {

    document.getElementsByTagName('body')[0].addEventListener('pointermove', ditBodyTouchMove, false);
    window.onorientationchange = ditBodyOrientationChange;

    ditScreenWidth = window.innerWidth;
    ditScreenHeight = window.innerHeight;
    ditDevicePixelRatio = window.devicePixelRatio || 1;
    ditStorage = window.localStorage;
    ditScrollbarWidth = ditGetScrollBarWidth();

    // See if we are running from the home screen or inside the mobile browser.

    ditRunningFromHomeScreen = false;
    if (("standalone" in window.navigator) && window.navigator.standalone == true) {
        ditRunningFromHomeScreen = true;
    }

    // Check for CSS transition support

    var div = document.createElement('div');
    div.innerHTML = '<div style="-webkit-transition:color 1s linear;-moz-transition:color 1s linear;"></div>';
    ditCssTransitions = (div.firstChild.style.webkitTransition !== undefined) || (div.firstChild.style.MozTransition !== undefined);

	// Check for Cordova (PhoneGap)
	ditUsingCordova = false;
	ditResourceBase = "";
	if(typeof cordova !== 'undefined'){
		ditUsingCordova = true;
		ditResourceBase = "www/";
	}
}

function ditDelayedFocus(uiObject) {

    // Not sure why we need to delay calling focus but focus doesn't always work if we don't.  
    // Probably something to do with setting focus to an object before it is actually visible.

    setTimeout(delayedFocus, 500);

    function delayedFocus() {
        uiObject.focus();
    }
}

function ditFastClickCanvas(can, drawFunction, clickFunction) {

    var markAsSelectedFunction;

    can.Selected = false;
    can.DrawFunction = drawFunction;
    can.ClickFunction = clickFunction;

    can.addEventListener('pointerdown', clickStart, false);
    can.addEventListener('pointermove', clickMove, false);
    can.addEventListener('pointerup', clickEnd, false);

    drawFunction(can);

    function clickStart(e) {
        var touches = e.getPointerList();
        if(touches.length == 0)
            return;

        this.ClickX = touches[0].clientX;
        this.ClickY = touches[0].clientY;
        this.ClickX -= can.offsetLeft;

        //Mark as selected immediately to resolve strange click problems
        markAsSelected();

        // Don't draw/mark it as selected unless they haven't moved or released after a short period.
//        markAsSelectedFunction = setTimeout(markAsSelected, 10);
    }

    function markAsSelected() {
        markAsSelectedFunction = null;
        can.Selected = true;
        can.DrawFunction(can);
    }

    function clickMove(e) {
        var touches = e.getPointerList();
        if(touches.length == 0) //The move event will fire with no touches when using a mouse
            return;

        if (markAsSelectedFunction != null) {
            clearTimeout(markAsSelectedFunction);
        }

        var clickX = touches[0].clientX;
        var clickY = touches[0].clientY;
        clickX -= this.offsetLeft;

        if (Math.abs(this.ClickX - clickX) > 5 || Math.abs(this.ClickY - clickY) > 5) {
            this.Selected = false;
            this.DrawFunction(this);
        }
    }

    function clickEnd(e) {
        if (markAsSelectedFunction != null) {
            clearTimeout(markAsSelectedFunction);
        }

        if (this.Selected == true) {
            e.originalEvent.stopPropagation();
            e.originalEvent.preventDefault();
            this.Selected = false;
            this.DrawFunction(this);
            this.ClickFunction(this);
        }
    }
}

function ditScrollableDivInit(listBox) {

    listBox.addEventListener('pointerdown', ditListBoxTouchStart, false);
    listBox.addEventListener('pointermove', ditListBoxTouchMove, false);
    listBox.addEventListener('pointerup', ditListBoxTouchEnd, false);
    listBox.addEventListener('scroll', ditListBoxScroll, false);

    listBox.ContentHeight = 0;
    listBox.LastScrollTop = 0;
}

function ditListBoxTouchStart(e) {

    var touches = e.getPointerList();
    if(touches.length == 0) //The move event will fire with no touches when using a mouse
        return;

    // Scrolling.

    var startScrollTop = this.scrollTop;

    if (startScrollTop <= 0) {
        this.scrollTop = 1;
    }
    else if (startScrollTop + this.offsetHeight >= this.scrollHeight) {
        this.scrollTop = this.scrollHeight - this.offsetHeight - 1;
    }

    // Detecting swipes.

    this.TouchMoves = 0;
    this.TouchDownX = touches[0].clientX;
    this.TouchDownY = touches[0].clientY;
}

function ditListBoxTouchMove(e) {

    var touches = e.getPointerList();
    if(touches.length == 0) //The move event will fire with no touches when using a mouse
        return;

    // Scrolling.

    if (this.ContentHeight >= this.offsetHeight) {
        e.stopPropagation();
    }

    // Detecting swipes.

    this.TouchMoves++;
    this.TouchMoveX = touches[0].clientX;
    this.TouchMoveY = touches[0].clientY;
}

function ditListBoxTouchEnd(e) {
    
    // Don't count as a swipe if:
    //  there weren't enough touch moves, or 
    //  the Y value changed too much, or
    //  the X value didn't change enough


    if (this.TouchMoves < 3
    || Math.abs(this.TouchDownY - this.TouchMoveY) > 100
    || Math.abs(this.TouchDownX - this.TouchMoveX) < 50) {
        return;
    }

    // If the end X is > beginning X, then they swiped to the right.

    if (this.TouchMoveX > this.TouchDownX) {
        if (this.OnSwipeRight != null) {
            this.OnSwipeRight();
        }
    }

    // If the end X is <= beginning X, then they swiped to the left.

    else {
        if (this.OnSwipeLeft != null) {
            this.OnSwipeLeft();
        }
    }
}

function ditListBoxScroll(e) {
    this.LastScrollTop = this.scrollTop;
}

function ditWaitShow() {

    window.scrollTo(0, 0);  // Just in case the page got moved from the default position.

    // No need to do anything if the Wait page is already being shown.

    if (gWaitPageVisible == true) {
        return;
    }
    gWaitPageVisible = true;

    gWaitAnimateDegrees = new Array(0, 30, 60, 90, 120, 150, 180, 210, 240, 270, 300, 330);
    gWaitAnimateColors = new Array(
        "#eeeeee", "#dddddd", "#cccccc", "#bbbbbb", "#999999", "#888888",
        "#777777", "#666666", "#555555", "#333333", "#222222", "#050505");

    gWaitAnimateCounter = 0;
    setTimeout(ditWaitAnimate, 1500);
}

function ditWaitAnimate() {

    if (gWaitPageVisible == false) {
        return;
    }

    var page = document.getElementById("ditWaitPage");

    if (page == undefined) {

        page = ditCanvasNew(window.innerWidth, window.innerHeight);

        page.id = "ditWaitPage";
        page.style.opacity = 0.0;
        page.style.position = "absolute";
        page.style.left = "0px";
        page.style.top = "0px";
        page.style.width = window.innerWidth + "px";
        page.style.height = window.innerHeight + "px";

        page.Rect = ditRectNew(0, 0, window.innerWidth, window.innerHeight);

        document.body.appendChild(page);

        setTimeout(
            function () {
                ditApplyCssTransition(page, "opacity 0.4s");
                page.style.opacity = 1.0;
            }
        , 1);
    }

    ditCanvasClearRect(page, page.Rect);
    ditCanvasFillRect(page, page.Rect, "rgba(0, 0, 0, 0.2)");

    var centerX = page.Rect.Width / 2;
    var centerY = page.Rect.Height / 2;

    var innerRadius = 20;
    var outerRadius = 32;

    gWaitAnimateCounter++;

    if (gWaitAnimateCounter == 12) {
        gWaitAnimateCounter = 0;
    }

    for (var i = 0; i < gWaitAnimateDegrees.length; i++) {

        var radians = Math.PI / 180 * gWaitAnimateDegrees[i];

        var x1 = centerX + (innerRadius * Math.cos(radians));
        var y1 = centerY + (innerRadius * Math.sin(radians));

        var x2 = centerX + (outerRadius * Math.cos(radians));
        var y2 = centerY + (outerRadius * Math.sin(radians));

        var lineColor = gWaitAnimateColors[i];
        ditCanvasDrawLine(page, lineColor, 6, x1, y1, x2, y2);
    }

    var tempColor = gWaitAnimateColors.pop();
    gWaitAnimateColors.unshift(tempColor);

    setTimeout(ditWaitAnimate, 90);
}

function ditWaitHide() {
    gWaitPageVisible = false;

    var page = document.getElementById("ditWaitPage");

    if (page != undefined) {
        document.body.removeChild(page);
    }
}

// ditPageShow and ditPageHide are called instead of display = 'block/none' so that we can do page animations.

function ditPageShowSlideLeftRight(page) {
    window.scrollTo(0, 0);  // Just in case the page got moved from the default position.

    ditPageShowWithPreviousCancel(page);
    setTimeout(function(){
            ditApplyCssTransition(page, "left .2s ease-in");
            page.style.left = 0;
        }, 1);
}

function ditPageShowSlideRightLeft(page) {
    window.scrollTo(0, 0);  // Just in case the page got moved from the default position.

    ditPageShowWithPreviousCancel(page);
    setTimeout(function(){
        ditApplyCssTransition(page, "right .2s ease-in");
        page.style.left = 0;
    }, 1);
}

function ditPageShowSlideTop(page) {
    window.scrollTo(0, 0);  // Just in case the page got moved from the default position.

    ditPageShowWithPreviousCancel(page);
    setTimeout(function(){
            ditApplyCssTransition(page, "top .2s ease-in");
            page.style.top = 0;
        }, 1);
}

function ditPageShowWithPreviousCancel(page){
    if(page.hasAttribute(gPageHideAttributeKey)){
        page.removeAttribute(gPageHideAttributeKey);
    }
    page.style.display = "block";
}

function ditPageHide(page){
    if(page.hasAttribute(gPageHideAttributeKey)){
        page.removeAttribute(gPageHideAttributeKey);
    }
    page.style.display = "none";
}

function ditPageHideFutureHidden(){
    var elements = document.querySelectorAll('[' + gPageHideAttributeKey + ']');
    for(var i = 0; i < elements.length; i++){
        ditPageHide(elements[i]);
    }
}

function ditPageHideWithPreviousCancel(page){
	if(gPageHideTimeout){
		clearTimeout(gPageHideTimeout);
	}
    if(!page.hasAttribute(gPageHideAttributeKey)){
        page.setAttribute(gPageHideAttributeKey, true);
    }
	gPageHideTimeout = setTimeout(ditPageHideFutureHidden, 400);
}

function ditPageHideOnLeft(page) {
    setTimeout(function(){
            ditApplyCssTransition(page, "left .2s ease-in");
            page.style.left = (-ditScreenWidth) + "px";
        }, 1);

    ditPageHideWithPreviousCancel(page);
}

function ditPageHideOnRight(page) {
    setTimeout(function(){
            ditApplyCssTransition(page, "right .2s ease-in");
            page.style.left = ditScreenWidth + "px";
        }, 1);

    ditPageHideWithPreviousCancel(page);
}

function ditPageHideOnBottom(page) {
    setTimeout(function(){
            ditApplyCssTransition(page, "top .2s ease-in");
            page.style.top = ditScreenHeight + "px";
        }, 1);

    ditPageHideWithPreviousCancel(page);
}

function ditMessageBoxOK(msg) {

    var divBox = ditMessageBoxInit(msg);

    var btnOK = ditMessageBoxButtonNew("OK");

    divBox.appendChild(btnOK);
}

function ditMessageBoxYesNo(msg, yesCallbackFunction, noCallbackFunction) {

    var divBox = ditMessageBoxInit(msg);

    var btnYes = ditMessageBoxButtonNew("Yes", yesCallbackFunction);
    btnYes.style.color = "#ffffff";
    btnYes.style.backgroundColor = "#548F3F";
    btnYes.style.borderColor = "#3F6B30";
    divBox.appendChild(btnYes);

    var btnNo = ditMessageBoxButtonNew("No", noCallbackFunction);
    btnNo.style.color = "#ffffff";
    btnNo.style.backgroundColor = "#bb2034";
    btnNo.style.borderColor = "#911a2a";
    divBox.appendChild(btnNo);
}

function ditMessageBoxInit(msg) {

    window.scrollTo(0, 0);  // Just in case the page got moved from the default position.
    ditWaitHide();          // Just in case the Wait page is shown which could prevent the message box button(s) from being clicked.

    // Create the div that covers the entire screen.

    var divPage = document.createElement("div");

    divPage.id = "ditMessagePage";
    divPage.style.display = "block";
    divPage.style.opacity = 0.0;
    divPage.style.position = "absolute";
    divPage.style.left = "0px";
    divPage.style.top = "0px";
    divPage.style.width = "100%";
    divPage.style.height = "100%";
    divPage.style.background = "rgba(0, 0, 10, 0.4)";

    divPage.style.touchCallout = "none";
    divPage.style.userSelect = "none";
    divPage.style.webkitTouchCallout = "none";
    divPage.style.webkitUserSelect = "none";

    document.body.appendChild(divPage);

    // Create the div that looks like the message box.

    var boxWidth = 260;
    var boxPadding = 8;

    var divBox = document.createElement("div");

    divBox.id = "ditMessageBox";
    divBox.style.position = "absolute";
    divBox.style.left = (((ditScreenWidth - boxWidth) / 2) - boxPadding) + "px";

    divBox.style.top = "25%";

    divBox.style.width = boxWidth + "px";
    divBox.style.fontFamily = "Helvetica";
    divBox.style.fontSize = "20px";
    divBox.style.color = "#ffffff";
    divBox.style.backgroundColor = "#4b4b4b";
    divBox.style.borderStyle = "solid";
    divBox.style.borderWidth = "2px";
    divBox.style.borderColor = "#ffffff";
    divBox.style.borderRadius = "0px";      // 10px
    divBox.style.textAlign = "center";
    divBox.style.padding = boxPadding + "px";
    divBox.style.paddingTop = "20px";
    divBox.style.overflow = "auto";

    divBox.innerHTML = msg;

    divPage.appendChild(divBox);

    // Show the message box by fading it in.

    setTimeout(
        function () {
            ditApplyCssTransition(divPage, "opacity 0.4s");
            divPage.style.opacity = 1.0;
        }
        , 1);

    return divBox;
}

function ditMessageBoxButtonNew(textValue, onClickCallbackFunction) {

    var btn = document.createElement("button");

    var btnText = document.createTextNode(textValue);
    btn.appendChild(btnText);

    btn.style.width = "100%";
    btn.style.height = "40px";
    btn.style.fontFamily = "Helvetica";
    btn.style.fontSize = "20px";
    btn.style.fontWeight = "bold";
    btn.style.color = "#000000";
    btn.style.backgroundColor = "#c8c8c8";
    btn.style.borderRadius = "0px";     // 8px
    btn.style.borderColor = "#464951";
    btn.style.textAlign = "center";
    btn.style.marginTop = "20px";

    // When the button is clicked...

    btn.addEventListener('click', function(e){
        // Fade-out the page.

        var page = document.getElementById("ditMessagePage");
        ditApplyCssTransition(page, "opacity 0.4s");
        page.style.opacity = 0.0;

        // Call the callback function if one was provided.

        if (onClickCallbackFunction != undefined) {
            onClickCallbackFunction();
        }

        // Remove all related elements from the page.

        setTimeout(
            function () {
                var page = document.getElementById("ditMessagePage");
                document.body.removeChild(page);
            }
            , 400);
    }, false);

    return btn;
}

function ditBodyTouchMove(e) {
    var touches = e.getPointerList();
    if(touches.length == 0) //The move event will fire with no touches when using a mouse
        return;

    e.originalEvent.preventDefault(); // Prevents elastic scrolling.
}

function ditBodyOrientationChange(event) {
    window.scrollTo(0, 0);
}

function ditHttpGetAsync(url, callbackFunction) {

    var httpRequest = new XMLHttpRequest();
    var aborted = false;

    // Abort the request if it hasn't finished in 30 seconds.

    var timeoutFunction = setTimeout(httpGetAsyncTimeout, 30000);

    httpRequest.onreadystatechange = httpStateChange;
    httpRequest.open("GET", url, true);
    httpRequest.send(null);

    function httpGetAsyncTimeout() {
        aborted = true;
        httpRequest.abort();
        ditMessageBoxOK("Timeout connecting with the server.");
    }

    function httpStateChange() {

        // If the request is finished and the response is ready...

        if (httpRequest.readyState == 4) {

            // Stop the abort function from being called.

            clearTimeout(timeoutFunction);

            // Check to see if the server processed the request OK.

            if (httpRequest.status == 200) {
                callbackFunction(httpRequest.responseText);
            }
            else {
                if (aborted == false) {
                    ditMessageBoxOK("Error received from the server.");
                }
            }
        }
    }
}

function ditHttpPostAsync(url, data, callbackFunction) {

    var httpRequest = new XMLHttpRequest();
    var aborted = false;

    // Abort the request if it hasn't finished in 30 seconds.

    var timeoutFunction = setTimeout(httpPostAsyncTimeout, 30000);

    httpRequest.onreadystatechange = httpStateChange;
    httpRequest.open("POST", url, true);
    httpRequest.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
    httpRequest.send(data);

    function httpPostAsyncTimeout() {
        aborted = true;
        httpRequest.abort();
        ditMessageBoxOK("Timeout connecting with the server.");
    }

    function httpStateChange() {

        // If the request is finished and the response is ready...

        if (httpRequest.readyState == 4) {

            // Stop the abort function from being called.

            clearTimeout(timeoutFunction);

            // Check to see if the server processed the request OK.

            if (httpRequest.status == 200) {
                if (httpRequest.responseText.substring(0, 3) != "ok~") {
                    ditMessageBoxOK("Error returned from server: " + httpRequest.responseText.substring(3));
                }
                callbackFunction(httpRequest.responseText);
            }
            else {
                if (aborted == false) {
                    ditMessageBoxOK("Error received from the server.");
                }
            }
        }
    }
}

// The function returns true if the input string can be converted to a number.
// Rules for the input string are:
//  Can contain only number characters
//  Contains at most one dot character to indicate the decimal position
//  If a hyphen is present it must be the first character in the string

function ditTryParse(stringValue) {
    var character;
    var dotFound = false;

    if (stringValue == "") {
        return false;
    }

    for (var i = 0; i < stringValue.length; i++) {

        character = stringValue.substring(i, i + 1);

        if (character == ".") {
            if (dotFound == true) {
                return false;
            }
            dotFound = true;
        }
        else if (character == "-") {
            if (i > 0) {
                return false;
            }
        }
        else if (character < "0" || character > "9") {
            return false;
        }
    }

    // If we get here, the value is OK.

    return true;
}

function ditReplaceSpecialChars(stringValue) {

    stringValue = stringValue.replace(/\n/g, " ");
    stringValue = stringValue.replace(/\r/g, " ");
    stringValue = stringValue.replace(/\t/g, " ");
    stringValue = stringValue.replace(/~/g, " ");     // ~ is used as the field separator for many client/server calls.

    return stringValue;
}

function ditWordCap(stringValue) {

    var words = stringValue.split(" ");

    for (var i = 0; i < words.length; i++) {
        var c = words[i].charAt(0).toUpperCase();
        words[i] = c + words[i].substr(1);
    }

    return words.join(" ");
}

function ditRectNew(left, top, width, height) {

    var rect = {};

    rect.Left = left;
    rect.Top = top;
    rect.Width = width;
    rect.Height = height;

    return rect;
}

// Adjust the rectangle values for high-density displays.

function ditRectAdjustForPixelRatio(rect) {

    var adjustedRect = ditRectNew(
        rect.Left * ditDevicePixelRatio,
        rect.Top * ditDevicePixelRatio,
        rect.Width * ditDevicePixelRatio,
        rect.Height * ditDevicePixelRatio);

    return adjustedRect;
}

function ditCanvasNew(width, height) {

    var canvas = document.createElement("canvas");

    canvas.width = width * ditDevicePixelRatio;
    canvas.height = height * ditDevicePixelRatio;

    canvas.style.width = width + "px";
    canvas.style.height = height + "px";

    canvas.Rect = ditRectNew(0, 0, width, height);

    return canvas;
}

function ditCanvasClearRect(canvas, rect) {

    var canvas2d = canvas.getContext("2d");
    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    canvas2d.clearRect(
        adjustedRect.Left,
        adjustedRect.Top,
        adjustedRect.Width,
        adjustedRect.Height);
}

function ditCanvasDrawLeftArrowButton(canvas, rect, backColor, foreColor, text) {

    var canvas2d = canvas.getContext("2d");

    ditCanvasLeftArrowPath(canvas, rect);

    canvas2d.fillStyle = backColor;
    canvas2d.fill();

    ditCanvasLeftArrowPath(canvas, rect);

    canvas2d.lineWidth = 2 * ditDevicePixelRatio;
    canvas2d.strokeStyle = foreColor;
    canvas2d.stroke();
}

function ditCanvasLeftArrowPath(canvas, rect) {
    
    var radius = 5 * ditDevicePixelRatio;
    var ar = ditRectAdjustForPixelRatio(rect);
    var arrowWidth = 14 * ditDevicePixelRatio;

    var canvas2d = canvas.getContext("2d");

    canvas2d.beginPath();

    canvas2d.moveTo(ar.Left, ar.Top + (ar.Height / 2));
    canvas2d.lineTo(ar.Left + arrowWidth, ar.Top);
    canvas2d.arcTo(ar.Left + arrowWidth, ar.Top, ar.Left + ar.Width, ar.Top, radius);
    canvas2d.arcTo(ar.Left + ar.Width, ar.Top, ar.Left + ar.Width, ar.Top + ar.Height, radius);
    canvas2d.arcTo(ar.Left + ar.Width, ar.Top + ar.Height, ar.Left, ar.Top + ar.Height, radius);
    canvas2d.lineTo(ar.Left + arrowWidth, ar.Top + ar.Height);
    canvas2d.lineTo(ar.Left, ar.Top + (ar.Height / 2));
}

function ditCanvasFillRect(canvas, rect, fillColor) {

    var canvas2d = canvas.getContext("2d");
    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    canvas2d.fillStyle = fillColor;

    canvas2d.fillRect(
        adjustedRect.Left,
        adjustedRect.Top,
        adjustedRect.Width,
        adjustedRect.Height);
}

function ditCanvasFillRectRound(canvas, rect, fillColor, radius) {

    ditCanvasRoundRectPath(canvas, rect, radius);

    var canvas2d = canvas.getContext("2d");
    canvas2d.fillStyle = fillColor;
    canvas2d.fill();
}

function ditCanvasRoundRectPath(canvas, rect, radius) {

    var canvas2d = canvas.getContext("2d");
    var ar = ditRectAdjustForPixelRatio(rect);

    radius = radius * ditDevicePixelRatio;

    canvas2d.beginPath();

    canvas2d.arcTo(ar.Left + ar.Width, ar.Top, ar.Left + ar.Width, ar.Top + ar.Height, radius);
    canvas2d.arcTo(ar.Left + ar.Width, ar.Top + ar.Height, ar.Left, ar.Top + ar.Height, radius);
    canvas2d.arcTo(ar.Left, ar.Top + ar.Height, ar.Left, ar.Top, radius);
    canvas2d.arcTo(ar.Left, ar.Top, ar.Left + ar.Width, ar.Top, radius);
}

function ditCanvasDrawImage(canvas, image, rect) {

    var canvas2d = canvas.getContext("2d");
    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    canvas2d.drawImage(image, adjustedRect.Left, adjustedRect.Top, adjustedRect.Width, adjustedRect.Height);
}

function ditCanvasDrawTextLeftTop(canvas, text, rect, textColor, fontName, fontSize, fontStyle) {

    var adjustedRect = ditRectAdjustForPixelRatio(rect);
    var canvas2d = canvas.getContext("2d");

    canvas2d.textBaseline = "top";
    canvas2d.textAlign = "left";
    canvas2d.fillStyle = textColor;
    canvas2d.font = fontStyle + " " + (fontSize * ditDevicePixelRatio) + "px " + fontName;

    ditCanvasDrawText(canvas2d, text, adjustedRect.Width, adjustedRect.Left, adjustedRect.Top);
}

function ditCanvasDrawTextLeftMiddle(canvas, text, rect, textColor, fontName, fontSize, fontStyle) {

    var adjustedRect = ditRectAdjustForPixelRatio(rect);
    var canvas2d = canvas.getContext("2d");

    canvas2d.textBaseline = "middle";
    canvas2d.textAlign = "left";
    canvas2d.fillStyle = textColor;
    canvas2d.font = fontStyle + " " + (fontSize * ditDevicePixelRatio) + "px " + fontName;

    ditCanvasDrawText(canvas2d, text, adjustedRect.Width, adjustedRect.Left, adjustedRect.Top + (adjustedRect.Height / 2));
}

function ditCanvasDrawTextRightMiddle(canvas, text, rect, textColor, fontName, fontSize, fontStyle) {

    var adjustedRect = ditRectAdjustForPixelRatio(rect);
    var canvas2d = canvas.getContext("2d");

    canvas2d.textBaseline = "middle";
    canvas2d.textAlign = "right";
    canvas2d.fillStyle = textColor;
    canvas2d.font = fontStyle + " " + (fontSize * ditDevicePixelRatio) + "px " + fontName;

    ditCanvasDrawText(canvas2d, text, adjustedRect.Width, adjustedRect.Left + adjustedRect.Width, adjustedRect.Top + (adjustedRect.Height / 2));
}

function ditCanvasDrawTextCenterMiddle(canvas, text, rect, textColor, fontName, fontSize, fontStyle) {

    var adjustedRect = ditRectAdjustForPixelRatio(rect);
    var canvas2d = canvas.getContext("2d");

    canvas2d.textBaseline = "middle";
    canvas2d.textAlign = "center";
    canvas2d.fillStyle = textColor;
    canvas2d.font = fontStyle + " " + (fontSize * ditDevicePixelRatio) + "px " + fontName;

    ditCanvasDrawText(canvas2d, text, adjustedRect.Width, adjustedRect.Left + (adjustedRect.Width / 2), adjustedRect.Top + (adjustedRect.Height / 2));
}

function ditCanvasDrawText(canvas2d, text, width, x, y) {

    var line = "";
    var metrics;
    var character;

    // See if the entire text will fit.

    metrics = canvas2d.measureText(text);

    if (metrics.width <= width) {
        canvas2d.fillText(text, x, y);
        return;
    }

    // If not, make a new line, character by character, until we fill up the rectangle.

    for (var i = 0; i < text.length; i++) {

        character = text.substr(i, 1);
        metrics = canvas2d.measureText(line + character);

        if (metrics.width > width) {
            line = line.substr(0, line.length - 2) + "…"; //Unicode ellipsis!
            break;
        }
        line += character;
    }

    canvas2d.fillText(line, x, y);
}

function ditCanvasDrawTextLeftTopWrap(canvas, text, rect, textColor, fontName, fontSize, fontStyle) {

    var line = "";
    var lineWidth = 0;
    var metrics;
    var wordWidth;
    var wordHeight = fontSize * ditDevicePixelRatio;

    var canvas2d = canvas.getContext("2d");

    canvas2d.textBaseline = "top";
    canvas2d.textAlign = "left";
    canvas2d.fillStyle = textColor;
    canvas2d.font = fontStyle + " " + (fontSize * ditDevicePixelRatio) + "px " + fontName;

    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    // Create an array of words.

    var words = text.split(" ");

    // Print multiple lines by concatenating words together until they no 
    // longer fit within the rectangle.

    for (var i = 0; i < words.length; i++) {

        metrics = canvas2d.measureText(words[i] + " ");
        wordWidth = metrics.width;

        // Will this word still fit on the existing line?

        if (lineWidth + wordWidth <= adjustedRect.Width) {

            line += words[i] + " ";
            lineWidth += wordWidth;
        }
        else {

            // Draw the existing line and start a new line with the current word.

            canvas2d.fillText(line, adjustedRect.Left, adjustedRect.Top);
            adjustedRect.Top += wordHeight;
            line = words[i] + " ";
            lineWidth = wordWidth;
        }
    }
    if (line != "") {
        canvas2d.fillText(line, adjustedRect.Left, adjustedRect.Top);
    }
}

function ditCanvasDrawLine(canvas, lineColor, lineWidth, x1, y1, x2, y2) {
    var canvas2d = canvas.getContext("2d");

    canvas2d.lineCap = "round";

    canvas2d.strokeStyle = lineColor;
    canvas2d.lineWidth = lineWidth * ditDevicePixelRatio;

    canvas2d.beginPath();
    canvas2d.moveTo(x1 * ditDevicePixelRatio, y1 * ditDevicePixelRatio);
    canvas2d.lineTo(x2 * ditDevicePixelRatio, y2 * ditDevicePixelRatio);
    canvas2d.stroke();
}

function ditCanvasDrawActionArrow(canvas, rect) {
    var canvas2d = canvas.getContext("2d");

    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    var centerX = adjustedRect.Left + (adjustedRect.Width / 2);
    var centerY = adjustedRect.Top + (adjustedRect.Height / 2);

    var leftTopX = centerX - (3 * ditDevicePixelRatio);
    var leftTopY = centerY - (6 * ditDevicePixelRatio);
    var centerRightX = centerX + (3 * ditDevicePixelRatio);
    var centerRightY = centerY;
    var leftBottomX = centerX - (3 * ditDevicePixelRatio);
    var leftBottomY = centerY + (6 * ditDevicePixelRatio);

    canvas2d.beginPath();
    canvas2d.moveTo(leftTopX, leftTopY);
    canvas2d.lineTo(centerRightX, centerRightY);
    canvas2d.lineTo(leftBottomX, leftBottomY);

    canvas2d.lineWidth = 3 * ditDevicePixelRatio;
    canvas2d.lineCap = "butt";
    canvas2d.strokeStyle = "#7f7f7f";

    canvas2d.stroke();
}

function ditCanvasDrawCheckBox(canvas, rect, checked) {

    var canvas2d = canvas.getContext("2d");
    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    // Draw the box for the checkmark.

    ditCanvasFillRect(canvas, rect, "#ffffff");

    canvas2d.strokeStyle = "#7f7f7f";
    canvas2d.lineCap = "square";
    canvas2d.lineWidth = 2 * ditDevicePixelRatio;
    canvas2d.strokeRect(adjustedRect.Left, adjustedRect.Top, adjustedRect.Width, adjustedRect.Height);

    // If indicated, draw the checkmark.

    if (checked == true) {
        ditCanvasDrawCheckmark(canvas, rect);
    }
}

function ditCanvasDrawCheckmark(canvas, rect) {

    var canvas2d = canvas.getContext("2d");
    var adjustedRect = ditRectAdjustForPixelRatio(rect);

    var centerX = adjustedRect.Left + (adjustedRect.Width / 2);
    var centerY = adjustedRect.Top + (adjustedRect.Height / 2);

    var leftTopX = centerX - (5 * ditDevicePixelRatio);
    var leftTopY = centerY + (1 * ditDevicePixelRatio);
    centerX = centerX - (1 * ditDevicePixelRatio);
    centerY = centerY + (5 * ditDevicePixelRatio);
    var rightTopX = centerX + (6 * ditDevicePixelRatio);
    var rightTopY = centerY - (10 * ditDevicePixelRatio);

    canvas2d.beginPath();
    canvas2d.moveTo(leftTopX, leftTopY);
    canvas2d.lineTo(centerX, centerY);
    canvas2d.lineTo(rightTopX, rightTopY);

    canvas2d.lineWidth = 3 * ditDevicePixelRatio;
    canvas2d.lineCap = "round";
    canvas2d.strokeStyle = "#000000";

    canvas2d.stroke();
}

function ditImageNew(sourceFileName, onLoadCallback) {

    var im = new Image();
    im.src = sourceFileName;
    im.onload = onLoadCallback;
    return im;
}

function ditApplyCssTransition(element, ifCssSupported){
    if(ditCssTransitions){
        element.style.transition = ifCssSupported;
        element.style.webkitTransition = ifCssSupported;
    }
}

function ditNowDate() {
    var currentDate = new Date();

    currentDate.setHours(0);
    currentDate.setMinutes(0);
    currentDate.setSeconds(0);
    currentDate.setMilliseconds(0);

    return currentDate;
}

// Parse a date from a string in format "yyyy-MM-dd".
// TODO: Fix this!  Do we need any validation?
function ditDateParse(dateString) {

    var dateValue = new Date();

    var year = dateString.substring(0, 4);
    var month = dateString.substring(5, 7);
    month = month - 1;
    var day = dateString.substring(8, 10);

    dateValue.setFullYear(year);
    dateValue.setMonth(month);
    dateValue.setDate(day);

    dateValue.setHours(0);
    dateValue.setMinutes(0);
    dateValue.setSeconds(0);
    dateValue.setMilliseconds(0);

    return dateValue;
}

function ditDateCompare(date1, date2) {

    if (date1.getFullYear() < date2.getFullYear()) {
        return -1;
    }

    if (date1.getFullYear() > date2.getFullYear()) {
        return 1;
    }

    // Years are equal if we get here.

    if (date1.getMonth() < date2.getMonth()) {
        return -1;
    }

    if (date1.getMonth() > date2.getMonth()) {
        return 1;
    }

    // Years and Months are equal if we get here.

    if (date1.getDate() < date2.getDate()) {
        return -1;
    }

    if (date1.getDate() > date2.getDate()) {
        return 1;
    }

    // Years, Months, and Days are equal if we get here.

    return 0;
}

function ditDateAddDays(dateValue, days) {
    var newDateValue = new Date();

    newDateValue.setFullYear(dateValue.getFullYear());
    newDateValue.setMonth(dateValue.getMonth());
    newDateValue.setDate(dateValue.getDate() + days);

    return newDateValue;
}

// Return a string with time formatted as hh:mm am/pm.
function ditTimeFormat1(timeValue) {

    var hour = timeValue.getHours();
    var mins = timeValue.getMinutes();
    var timeString;

    if (mins < 10) {
        mins = "0" + mins;
    }

    if (hour > 11) {
        if (hour > 12) {
            hour = hour - 12;
        }
        timeString = hour + ":" + mins + " pm";
    }
    else {
        timeString = hour + ":" + mins + " am";
    }

    return timeString;
}

// Return a string with date formatted as mmm d, yyyy.
function ditDateFormat1(dateValue) {
    var months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

    var dateString = months[dateValue.getMonth()]
        + " "
        + dateValue.getDate()
        + ", "
        + dateValue.getFullYear();

    return dateString;
}

// Return a string with date formatted as yyyy-mm-dddd.
function ditDateFormat2(dateValue) {

    var month = dateValue.getMonth() + 1;
    var day = dateValue.getDate();

    var dateString = dateValue.getFullYear() + "-";

    if (month < 10) {
        dateString += "0" + month + "-";
    }
    else {
        dateString += month + "-";
    }

    if (day < 10) {
        dateString += "0" + day;
    }
    else {
        dateString += day;
    }

    return dateString;
}

// Return a string with date formatted as mmm d.
function ditDateFormat3(dateValue) {
    var months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

    var dateString = months[dateValue.getMonth()]
        + " "
        + dateValue.getDate();

    return dateString;
}

// Return a string with date formatted as mm/dd/yyyy.
function ditDateFormat4(dateValue) {

    var month = dateValue.getMonth() + 1;
    var day = dateValue.getDate();

    var dateString;

    dateString = month + "/";

    if (day < 10) {
        dateString += "0" + day;
    }
    else {
        dateString += day;
    }

    dateString += "/" + dateValue.getFullYear();

    return dateString;
}

// Return a string with date formatted as "mmm d, yyyy" but don't display year if = the current system year.
function ditDateFormat5(dateValue) {
    var months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

    var currentDate = new Date();
    var currentYear = currentDate.getFullYear();
    var dateString;

    dateString = months[dateValue.getMonth()]
        + " "
        + dateValue.getDate();

    if (dateValue.getFullYear() != currentYear) {
        dateString += ", " + dateValue.getFullYear();
    }

    return dateString;
}

function ditBoolParse(boolString) {
    if (boolString == "y") {
        return true;
    }
    else if (boolString == "true") {
        return true;
    }
    else {
        return false;
    }
}

function ditBoolToYN(boolValue) {
    if (boolValue == true) {
        return "y";
    }
    else {
        return "n";
    }
}

function ditStorageGet(key, defaultValue) {
    var value = ditStorage.getItem(key);
    if (value == undefined) {
        return defaultValue;
    }
    return value;
}

function ditStorageSet(key, value) {
	ditStorage.setItem(key, value);
}

function ditStorageRemove(key) {
	ditStorage.removeItem(key);
}

function ditStorageClear() {
	ditStorage.clear();
}

// See reference:
// http://stackoverflow.com/questions/986937/how-can-i-get-the-browsers-scrollbar-sizes
function ditGetScrollBarWidth() {
    var inner = document.createElement('p');
    inner.style.width = "100%";
    inner.style.height = "200px";

    var outer = document.createElement('div');
    outer.style.position = "absolute";
    outer.style.top = "0px";
    outer.style.left = "0px";
    outer.style.visibility = "hidden";
    outer.style.width = "200px";
    outer.style.height = "150px";
    outer.style.overflow = "hidden";
    outer.appendChild(inner);

    document.body.appendChild(outer);
    var w1 = inner.offsetWidth;
    outer.style.overflow = 'scroll';
    var w2 = inner.offsetWidth;
    if (w1 == w2) w2 = outer.clientWidth;

    document.body.removeChild(outer);

    return (w1 - w2);
};