var Feed = Backbone.Model.extend({
    defaults: {
        url    : '',
        title  : '',
        unread : 0,
        items  : []
    },

    initialize: function( ) {
        this.bind('change', this.changeFeed);
    },

    changeFeed: function () {
        database.addFeed(this.attributes);

        // If url changes, remove previous feed (and keep the new one only)
        if (this._previousAttributes.url && this._previousAttributes.url != this.attributes.url) {
            database.deleteFeed(this._previousAttributes.url);
        }
    },

    markAsRead: function() {
        //~ console.log('Feed.markAsRead');

        var items = this.attributes.items;

        $(items).each(function (index, item) {
            if (!item.read) {
                items[index].read = true;
            }
        });
        this.set({'unread': 0});
    },

    clearReload: function(httpRequest, httpTimeout) {
        httpRequest.abort();
        clearTimeout(httpTimeout);

        this.reloading = false;
        this.trigger('reload');
    },

    reload: function() {
        this.reloading = true;
        this.trigger('reload');

        var self = this;

        // CORS
        var httpRequest = new XMLHttpRequest({
            mozSystem: true
        });
        var httpTimeout;

        function parseRss() {

            if (httpRequest.status === 200) {

                try {
                    var feedParser = new FeedParser();
                    feedParser.parse(httpRequest.responseText, self);
                } catch (e) {
                    console.error("Error parsing : " + self.get('url'), e);
                }

            }

            self.clearReload(httpRequest, httpTimeout);
        }

        httpRequest.onload = parseRss;
        httpRequest.onerror = function() {
            self.clearReload(httpRequest, httpTimeout);
        };
        httpRequest.open('GET', this.get('url'));

        try {
            httpRequest.send();

            httpTimeout = setTimeout(function() {
                self.clearReload(httpRequest);

                console.log("Request timed out : " + self.get('url'));
            }, 30000);

        } catch (e) {
            this.reloading = false;
            this.trigger('reload');

            var error = "There is a problem with the url\n'" + this.get('url') + "'";
            console.error(error);
        }

    }

});

var FeedList = Backbone.Collection.extend({
    model: Feed,

    initialize: function() {
        this.unreadItems   = document.getElementById("unreadItems");
    },

    initEventHandler: function() {
        // Func called in db.js. Init event handler after that db loaded all feeds
        this.bind('add',    this.saveFeed);
        this.bind('remove', this.removeFeed);

        Backbone.pubSub.on('feed:toggleReadItem'  , this.toggleReadItem, this);
        Backbone.pubSub.on('feed:markAllRead'     , this.markAllRead, this);
        Backbone.pubSub.on('feed:markAllFeedsRead', this.markAllFeedsRead, this);
    },
    saveFeed: function(model){
        // Add to DB
        database.addFeed(model.attributes);
    },

    removeFeed: function(model){
        console.log('Collection.removeFeed');
        // Delete from DB
        database.deleteFeed(model.get('url'));
        model.destroy();
    },

    toggleReadItem: function(newItem){
        //~ console.log('FeedList.toggleReadItem');

        var models_list = this.where({'url': newItem.url});

        if (models_list.length != 1) {
            return;
        }

        var feedModel = models_list[0];
        var unread    = feedModel.get('unread');
        var oldItems  = feedModel.attributes.items;

        $(oldItems).each(function (index, oldItem) {
            if ( (oldItem.link == newItem.link) && (oldItem.read != newItem.read) ) {
                oldItems[index].read = newItem.read;

                if (newItem.read) {
                    feedModel.set({'unread': --unread});
                } else {
                    feedModel.set({'unread': ++unread});
                }

                // Stop the loop
                return false;
            }
        });

        this.unreadItems.innerHTML = unread;
    },

    markAllRead: function(url){
        //~ console.log('FeedList.markAllRead');

        var models_list = this.where({'url': url});

        if (models_list.length != 1) {
            return;
        }

        var model = models_list[0];
        model.markAsRead();
    },

    markAllFeedsRead: function(){
        console.log('FeedList.markAllFeedsRead');

        this.each(function(model) {
            model.markAsRead();
        });
    },

    comparator: function(model) {
        //console.log('comparator');
        return model.get('title').toLowerCase();
    }
});

// Make sure that feed is unique on url before adding
FeedList.prototype.add = function(feed) {

    // Check if it's the same, without the trailing slash
    var isDupe = this.any(function(_feed) {
        return _feed.get('url').replace(/\/$/, '') === feed.url.replace(/\/$/, '');
    });
    if (isDupe) {
        console.log("FeedList.add : duplicate feed", feed.url);

        return false;
    }

    return Backbone.Collection.prototype.add.call(this, feed);
};

var FeedView = Backbone.View.extend({
    tagName: 'li',

    events: {
        'click' : 'loadItems',
    },

    initialize: function(){
        _.bindAll(this, 'render', 'unrender', 'loadItems');

        this.listenTo(this.model, 'change' , this.render);
        this.listenTo(this.model, 'reload' , this.render);
        this.listenTo(this.model, 'destroy', this.unrender);

        this.feedEditTitle = document.getElementById("feed-edit-title");
        this.feedEditUrl   = document.getElementById("feed-edit-url");
        this.feedEditCid   = document.getElementById("feed-edit-cid");
    },

    render: function() {
        var classes   = '';
        var disabled  = '';
        var indicator = '';

        // Show unread number
        if (this.model.get('unread')) {
            indicator = '<span class="counter">' + this.model.get('unread') + '</span>';
        } else {
            classes += "read ";
        }

        if (this.model.reloading) {
            disabled = "disabled";
        }

        this.el.innerHTML = '<button ' + disabled + ' class="' + classes + '">' + this.model.get('title')  + '</button>' + indicator;

        return this;
    },

    unrender: function(){
        $(this.el).remove();

        var index = listFeedView.views.indexOf(this);
        listFeedView.views.splice(index, 1);
    },

    loadItems: function() {
        listItemView.loadItems(this.model);

        this.feedEditTitle.value = this.model.get('title');
        this.feedEditUrl.value   = this.model.get('url');
        this.feedEditCid.value   = this.model.cid;
    }
});

var ListFeedView = Backbone.View.extend({
    el: 'body',

    events: {
        'click .reload' : 'reloadAll',

        'click #mark-all-feeds-read' : 'confirmMarkAllFeedsRead',
        'click #export' : 'export'
    },

    initialize: function(){
        this.views      = [];
        this.counter    = 0;

        this.btnSearch = document.getElementById('search-btn');
        this.nofeed    = document.getElementById("nofeed");
    },

    initEventHandler: function() {
        _.bindAll(this, 'render', 'appendFeed', 'reloadAll');

        feedList.bind('add', this.render, this);
        Backbone.pubSub.on('feed:reloadAll', this.reloadAll, this);

        this.render();

        setTimeout(function() {
            loadingDialog = document.getElementById("loading-dialog");
            loadingDialog.classList.add('move-down');

            window.setTimeout(function() {
                loadingDialog.parentNode.removeChild(loadingDialog);

                // Bind activities by including JS to head
                var imported = document.createElement('script');
                imported.src = 'js/activities.js';
                document.head.appendChild(imported);
            }, 1000);

        }, 1);

        if (settings.values.auto_refresh == AUTOREFRESH_STARTUP) {
            this.reloadAll();
        }
    },

    destroyViews: function() {
        _.invoke(this.views, 'remove');
        this.views = [];
    },

    render: function(){
        var self = this;

        this.destroyViews();

        if (!feedList.length) {
            this.nofeed.hidden = false;
        } else {
            this.nofeed.hidden = true;

            feedList.forEach(function(feed){
                self.appendFeed(feed);
            }, this);
        }

    },

    appendFeed: function(feed){
        var feedView = new FeedView({
            model: feed
        });
        this.views.push(feedView);
        $('#feed-list', this.el).append(feedView.render().el);
    },

    reloadAll: function() {
        if (navigator.onLine) {
            showBanner("banner-feedreloading");

            var feedListLength = feedList.length;
            var i = 0;
            var self = this;

            var callReload = function() {
                setTimeout(function() {
                    if (i < feedListLength) {
                        feedList.at(i).reload();
                        i++;
                        callReload();
                    } else {
                        if (timelineView.newItemsCount) {
                            notifyUser(translate('newitems', { 'number': timelineView.newItemsCount }));
                        }
                    }
                }, 100);
            };
            callReload();

        }  else {
            console.log('not connected');
            showBanner("banner-notconnected");
        }

        if (this.btnSearch.parentNode.classList.contains('active')) {
            this.btnSearch.click();
        }
    },

    confirmMarkAllFeedsRead: function(event) {
        console.log("Feed.confirmMarkAllFeedsRead");

        var markAllFeedsReadConfirm = document.getElementById("mark-all-feeds-read-confirm");
        markAllFeedsReadConfirm.classList.remove('hidden');

        var self = this;

        $(markAllFeedsReadConfirm).bind('submit', function(event) {
            event.preventDefault();

            $(markAllFeedsReadConfirm).unbind();
            markAllFeedsReadConfirm.classList.add('hidden');

            if (event.originalEvent.explicitOriginalTarget.id == 'confirm-mark-all-feeds-read-btn') {
                self.markAllFeedsRead();
            }
        });

    },

    markAllFeedsRead: function() {
        console.log("Feed.markAllFeedsRead");

        Backbone.pubSub.trigger('feed:markAllFeedsRead');

        // Refresh timeline if needed
        if (!document.getElementById('div-timeline').hidden) {
            timelineView.loadTimeline();
        }

        showBanner("banner-feedallread");
    },

    export: function() {
        console.log('export');

        var opmlExport = '<?xml version="1.0"?>\n<opml version="1.0">\n<head>\n<title>SimpleRSS export - param_date</title>\n</head>\n<body>\nparam_outlines</body>\n</opml>';

        var outlineTemplate = '<outline type="rss" xmlUrl="param_xmlUrl" htmlUrl="param_htmlUrl" title="param_title"/>\n';

        // Create outline for each feed
        var outlines = "";
        var encoderDiv = document.createElement("div");

        feedList.forEach(function(feed){
            // Hack to encode urls and titles
            encoderDiv.innerText = encoderDiv.textContent = feed.get('url');
            var url = encoderDiv.innerHTML;

            encoderDiv.innerText = encoderDiv.textContent = feed.get('title');
            var title = encoderDiv.innerHTML;

            var newOutline = outlineTemplate.replace('param_xmlUrl', url);
            newOutline = newOutline.replace('param_htmlUrl', url);
            newOutline = newOutline.replace('param_title', title);

            outlines += newOutline;
        }, this);

        // Set date of today in title
        opmlExport = opmlExport.replace('param_date', moment().format("MMM Do YY"));

        // Add outlines in body
        opmlExport = opmlExport.replace('param_outlines', outlines);
        //~ console.log(opmlExport);

        // Create blob
        var aFileParts = [opmlExport];
        var opmlBlob = new Blob(aFileParts, {type : 'text/xml'});

        // Title of the file : SimpleRSS_export_YYMMDD.opml
        var filename = "SimpleRSS_export_" + moment().format("YYMMDDHHmmss") + ".opml";
        console.log(filename);

        // Write on SD card
        var sdcard = navigator.getDeviceStorage("sdcard");
        var request = sdcard.addNamed(opmlBlob, filename);

        request.onsuccess = function () {
           var name = this.result;

           console.log("File write on SD card");
           showBanner("banner-sdcardsuccess", ":<br/><small>param</small>".replace('param', name));
        };

        request.onerror = function () {
           console.error('Unable to write the file: ' + this.error);

           showBanner("banner-sdcarderror");
        };

    }
});

var AddFeedView = Backbone.View.extend({
    el: '#add-view',

    events: {
        'submit #add-form' : 'addFeed',
        'submit #opml-form'  : 'importOpml',
    },

    initialize: function() {
        _.bindAll(this, 'addFeed', 'importOpml');

        this.viewAdd = document.getElementById("add-view");
        this.importBtn = document.getElementById('import-btn');
    },

    addFeed: function(event){
        event.preventDefault();

        var form = event.target;

        if (form.checkValidity())
        {
            var feed = feedList.add({
                url   : $('input#feed-url').val(),
                title : $('input#feed-url').val()
            });

            if (feed) {
                this.el.classList.remove('move-up');
                this.el.classList.add('move-down');

                form.reset();

                showBanner("banner-feedadded");
            } else {
                showBanner("banner-feedduplicated");
            }
        }
    },

    importOpml: function(event){
        event.preventDefault();

        var form = event.target;

        if (form.checkValidity()) {

            var self = this;
            var opmlUrl = $('input#opml-url').val();

            if (navigator.onLine) {

                var resultOpml = function() {
                    if (httpRequest.status === 200) {
                        clearTimeout(httpTimeout);

                        var opmlParser = new OpmlParser();

                        self.viewAdd.classList.remove('move-up');
                        self.viewAdd.classList.add('move-down');

                        $('#add-switch-btn').trigger('click');

                        event.target.reset();

                        opmlParser.parse(httpRequest.responseText);

                        // Enable import button
                        self.importBtn.disabled = false;
                    }
                };

                // Disable import button
                this.importBtn.disabled = true;

                // CORS
                var httpRequest = new XMLHttpRequest({
                    mozSystem: true
                });
                var httpTimeout;

                httpRequest.onload = resultOpml;
                httpRequest.open('GET', opmlUrl);

                try {
                    httpRequest.send();

                    httpTimeout = setTimeout(function() {
                        httpRequest.abort();

                        // Enable import button
                        self.importBtn.disabled = false;

                        console.log("Request timed out : " + opmlUrl);
                        showBanner("banner-requesttimeout");
                    }, 30000);

                } catch (e) {
                    var error = "There is a problem with the url\n'" + opmlUrl + "'";
                    console.error(error);
                }

            } else {
                console.log('not connected');
                showBanner("banner-notconnected");
            }

        }
    },

});

var EditFeedView = Backbone.View.extend({
    el: '#edit-view',

    events: {
        'submit #form-edit' : 'editFeed',
    },

    initialize: function(){
        this.feedEditUrl   = document.getElementById('feed-edit-url');
        this.feedEditCid   = document.getElementById('feed-edit-cid');
        this.feedEditTitle = document.getElementById('feed-edit-title');

        this.titleItems  = document.getElementById("titleItems");
        this.titleDetail = document.getElementById("titleDetail");
    },

    editFeed: function(event){
        event.preventDefault();

        var form = event.target;
        var url = this.feedEditUrl.value;

        if (event.originalEvent.explicitOriginalTarget.id == 'save-edit-btn')
        {
            if (form.checkValidity()) {
                var cid   = this.feedEditCid.value;
                var title = this.feedEditTitle.value;

                var model = feedList.get(cid);

                model.set(
                    {
                        'title': title,
                        'url': url,
                    }
                );

                $('#edit-back-btn').trigger('click');

                showBanner("banner-feedsaved");
                return;
            }
        }

        if (event.originalEvent.explicitOriginalTarget.id == 'delete-edit-btn') {
            var viewConfirmDeleteEdit = document.getElementById("delete-form-confirm");
            viewConfirmDeleteEdit.classList.remove('hidden');

            var self = this;

            $(viewConfirmDeleteEdit).bind('submit', function(event) {
                event.preventDefault();

                if (event.originalEvent.explicitOriginalTarget.id == 'confirm-delete-edit-btn') {
                    var cid   = self.feedEditCid.value;
                    var model = feedList.get(cid);

                    feedList.remove(model);

                    $('#edit-back-btn').trigger('click');
                    $('#back-feed-btn').trigger('click');

                    showBanner("banner-feeddeleted");
                }

                $(viewConfirmDeleteEdit).unbind();
                viewConfirmDeleteEdit.classList.add('hidden');
            });
        }
    },

});
