var Item = Backbone.Model.extend({
    defaults: {
        id          : '',
        url         : '',
        title       : '',
        description : '',
        summary     : '',
        link        : '',
        author      : '',
        date        : '',
        media       : [],
        read        : false,
    },

    getTitle: function() {
        var titleEl = document.createElement('DIV');

        if (this.get('title')) {
            titleEl.innerHTML = this.get('title');

            return titleEl.textContent;
        }

        if (this.get('description')) {
            titleEl.innerHTML = this.get('description');

            return titleEl.textContent.substring(0, 140);
        }

        return translate('untitled');
    },

    getContent: function() {
        if (this.get('description').length > 0) {
            return this.get('description');
        }

        if (this.get('summary').length > 0) {
            return this.get('summary');
        }

        return '<p class="no-content">' + translate("nocontent") + '</p>';
    },

    getDate: function() {
        var date = this.get('date');

        if (!date) {
            return "";
        }

        return moment(this.get('date')).format('ll');
    },

    getDateDetail: function() {
        var date = this.get('date');

        if (!date) {
            return "";
        }

        return moment(this.get('date')).format("LL");
    },

    getDateTimeline: function() {
        var date = this.get('date');

        if (!date) {
            return "&nbsp;";
        }

        return moment(this.get('date')).fromNow();
    },

    renderMedia: function() {
        var mediaHTML = "";

        for (var i = 0; i < this.get('media').length; i++) {
            var media = this.get('media')[i];

            if (MIME_TYPES.AUDIO.indexOf(media.type) >= 0) {
                mediaHTML += "<audio controls>";
                mediaHTML += "<source src='" + media.url +"' type='" + media.type + "' length='" + media.length + "'>";
                mediaHTML += "</audio>";
            }

            if (MIME_TYPES.VIDEO.indexOf(media.type) >= 0) {
                mediaHTML += "<video controls>";
                mediaHTML += "<source src='" + media.url +"' type='" + media.type + "' length='" + media.length + "'>";
                mediaHTML += "</video>";
            }

            if (MIME_TYPES.IMAGE.indexOf(media.type) >= 0) {
                mediaHTML += "<img src='" + media.url +"'>";
            }
        }

        return mediaHTML;

    }

});

var ItemList = Backbone.Collection.extend({
    model: Item
});

var ItemView = Backbone.View.extend({
    tagName: 'li',

    events: {
        'touchstart' : 'touchstart',
        'touchmove'  : 'touchmove',
        'touchend'   : 'touchend'
    },

    initialize: function(){
        _.bindAll(this, 'render', 'detail');

        this.model.listenTo(this.model, 'change', this.render);

        this.viewDetail  = document.getElementById("detail-view");
        this.divDetail   = document.getElementById('div-detail');
        this.titleDetail = document.getElementById("titleDetail");

        this.detailTitle       = document.getElementById('detail-title');
        this.detailDate        = document.getElementById('detail-date');
        this.detailAuthor      = document.getElementById('detail-author');
        this.detailDescription = document.getElementById('detail-description');
        this.detailMedia       = document.getElementById('detail-media');

        this.readOnlineBtn = document.getElementById('read-online');
        this.sendMailBtn   = document.getElementById('sendMail');
        this.shareBtn      = document.getElementById('share');
    },

    render: function(){
        var classes = '';

        // If unread
        if (!this.model.get('read'))
        {
            this.el.classList.add('unread');
        } else {
            this.el.classList.remove('unread');

            // If hide read
            if (settings.values.hide_read) {
                this.el.classList.add('hidden');
            } else {
                this.el.classList.remove('hidden');
            }
        }

        // If innerHTML's content already exists, no need to regenerate it
        if (!this.el.innerHTML) {

            var title     = this.model.getTitle();
            var date      = this.model.getDate();
            var feedTitle = "";

            // Hack, If timeline
            if (this.timeline) {

                // If need highlight some keywords
                if (this.timeline.highlight) {
                    title = highlight(this.model.getTitle(), this.timeline.highlight);
                }

                // Display timeAgo and feedTitle
                date      = this.model.getDateTimeline();
                feedTitle = '<p>' + this.timeline.feedTitle + '</p>';

            }

            var html = '<a>';

            // Title
            html += '<p>' + title + '</p>';

            // Date
            html += '<p>' + date + '</p>';

            // feedTitle (if timeline)
            html += feedTitle;

            html += '</a>';

            this.el.innerHTML = html;
        }

        return this;
    },

    detail: function(){
        this.divDetail.scrollTop = 0;

        var self = this;

        // If timeline, hack to set detail view title in header now.
        if (this.timeline) {
            this.titleDetail.textContent = this.timeline.feedTitle;
        }

        // Title
        this.detailTitle.textContent = this.model.getTitle();

        // Author and date
        this.detailDate.textContent   = this.model.getDateDetail();
        this.detailAuthor.textContent = this.model.get('author');

        // Media
        this.detailMedia.innerHTML += this.model.renderMedia();

        // Buttons
        this.readOnlineBtn.setAttribute('href', this.model.get('link'));

        this.sendMailBtn.setAttribute('href', encodeURIComponent(this.model.get('link')));
        this.sendMailBtn.setAttribute('title', encodeURIComponent(this.model.getTitle()));

        this.shareBtn.setAttribute('href', encodeURIComponent(this.model.get('link')));
        this.shareBtn.setAttribute('title', encodeURIComponent(this.model.getTitle()));

        // Show the view
        if (this.timeline) {
            var drawer = document.getElementById('drawer');
            drawer.classList.add('fade-left');
        } else {
            var viewItems = document.getElementById('items-view');
            viewItems.classList.add('fade-left');
        }

        this.viewDetail.classList.add('move-center');

        setTimeout(function() {
            // Mark as read
            if (!self.model.get('read')) {

                self.model.set( { 'read' : true } );

                Backbone.pubSub.trigger('feed:toggleReadItem', self.model.attributes);

                if (settings.values.hide_read) {
                    if (self.timeline) {
                        Backbone.pubSub.trigger('timeline:appendOne');
                    } else {
                        Backbone.pubSub.trigger('item:appendOne');
                    }
                }
            }

            // Description or summary
            self.detailDescription.innerHTML = self.model.getContent();

            setTimeout(function() {
                // Show description
                self.detailDescription.classList.add('fadein');
            }, 100);
        }, 400);
    },

    touchstart: function(event){
        //~ console.log('touchstart');

        this.touchPosition = event.originalEvent.changedTouches[0].clientY;
        this.isTouchClickActive = true;

        var self = this;
        this.touch_id = window.setTimeout(function() {

            if (self.isTouchClickActive) {
                self.isTouchClickActive = false;
                self.toggleRead();
            }

        }, MARK_UNREAD_TOUCH_DELAY);
    },

    touchmove: function(event) {
        // Cancel toggleRead if touch move too much

        var currentPosition = Math.abs(event.originalEvent.changedTouches[0].clientY - this.touchPosition);

        if (currentPosition > TOUCH_OFFSET) {
            this.isTouchClickActive = false;
        }
    },

    touchend: function() {
        //~ console.log('touchend');

        if (this.isTouchClickActive) {
            this.isTouchClickActive = false;
            this.detail();
        }

    },

    toggleRead: function() {
        this.model.set({'read': !this.model.get('read')});

        Backbone.pubSub.trigger('feed:toggleReadItem', this.model.attributes);
    },

});

var ListItemView = Backbone.View.extend({
    el: 'body',

    events: {
        'click #read-online'     : 'readOnline',
        'click #div-detail a'    : 'readOnline',
        'click #sendMail'        : 'sendMail',
        'click #share'           : 'share',

        'click #mark-all-read'   : 'markAllRead',
        'click #back-items-btn'  : 'cleanDetail',

        'click #reload-feed' : 'reloadFeed',
        'click #updateItems' : 'renderItems',

        'click #back-feed-btn': 'backToFeed',
    },

    initialize: function(){
        _.bindAll(this, 'markAllRead', 'readOnline', 'loadItems', 'checkScroll', 'reloadFeed', 'backToFeed', 'notifyUpdate', 'renderView');

        Backbone.pubSub.on('item:appendOne', this.appendOne, this);

        $("#div-items").scroll(this.checkScroll);

        this.collection = new ItemList();
        this.collection.bind('add', this.addItem);

        this.pagination = 0;

        this.titleItems  = document.getElementById("titleItems");
        this.titleDetail = document.getElementById("titleDetail");
        this.unreadItems = document.getElementById("unreadItems");
        this.totalItems  = document.getElementById("totalItems");
        this.itemList    = document.getElementById("item-list");

        this.viewItems = document.getElementById("items-view");
        this.divItems  = document.getElementById('div-items');

        this.detailTitle       = document.getElementById('detail-title');
        this.detailDescription = document.getElementById('detail-description');
        this.detailDate        = document.getElementById('detail-date');
        this.detailAuthor      = document.getElementById('detail-author');
        this.detailMedia       = document.getElementById('detail-media');

        this.btnShare      = document.getElementById('share');
        this.btnSendMail   = document.getElementById('sendMail');
        this.btnReadOnline = document.getElementById('read-online');

        this.backFeedBtn = document.getElementById("back-feed-btn");

        this.noitem = document.getElementById('noitem');
        this.updateItemsButton = document.getElementById("updateItems");
    },

    checkScroll: function (evt) {
        if(evt.target.scrollTopMax - evt.target.scrollTop !== 0) {
            return;
        }

        if (!this.items) {
            return;
        }

        this.appendItems(20);
    },

    appendOne: function () {
        console.log('Item.appendOne');
        this.appendItems(1);
    },

    appendItems: function(number) {
        // Append max 20 el in UI
        var pagination_max = this.pagination + number;

        if (this.pagination > this.items.length) {
            this.pagination = this.items.length;
        }

        if (pagination_max > this.items.length) {
            pagination_max = this.items.length;
        }

        // Loop through items
        for (var i = this.pagination; i < pagination_max; i++) {

            if (!(settings.values.hide_read && this.items[i].read)) {
                this.collection.add(this.items[i]);
            } else {

                // Don't add if hide read but get the next unread to fill with number
                if (pagination_max < this.items.length) {
                    pagination_max++;
                }

            }
        }
        this.pagination = i;
    },

    clearItems: function() {
        this.items = null;

        // Clear view
        this.collection.reset();
        this.pagination         = 0;
        this.divItems.scrollTop = 0;
        this.itemList.innerHTML = '';

        // Hide notification button
        this.updateItemsButton.classList.add('hidden');
        this.updateItemsButton.innerHTML = "";
    },

    loadItems: function (feed) {
        console.log('ListItemView.loadItems');

        // Attach the feed model and listen if changing
        this.feed = feed;

        this.listenTo(feed, 'notifyUpdate', this.notifyUpdate);
        this.listenTo(feed, 'change', this.feedChange);

        this.render();
    },

    notifyUpdate: function(newItems) {
        console.log('item:notifyUpdate');

        // Update button notifying new items
        this.updateItemsButton.innerHTML = translate('newitems', { 'number' : newItems.length });
        this.updateItemsButton.classList.remove('hidden');
    },

    feedChange: function(feed) {
        this.feed = feed;

        this.renderView();
    },

    render: function() {
        this.renderItems();
        this.renderView();
    },

    renderView: function() {
        // Set data in UI
        this.titleItems.innerHTML  = this.feed.get('title');
        this.titleDetail.innerHTML = this.feed.get('title');
        this.unreadItems.innerHTML = this.feed.get('unread');
        this.totalItems.innerHTML  = this.feed.get('items').length;

        this.showHideNoitem();
    },

    renderItems: function() {
        this.clearItems();
        this.items = this.feed.get('items');

        // Load new items
        this.appendItems(20);

        // Display view
        var self = this;
        setTimeout(function() {
            var drawer = document.getElementById('drawer');
            drawer.classList.add('fade-left');

            self.viewItems.classList.add('move-center');
        }, 1);
    },

    cleanDetail: function () {
        //~ console.log('cleanDetail');

        this.detailTitle.innerHTML       = '';
        this.detailDate.innerHTML        = '';
        this.detailAuthor.innerHTML      = '';
        this.detailMedia.innerHTML       = '';
        this.detailDescription.innerHTML = '';
        this.detailDescription.classList.remove('fadein');
    },

    markAllRead: function (event) {
        console.log('mark all as read');

        var url = this.items[0].url;

        this.backToFeed();

        Backbone.pubSub.trigger('feed:markAllRead', url);
    },

    readOnline: function (event) {
        event.preventDefault();

        var url = event.currentTarget.getAttribute('href');
        console.log('open url in browser: ' + url);

        // Open url in browser
        var activity = new MozActivity({
            name: "view",
            data: {
                type: "url",
                url: url
            }
        });

        activity.onerror = function() {
            console.log(this.error);
        };
    },

    sendMail: function (event) {
        event.preventDefault();

        var url     = this.btnSendMail.getAttribute('href');
        var subject = this.btnSendMail.getAttribute('title');
        console.log('send email: ' + subject);

        // Create new email
        new MozActivity({
            name: "new",
            data: {
                type: "mail",
                url: "mailto:?subject=" + subject + "&body=" + url
            }
        });
    },

    // See for examples https://github.com/robnyman/Firefox-OS-Boilerplate-App/blob/gh-pages/js/webapp.js
    share: function (event) {
        event.preventDefault();

        var url   = decodeURIComponent(this.btnShare.getAttribute('href'));
        var title = decodeURIComponent(this.btnShare.getAttribute('title'));
        console.log('share: ' + title + url);

        // Open url in browser
        new MozActivity({
            name: "share",
            data: {
                type      : "url",
                number    : 99, // ugly workaround to avoid SimpleRSS to be listed in share activity
                url       : url,
                title     : title
            }
        });
    },

    addItem: function(item){
        var itemView = new ItemView({
            model: item
        });
        $('#item-list', this.el).append(itemView.render().el);
    },

    reloadFeed: function() {
        if (navigator.onLine) {
            showBanner("banner-feedreloading");

            this.feed.reload();
        }  else {
            console.log('not connected');
            showBanner("banner-notconnected");
        }
    },

    backToFeed: function() {
        // Stop listening to events of previous feed
        this.stopListening();

        // Set this feed to null
        this.feed  = null;

        this.clearItems();

        //~ this.el.classList.remove('move-center');
        var itemsView = document.getElementById("items-view");
        itemsView.classList.remove('move-center');

        var drawer = document.getElementById('drawer');
        drawer.classList.remove('fade-left');
    },

    showHideNoitem: function() {
        var isNoItem = (!this.feed.get('items').length) || (settings.values.hide_read) && (!this.feed.get('unread'));

        if (isNoItem) {
            this.noitem.hidden = false;
        } else {
            this.noitem.hidden = true;
        }
    }

});
