function parseDate(dateString){
    var date = Date.parse(dateString);

    if (isNaN(date)) {
        // Hack to fix a specific non-ISO date format.
        // "Mon, 07 Oct 2013 16:49:10 Z" = "D, d M Y H:i:s Z"
        dateString = dateString.substring(0, dateString.length - 1);
        date = Date.parse(dateString);
    }

    return date;
}

function FeedParser() {

    // Avoid clobbering the window scope:
    // return a new FeedParser object if we're in the wrong scope
    if ( !(this instanceof FeedParser) ) {
        return new FeedParser();
    }

    return this;

}

// data = httpRequest.responseText
FeedParser.prototype = {
    parse: function (data, feedModel) {

        var self = this;

        // Parse XML
        var xmlDoc = $.parseXML(data);
        var $xml   = $(xmlDoc);

        // Attach feedModel to this parser
        this.feed = feedModel;

        // Get local items
        this.newItems = [];

        // Check last date
        this.recent_date = 0;

        // Set feed title
        var title = this.parseFeedTitle($xml.find('title:first').text());

        if (this.feed.get('items').length !== 0)
        {
            this.recent_date = this.feed.get('items')[0].date;
        }

        // RSS
        $xml.find("item").each(function() {
            var allTags = this.getElementsByTagName("*");
            var xml_item = {
                "url": self.feed.get('url'),
                "read": false,
                "media" : []
            };

            for (var i = 0; i < allTags.length; i++) {

                switch (allTags[i].tagName) {
                    case "pubDate":
                        xml_item.date = parseDate(allTags[i].textContent);
                        break;
                    case "dc:date":
                        xml_item.date = parseDate(allTags[i].textContent);
                        break;

                    case "author":
                        xml_item.author = allTags[i].textContent;
                        break;
                    case "dc:creator":
                        xml_item.author = allTags[i].textContent;
                        break;

                    case "title":
                        xml_item.title = allTags[i].textContent;
                        break;
                    case "link":
                        xml_item.link = allTags[i].textContent;
                        break;
                    case "guid":
                        xml_item.id = allTags[i].textContent;
                        break;

                    case "description":
                        xml_item.description = $.trim(allTags[i].textContent);
                        break;
                    case "content:encoded":
                        xml_item.description = $.trim(allTags[i].textContent);
                        break;

                    case "enclosure":
                        var media = {
                            'url'    : allTags[i].getAttribute('url'),
                            'type'   : allTags[i].getAttribute('type'),
                            'length' : allTags[i].getAttribute('length')
                        };

                        xml_item.media.push(media);
                        break;

                    default:
                        break;
                }

            }

            self.add(xml_item);

        });

        // Atom
        $xml.find("entry").each(function() {
            var allTags = this.getElementsByTagName("*");
            var xml_item = {
                "url": self.feed.get('url'),
                "read": false
            };

            for (var i = 0; i < allTags.length; i++) {

                switch (allTags[i].tagName) {
                    case "updated":
                        var date = parseDate(allTags[i].textContent);

                        if (date) {
                            xml_item.date = date;
                        }
                        break;
                    case "published":
                        var date = parseDate(allTags[i].textContent);

                        if (date) {
                            xml_item.date = date;
                        }
                        break;

                    case "author":
                        xml_item.author = allTags[i].textContent;
                        break;
                    case "dc:creator":
                        xml_item.author = allTags[i].textContent;
                        break;

                    case "title":
                        xml_item.title = allTags[i].textContent;
                        break;
                    case "link":
                        xml_item.link = allTags[i].getAttribute('href');
                        break;
                    case "id":
                        xml_item.id = allTags[i].textContent;
                        break;

                    case "summary":
                        xml_item.summary = $.trim(allTags[i].textContent);
                        break;
                    case "content":
                        xml_item.description = $.trim(allTags[i].textContent);
                        break;

                    default:
                        break;
                }

            }

            self.add(xml_item);

        });

        if (this.newItems.length > 0) {
            // Concat and sort items
            var items = this.newItems.concat(this.feed.get('items').slice(0));
            items.sort(function(item1, item2) { return item2.date - item1.date; });

            // Tronc items to max history number
            if (items.length > settings.values.history)
            {
                items.length = settings.values.history;
            }

            // Compute unread numbers
            var unread = 0;
            for (var i=0, itemsLength = items.length; i < itemsLength; i++) {

                // If not read, increase unread number
                if (!items[i].read) {
                    unread++;
                }
            }

            // Finally, update feed model
            this.feed.set( {'title': title, 'items' : items, 'unread': unread} );

            // Notify timeline or items list
            Backbone.pubSub.trigger('timeline:notifyUpdate', this.newItems.slice(0));
            this.feed.trigger('notifyUpdate', this.newItems);
        }

        return this.newItems;
    },

    add: function(xml_item) {
        // If date exists, only append newer
        if (xml_item.date) {

            if (xml_item.date > this.recent_date) {
                this.newItems.push(xml_item);
            }

        } else {

            // If no date exists, only append new one
            var feedItems = this.feed.get('items');
            var alreadyExists = false;

            for (var j=0, length=feedItems.length; j<length; j++) {
                if (feedItems[j].link == xml_item.link) {
                    alreadyExists = true;
                    break;
                }
            }

            if (!alreadyExists) {
                this.newItems.push(xml_item);
            }

        }
    },

    parseFeedTitle: function(feedTitle) {
        var title = this.feed.get('title');
        var url   = this.feed.get('url');

        // If the title is the url, get title from feed
        if (title == url) {
            title = feedTitle;

            // Encode double quotes
            title = title.replace(/"/g,"&quot;");

            // Replace non-breaking char by a space
            title = title.replace(/\xA0/g," ");
        }

        return title;
    }

};


function OpmlParser() {

    // Avoid clobbering the window scope:
    // return a new OpmlParser object if we're in the wrong scope
    if ( !(this instanceof OpmlParser) ) {
        return new OpmlParser();
    }

    return this;

}

// data = httpRequest.responseText
OpmlParser.prototype = {
    parse: function (data) {
        var xmlDoc;
        var $xml;

        try {
            xmlDoc = $.parseXML(data);
            $xml   = $(xmlDoc);

            // Opml 1.0
            var outlines = $xml.find("outline");
            var outlineParsedNumber = 0;
            var self = this;

            if (!outlines) {
                console.error("Error parsing - empty file");
                showBanner("banner-opmlinvalid");

                return;
            }

            // Do not attack DB too fast
            var addOutline = function() {
                var outline = outlines[outlineParsedNumber];
                var xmlUrl  = outline.getAttribute('xmlUrl');
                var title   = outline.getAttribute('title') || outline.getAttribute('text') || xmlUrl;

                console.log(xmlUrl);

                // Do not attack DB too fast
                setTimeout(function() {

                    if (xmlUrl) {
                        feedList.add(
                            {
                                url   : xmlUrl,
                                title : title
                            }
                        );
                    }

                    showBanner("importing", "<br/><small>" + outlineParsedNumber + "/" + outlines.length + "</small>", true);

                    outlineParsedNumber++;

                    if (outlines.length == outlineParsedNumber) {
                        // Notify user
                        showBanner("banner-opmlimported");
                        console.log("banner-opmlimported");
                    } else {
                        addOutline();
                    }

                }, 1);

            };

            addOutline();

        } catch (e) {
            console.error("Error parsing");
            showBanner("banner-opmlinvalid");
        }

    },

};
