var Game = function() {
  // Set the width and height of the scene.
  this._width = window.innerWidth;
  this._height = window.innerHeight;

  // Setup the rendering surface.
  this.renderer = new PIXI.CanvasRenderer(this._width, this._height);
  this.renderer.backgroundColor = 0xffffff;
  document.body.appendChild(this.renderer.view);

  // Create the main stage to draw on.
  this.stage = new PIXI.Container();
  // setup our physics world simulation.
  this.world = new p2.World({ gravity: [0, 0] });

  this.enemies = [];
  this.inGame  = false;
  this.score   = 0;

  this.bestScore = localStorage.getItem("bestScore") || 0;

  // Start running the game.
  this.build();
};

Game.prototype = {
  /**
   * Build the scene and begin animating.
   */
  build: function() {
    // setup sounds to play with.
    this.setupAudio();
    // title screen object. 
    this.createtitleText();
    // create user object.
    this.createUser();
    // create score view
    this.createScore();
    // create best score view
    this.createBestScore()
    // start to add events.
    this.playEvents();
    // Begin the first frame.
    requestAnimationFrame(this.tick.bind(this));
  },

  createtitleText: function () {

    this.titleText = new PIXI.Text('Hold To Start',
      { 
        font : 'lighter 50px Arial',
        fill : 0x555555,
        align: 'center',
      });

    this.titleText.x = this._width / 2 - 295 / 2;
    this.titleText.y = this._height / 2 - 55 / 2 - 20;
  },

  createUser: function () {
    this.user = new p2.Body({
      position: [0, 0],
      mass: 1,
      damping: 0,
      angularDamping: 0,
      velocity: 0,
      angulerVelocity: 0
    });

    var userShape = new p2.Circle({ radius: 10 });
    userShape.sensor = true;
    this.user.addShape(userShape);
  },

  createScore: function () {
    this.scoreView = new PIXI.Text("Score " + this.score,
      { 
        font   : 'bold 16px Arial',
        fill   : 0x555555,
        align  : 'left',
      });

    this.scoreView.x = 10;
    this.scoreView.y = 5;
  },

  createBestScore: function () {
    this.bestScoreView = new PIXI.Text("Best Score " + this.bestScore,
      { 
        font   : '30px Arial',
        fill   : 0x555555,
        align  : 'center',
      });

    this.bestScoreView.x = this._width / 2 - 189 / 2;
    this.bestScoreView.y = this._height / 2 - 55 / 2 + 40;
  },

  addScreen: function () {
    this.stage.addChild(this.titleText);
    this.stage.addChild(this.bestScoreView);
  },

  removeScreen: function () {
    this.stage.removeChild(this.titleText);
    this.stage.removeChild(this.bestScoreView);
  },

  touchstart:function (e) {
    var x = e.touches[0].clientX,
        y = e.touches[0].clientY;

    this.user.position[0] = x;
    this.user.position[1] = y;

    this.startGame();
  },

  touchend: function (e) {
    this.endGame();
  },

  touchmove: function (e) {
    var x = e.touches[0].clientX,
        y = e.touches[0].clientY;

    this.user.position[0] = x;
    this.user.position[1] = y;
  },


  addEvents: function () {
    this.renderer.view.addEventListener('touchstart', this.touchstart, false);
    this.renderer.view.addEventListener('touchend', this.touchend, false);
    this.renderer.view.addEventListener('touchmove', this.touchmove, false);
  },

  removeEvents: function () {
    this.renderer.view.removeEventListener('touchstart', this.touchstart, false);
    this.renderer.view.removeEventListener('touchend', this.touchend, false);
    this.renderer.view.removeEventListener('touchmove', this.touchmove, false);
  },

  playEvents: function () {
    var self = this;

    this.touchstart = this.touchstart.bind(this);
    this.touchend   = this.touchend.bind(this);
    this.touchmove  = this.touchmove.bind(this);

    this.addScreen()
    this.stage.addChild(this.scoreView);

    this.addEvents();

    this.world.on('beginContact', function (e) {
      switch (self.user.id) {
        case e.bodyA.id:
        case e.bodyB.id:
          self.endGame();
      }
      self.crash.play();
    });
  },

  createEnemies: function () {
    var self = this;

    this.timeout = setInterval(function () {

      var ball = new Ball(self._width, self._height);
      self.world.addBody(ball.enemy);
      self.stage.addChild(ball.enemyGraphics);

      self.score += 1;
      self.updateScoreView();

      self.enemies.push(ball);
      
    }, 7000);

  },

  setupAudio: function () {
    this.crash = new Howl({ urls: ['sounds/pop.wav'] }); 
    this.music = new Howl({
      urls: ['sounds/coup.mp3'],
      buffer: true,
      loop: true,
      volume: 0.7
    }).play();
  },

  startGame: function () {
    if (!this.inGame) {
      this.inGame = true;

      this.removeScreen();

      this.score = 0;
      this.updateScoreView();

      this.world.addBody(this.user);
      this.createEnemies();
    } else {
      this.endGame();
    }
  },

  endGame: function () {
    var self = this;
    this.removeEvents();

    clearInterval(self.timeout);
    for (var i = 0, l = this.enemies.length; i < l; i++) {
      var enemyBody    = this.enemies[i].enemy,
          enemyGraphic = this.enemies[i].enemyGraphics;

      enemyBody.velocity[0] = 0;
      enemyBody.velocity[1] = 0;

    };

    self.inGame && self.world.removeBody(self.user);

    if (self.inGame) {

	    setTimeout(function () {
	      for (var i = 0, l = self.enemies.length; i < l; i++) {
	        var enemyBody    = self.enemies[i].enemy,
	            enemyGraphic = self.enemies[i].enemyGraphics;

	        self.stage.removeChild(enemyGraphic);
	        self.world.removeBody(enemyBody);

	      };

	      self.inGame && self.addScreen();

	      self.enemies.length = 0;
	      self.inGame = false;
	      self.addEvents();
	    }, 2000);
    } else {
    	self.addEvents();
    }

  },

  updateScoreView: function () {
    if (this.score > this.bestScore) {
      this.bestScore = this.score;
      localStorage.setItem("bestScore", this.bestScore);
      this.bestScoreView.text = "Best Score " + this.bestScore;
    }

    this.scoreView.text = "Score " + this.score;
    this.stage.removeChild(this.scoreView);
    this.stage.addChild(this.scoreView);
  },

  updatePhysics: function () {
    for (var i = 0, l = this.enemies.length; i < l; i++) {
      var enemyBody    = this.enemies[i].enemy,
          enemyGraphic = this.enemies[i].enemyGraphics;

      switch (true) {
        case enemyBody.position[0] > this._width: enemyBody.position[0] = 0; break;
        case enemyBody.position[0] < 0: enemyBody.position[0] = this._width; break;
        case enemyBody.position[1] > this._height: enemyBody.position[1] = 0; break;
        case enemyBody.position[1] < 0: enemyBody.position[1] = this._height; break;
      }

      enemyGraphic.x = enemyBody.position[0];
      enemyGraphic.y = enemyBody.position[1];
    };

    this.world.step(1 / 60);
  },

  /**
   * Fires at the end of the gameloop to reset and redraw the canvas.
   */
  tick: function() {
    // update physics
    this.updatePhysics();

    // Render the stage for the current frame.
    this.renderer.render(this.stage);

    // Begin the next frame.
    requestAnimationFrame(this.tick.bind(this));
  }
};