speedread.sectionMain = function($section) {
	var $header = $section.querySelectorAll("header")[0];
	var $article = $section.querySelectorAll("article")[0];
	var $footer = $section.querySelectorAll("footer")[0];
	
	var $title = $header.querySelectorAll(".title")[0];
	var $prev = $article.querySelectorAll("div.prev p")[0];
	var $word = $article.querySelectorAll(".word")[0];
	var $next = $article.querySelectorAll("div.next p")[0];
	var $search = $article.querySelectorAll(".search")[0];
	var $passed = $section.querySelectorAll(".pos .passed")[0];
	var $ahead = $section.querySelectorAll(".pos .ahead")[0];
	var $progress = $section.querySelectorAll(".pos progress")[0];
	var $pos = $section.querySelectorAll(".pos input")[0];
	var $bookmark = $footer.querySelectorAll(".bookmark")[0];
	var $searchInput = $search.querySelectorAll("input")[0];
	var $speed = $section.querySelectorAll(".speed .value")[0];

	var file, text, pos, pageSize = 1024, minChars, maxChars, durationByLength, totalDuration;
	var timer = null;
	var reSpace = /[ \f\v\t\n]/;
	var reNSpace = /[^ \f\v\t\n]/;

	function updateWord() {
		for (var end = pos; end < text.length && reNSpace.test(text.charAt(end)); end++);
		$progress.value = pos;
		$pos.value = pos;
		var t = Math.round(totalDuration * pos / text.length) || 0;
		var m = t % 60;
		var h = Math.floor(t / 60);
		$passed.textContent = h + ":" + (m < 10 ? "0" : "") + m;
		t = totalDuration - t;
		m = t % 60;
		h = Math.floor(t / 60);
		$ahead.textContent = h + ":" + (m < 10 ? "0" : "") + m;
		$word.textContent = text.substring(pos, end);
		return end;
	}
	
	function updateText() {
		var end = updateWord();
		$prev.textContent = text.substring(Math.max(0, pos - pageSize), pos);
		$next.textContent = text.substring(end, Math.min(text.length, end + pageSize));
		localStorage.setItem(file + ".pos", pos);
	}
	
	function updateSpeed() {
		var speed = Number(localStorage.speed);
		var totalWordCount = 0;
		var totalChars = 0;
		var countByLength = {};
		for (var l = 1; l <= 32; l++) {
			var length = Math.max(minChars, Math.min(maxChars, l));
			var wordCount = Number(localStorage.getItem("word-count-" + l)) || 0;
			countByLength[length] = (countByLength[length] || 0) + wordCount;
			totalWordCount += wordCount;
			totalChars += length * wordCount;
		}
		var avgLength = totalWordCount > 0 ? totalChars / totalWordCount : 0;
		durationByLength = avgLength > 0 ? 60000 / speed / avgLength : 0;
		totalDuration = 0;
		for (var length in countByLength) {
			var duration = Math.round(length * durationByLength);
			totalDuration += countByLength[length] * duration;
		}
		totalDuration = Math.ceil(totalDuration / 60000);
		$speed.textContent = localStorage.speed + "\nw/min";
	}

	function updateSettings() {
		minChars = Number(localStorage.minChars);
		maxChars = Number(localStorage.maxChars);
		updateSpeed();
		if (localStorage.theme === "default") {
			$section.style.background = "rgb(" + localStorage.defaultContrast + "," + localStorage.defaultContrast + "," + localStorage.defaultContrast + ")";
			$section.style.color = "black";
			$word.style.color = "black";
		} else {
			$section.style.background = "black";
			$section.style.color = "rgb(" + localStorage.darkContrast + "," + localStorage.darkContrast + "," + localStorage.darkContrast + ")";
			$word.style.color = "rgb(" + localStorage.darkContrast + "," + localStorage.darkContrast + "," + localStorage.darkContrast + ")";
		}
		$prev.style.fontSize = localStorage.textSize + "px";
		$next.style.fontSize = localStorage.textSize + "px";
		$word.style.fontSize = localStorage.wordSize + "px";
		if (text) updateWord();
	}
	
	function updateFile() {
		file = localStorage.file || "";
		text = localStorage.text || "";
		pos = Number(localStorage.getItem(file + ".pos")) || 0;
		updateSpeed();
		$bookmark.textContent = localStorage.getItem(file + ".bookmark") !== null ? "\ue614" : "\ue613";
		var title = "SpeedRead";
		if (file) {
			var parts = file.split("/");
			title = parts[parts.length - 1].replace(/\.txt$/, "");
		}
		$title.textContent = title;
		$progress.max = text.length;
		$pos.max = text.length;
		updateText();
	}
	
	$pos.addEventListener("change", function() {
		updateText();
	}, false);

	$pos.addEventListener("mouseup", function() {
		updateText();
	}, false);

	$pos.addEventListener("input", function() {
		pos = $pos.value;
		if (reNSpace.test(text.charAt(pos))) {
			while (pos > 0 && reNSpace.test(text.charAt(pos-1))) pos--;
		}
		var end = updateWord();
		$prev.textContent = "";
		$next.textContent = "";
	}, false);
	
	$pos.addEventListener("click", function(e) {
		e.stopPropagation();
	}, false);

	function show() {
		var end = updateWord();
		if (end < text.length) {
			var length = Math.max(minChars, Math.min(maxChars, end - pos));
			var timeout = Math.round(length * durationByLength);
			timer = setTimeout(showNext, timeout);
		} else {
			updateText();
			$section.classList.remove("run");
		}
	}
	
	function showNext() {
		while (pos < text.length && reNSpace.test(text.charAt(pos))) pos++;
		while (pos < text.length && reSpace.test(text.charAt(pos))) pos++;
		show();
	}

	$article.querySelectorAll("div.prev")[0].addEventListener("click", function() {
		var size = pageSize;
		var maxHeight = $prev.parentElement.clientHeight - $word.offsetHeight / 2;
		if ($prev.offsetHeight > maxHeight) {
			var minSize = 0;
			var maxSize = pageSize;
			for (;;) {
				size = Math.floor((maxSize + minSize) / 2);
				if (size === minSize) break;
				$prev.textContent = text.substring(Math.max(0, pos - size), pos);
				if ($prev.offsetHeight > maxHeight) {
					maxSize = size;
				} else {
					minSize = size;
				}
			}
		}
		pos = Math.max(0, pos - size);
		if (pos > 0) {
			while (pos < text.length - 1 && reNSpace.test(text.charAt(pos))) pos++;
		}
		updateText();
	}, false);

	$article.querySelectorAll("div.next")[0].addEventListener("click", function() {
		var size = pageSize;
		var maxHeight = $next.parentElement.clientHeight - $word.offsetHeight / 2;
		for (var end = pos; end < text.length && reNSpace.test(text.charAt(end)); end++);
		if ($next.offsetHeight > maxHeight) {
			var minSize = 0;
			var maxSize = pageSize;
			for (;;) {
				size = Math.floor((maxSize + minSize) / 2);
				if (size === minSize) break;
				$next.textContent = text.substring(end, Math.min(text.length, end + size));
				if ($next.offsetHeight > maxHeight) {
					maxSize = size;
				} else {
					minSize = size;
				}
			}
		}
		pos = Math.min(text.length, end + size);
		while (pos >= 0 && reNSpace.test(text.charAt(pos))) pos--;
		updateText();
	}, false);

	$footer.querySelectorAll("button.start")[0].addEventListener("click", function(e) {
		pos = 0;
		updateText();
		e.stopPropagation();
	}, false);

	$footer.querySelectorAll("button.search")[0].addEventListener("click", function(e) {
		$search.classList.toggle("open");
		if ($search.classList.contains("open")) {
			$searchInput.focus();
		}
		e.stopPropagation();
	});

	$search.querySelectorAll(".prev")[0].addEventListener("click", function(e) {
		var search = $searchInput.value;
		if (search) {
			var i = text.substring(0, pos).toLowerCase().lastIndexOf(search.toLowerCase());
			if (i >= 0) {
				pos = i;
				updateText();
			} else {
				alert("Not found");
			}
		}
		e.stopPropagation();
	}, false);
	
	function searchNext(e) {
		var search = $searchInput.value;
		if (search) {
			var re = new RegExp(search.replace(/\s+/g, " ").replace(/([^0-9A-Za-z ])/g, "\\$1").replace(" ", "\\s+"), "im");
			var res = re.exec(text.substring(pos + 1));
			if (res) {
				pos = pos + 1 + res.index;
				updateText();
			} else {
				alert("Not found");
			}
		}
		e.stopPropagation();
	}
	
	$search.querySelectorAll(".next")[0].addEventListener("click", searchNext, false);
	
	$search.addEventListener("click", function(e) {
		e.stopPropagation();
	}, false);
	
	$searchInput.addEventListener("keydown", function(e) {
		if (e.which === 13) {
			searchNext(e);
			e.preventDefault();
		}
	}, false);
	
	var longPressTimer = null;
	function bookmarkDown(e) {
		$bookmark.classList.add("active");
		longPressTimer = setTimeout(function() {
			console.log("long");
			longPressTimer = null;
			var bookmark = localStorage.getItem(file + ".bookmark");
			if (bookmark !== null) {
				localStorage.removeItem(file + ".bookmark");
				$bookmark.textContent = "\ue613";				
				$bookmark.classList.remove("active");
			}
		}, 1000);
		e.stopPropagation();
		e.preventDefault();
	}
	function bookmarkUp(e) {
		if (longPressTimer !== null) {
			clearTimeout(longPressTimer);
			longPressTimer = null;
		}
		if ($bookmark.classList.contains("active")) {
			$bookmark.classList.remove("active");
			var bookmark = localStorage.getItem(file + ".bookmark");
			if (bookmark !== null) {
				bookmark = Number(bookmark);
				if (pos === bookmark) {
					localStorage.removeItem(file + ".bookmark");
					$bookmark.textContent = "\ue613";
				} else {
					pos = bookmark;
					updateText();
				}
			} else {
				localStorage.setItem(file + ".bookmark", pos);
				$bookmark.textContent = "\ue614";
			}
		}
		e.stopPropagation();
		e.preventDefault();
	}
	$bookmark.addEventListener("touchstart", bookmarkDown, false);
	$bookmark.addEventListener("mousedown", bookmarkDown, false);
	$bookmark.addEventListener("touchend", bookmarkUp, false);
	$bookmark.addEventListener("mouseup", bookmarkUp, false);

	var defaultSpeed;
	$footer.querySelectorAll("button.play")[0].addEventListener("click", function(e) {
		if (text) {
			defaultSpeed = localStorage.speed;
			if (localStorage.theme === "dark") {
				var c = Math.round(localStorage.darkContrast / 2);
				$section.style.color = "rgb(" + c + "," + c + "," + c + ")";
			}
			$section.classList.add("run");
			clearTimeout(timer);
			show();
		}
		e.stopPropagation();
	}, false);
	
	var $helpDialog = document.querySelector(".dialog.help");
	$helpDialog.querySelectorAll("header button.close")[0].addEventListener("click", function() {
		speedread.closeDialog($helpDialog);
	}, false);
	
	$footer.querySelectorAll("button.help")[0].addEventListener("click", function(e) {
		speedread.openDialog($helpDialog);
		e.stopPropagation();
	}, false);

	function speedInc() {
		if (localStorage.speed < 100) {
			return 5;
		}
		if (localStorage.speed < 500) {
			return 10;
		}
		if (localStorage.speed < 1000) {
			return 25;
		}
		return 50;
	}

	$section.querySelectorAll(".speed .slower")[0].addEventListener("click", function(e) {
		localStorage.speed = Math.max(10, Math.round(Number(localStorage.speed) - speedInc()));
		updateSpeed();
		e.stopPropagation();
	}, false);

	$section.querySelectorAll(".speed .faster")[0].addEventListener("click", function(e) {
		localStorage.speed = Math.min(5000, Math.round(Number(localStorage.speed) + speedInc()));
		updateSpeed();
		e.stopPropagation();
	}, false);

	$section.querySelectorAll(".speed .value")[0].addEventListener("click", function(e) {
		localStorage.speed = defaultSpeed;
		updateSpeed();
		e.stopPropagation();
	}, false);

	function stop() {
		if (timer) {
			clearTimeout(timer);
			timer = null;
			if (localStorage.theme === "dark") {
				$section.style.color = "rgb(" + localStorage.darkContrast + "," + localStorage.darkContrast + "," + localStorage.darkContrast + ")";
			}
			updateText();
			$section.classList.remove("run");
			if (defaultSpeed !== localStorage.speed) {
				speedread.trigger("settings");
			}
		}
	}
	$section.addEventListener("click", stop, false);
	document.addEventListener("visibilitychange", stop, false);
	
	speedread.on("settings", updateSettings);
	speedread.on("file", updateFile);
	updateSettings();
	updateFile();
};

