/*

noteList : array of note object

note :
- type : "checklist_note" or "text_note"
- title : text
- content : content of a text note, null if not a text note
- contentChecklist : null if text note, else array of checklist element

checklist element : 
- content : text
- checkeck : boolean
*/

var noteList = [];

var baseUrl = "https://android.damongeot.fr/note-webserv";

/**
	* load noteList object from disk
	* @return true if sucessfully loaded, false if object does not exists
	*/
function loadNoteList() {
	//load noteList from disk
	var foo = localStorage.getItem('noteList');
	if(foo) {
		noteList = JSON.parse(foo);
		return true;
	}

	return false;
}

/**
	* Save global noteList object to disk
	*/
function saveNotelist() {
	//console.log('saveNotelist()');
	localStorage.setItem('noteList',JSON.stringify(noteList));

	//update last change date
	localStorage.setItem('lastChange',Date.now()/1000);
}

/**
	*	Get parameter value from query string (URL)
	*/
function getParameterByName(name) {
    name = name.replace(/[\[]/, "\\[").replace(/[\]]/, "\\]");
    var regex = new RegExp("[\\?&]" + name + "=([^&#]*)"),
        results = regex.exec(location.search);
    return results === null ? "" : decodeURIComponent(results[1].replace(/\+/g, " "));
}

/**
	* Escape special HTML char to use them in inputs
	*/
function htmlEncode(string) {
	var entityMap = {
		"&": "&amp;",
		"<": "&lt;",
		">": "&gt;",
		'"': '&quot;',
		"'": '&#39;',
		"/": '&#x2F;'
	};
	return String(string).replace(/[&<>"'\/]/g, function (s) {
			return entityMap[s];
			});
}

function htmlDecode(string){
	var entityMap = {
		"&amp;":"&",
		"&lt;":"<",
		"&gt;":">",
		'&quot;':'"',
		'&#39;':"'",
		'&#x2F;':"/"
	};
	return String(string).replace(/&amp;|&lt;|&gt;|&quot;|&#39;|&#x2F/g, function (s) {
			return entityMap[s];
			});
}

/**
 	* Log into server
	* @param callback function(boolean) to call where boolean is true if login ok, false otherwise
	*/
function login(callback) {
	if(! window.navigator.onLine) {
		updateCloudStatus("offline");
		return;
	}

	updateCloudStatus("connecting");

	var deviceId = getDeviceId();

	var loginResponse = function(callback) {
		return function(data, textStatus, jqXHR) {
			//console.log("app.js/loginResponse():");
			//console.log(data);
			if(data.match(/no matching device id/)) {
				register(deviceId,callback);
			} else if(data.match(/ok/)) {
				updateCloudStatus("ok");
				callback(true);
			} else {
				updateCloudStatus("error");
				callback(false);
			}
		};
	};

	$.ajax({
		type: "POST",
		url: baseUrl+"/ajax/device.php",
		xhrFields: {
			withCredentials: true
		},
		data: { device_type: "firefox", login: deviceId }
		})
		.always(loginResponse(callback));
}

function register(deviceId,callback) {
	var registerResponse = function(callback) {
		return function(data, textStatus, jqXHR) {
			console.log("app.js/loginResponse():");
			//console.log(data);
			if(data.match(/ok/)) {
				login(callback);
			}
		};
	};

	$.ajax({
		type: "POST",
		url: baseUrl+"/ajax/device.php",
		xhrFields: {
			withCredentials: true
		},
		data: { device_type: "firefox", register: deviceId }
		})
		.always(registerResponse(callback));
}

/**
	* get a unique device identifier
	* As Firefox OS doesnt allow us to get a unique identifier for a device
	* we genrate a random number and store it on disk
	*/
function getDeviceId() {
	var foo = localStorage.getItem('deviceId');
	if(foo) {
		//console.log('loaded device id from disk: '+foo);
		return foo;
	}

	//generate random string
	//fill an array of int with random numbers
	var array = new Uint32Array(15);
	window.crypto.getRandomValues(array);

	//convert them to base64 and concat them into one big string
	var base64text='';
	for (var i = 0; i < array.length; i++) {
		base64text += btoa(array[i]);
		//console.log("random number : "+array[i]+", base64: "+btoa(array[i]));
	}

	var deviceId = sha512(base64text);

	//console.log(base64text);
	//console.log(deviceId);

	localStorage.setItem('deviceId',deviceId);

	return deviceId;
}

/**
	* Check note synchronization with server
	* @param forceDownload Force synchronization by downloading whats on server
	*/
function checkSync(forceDownload) {
	if(! window.navigator.onLine) {
		updateCloudStatus("offline");
		return;
	}

	updateCloudStatus("connecting");

	//check that local last server sync is equal to last local modification
	var lastChange = localStorage.getItem('lastChange');
	if(lastChange && lastChange != getLastSync()) {
		console.log('Uploading notes : lastChange: '+lastChange+", lastSync: "+getLastSync());
		uploadNotes();
		return;
	}

	$.ajax({
		type: "GET",
		url: baseUrl+"/ajax/user.php",
		xhrFields: {
			withCredentials: true
		},
		data: { getLastSync: "" }
		})
		.always(function checkSyncResponse(data,textStatus, jqXHR) {
			//console.log("checkSyncResponse():");
  		//console.log(data);
			if(data.match(/[0-9]+-[0-9]+-[0-9]+ [0-9]+:[0-9]+:[0-9]+/)) {
				var serverTs = sqlToTimestamp(data);
				var localTs = getLastSync();

				if(forceDownload) {
					downloadNotes(serverTs);
					return;
				}

				console.log("lastSync "+localTs+", server last sync is "+serverTs);
				if(Math.abs(localTs-serverTs) > 120) { //time differs by more than 2 minutes
					if(localTs<serverTs) {
						console.log("checkSync() : we need to fetch notes from server");
						downloadNotes(serverTs);
					} else {
						console.log("checkSync() : uploading notes to server");
						uploadNotes();
					}
				} else {
					updateCloudStatus("ok");
				}
			} else {
				updateCloudStatus("error");
			}
	});
}

/**
	* Download note from server
	* @param serverLastSync Timestamp of the last sync note list from server
	*/
function downloadNotes(serverLastSync) {
	updateCloudStatus("connecting");

	$.ajax({
    type: "GET",
    url: baseUrl+"/ajax/note.php",
    xhrFields: {
      withCredentials: true
    },
    data: { download: '' }
    })
    .always(function (data,textStatus, jqXHR) {
			console.log('downloadNotes() : ');
			var ja = JSON.parse(data);
			//console.log(ja);
			var newNoteList = [];
			for(var i=0;i<ja.length;i++) {
				var note = {
					type : ja[i].type,
					title : ja[i].title,
					content : '',
					contentChecklist : []
				};
				if(ja[i].type=='text_note') {
					note.content=ja[i].content;
				} else {
					// true-false value are not interpreted correctly so we process them ourselves
					for(var j=0;j<ja[i].content.length;j++) {
						note.contentChecklist[j] = {
							content : ja[i].content[j].content,
							checked : ja[i].content[j].checked=='true' ? true : false
						}
					}
				}
				newNoteList.push(note);
			}
			//console.log('new note list:');
			//console.log(newNoteList);
			noteList = newNoteList;
			saveNotelist();
			console.log('downloadNotes() : server last sync=> '+serverLastSync);
			localStorage.setItem('lastSync',serverLastSync);
			localStorage.setItem('lastChange',serverLastSync);
			updateCloudStatus("ok");
			syncCompleted('download');
		});

}

/**
	* Upload notes to server
	*/
function uploadNotes() {
	updateCloudStatus("connecting");

	$.ajax({
    type: "POST",
    url: baseUrl+"/ajax/note.php",
    xhrFields: {
      withCredentials: true
    },
    data: { upload: JSON.stringify(noteList), last_modification: timestampToSql(localStorage.getItem('lastChange')) }
    })
    .always(function (data,textStatus, jqXHR) {
			if(data.match(/ok/)) {
				localStorage.setItem('lastSync',localStorage.getItem('lastChange'));
				console.log('uploadNotes() : success');
				console.log('lastChange:'+localStorage.getItem('lastChange')+', lastSync:'+getLastSync());
				updateCloudStatus("ok");
				syncCompleted('upload');
			} else {
				console.log('uploadNotes() : error => '+data);
				updateCloudStatus("error");
			}
		});
}

/**
  * This function upload notes to server but wait for a delay before doing so.
  * Its usefull for example when editing notes : we dont want to upload notes after each modification but after a certain amoun
t of time so the user should have finished making modifications to notes.
  */
var timeoutIdDelayedUpload;
function delayedUpload() {
	if(! window.navigator.onLine) {
		return;
	}
	if(timeoutIdDelayedUpload) window.clearTimeout(timeoutIdDelayedUpload);

	timeoutIdDelayedUpload = window.setTimeout(uploadNotes, 10000); //10s delay
}


/**
	* Get timestamp of the last sync date
	*/
function getLastSync() {
	var foo = localStorage.getItem('lastSync');
	if(foo) {
		//console.log('last saved sync is : '+timestampToSql(foo));
		return foo;
	}

	//server init at year 2000, we init just after to get sync precedence
	var date = new Date(2001,01,01);
	console.log('last created date sync is : '+timestampToSql(date/1000));
	return date.getTime()/1000;
}

/**
	* Convert timestamp (in seconds) to SQL datetime
	*/
function timestampToSql(ts) {
	var date = new Date(ts*1000);
	return date.getFullYear()+"-"+date.getMonth()+"-"+date.getDate()+" "+date.getHours()+":"+date.getMinutes()+":"+date.getSeconds();
}

/**
	* same as above but in the other way
	*/
function sqlToTimestamp(sql) {
	var re = /([0-9]+)-([0-9]+)-([0-9]+) ([0-9]+):([0-9]+):([0-9]+)/;
	var arr = sql.match(re);
	if(arr.length<7) {
		console.log("sqlToTimestamp() : invalid date =>"+sql);
		return null;
	}
	var date = new Date(arr[1],arr[2],arr[3],arr[4],arr[5],arr[6]);
	return date.getTime()/1000;
}

function updateCloudStatus(cloudStatus) {
	if(! $("img[name=cloudIcon]").length) { //if we're on a note or checklist, cloud icon is not shown
		return;
	}
	var src = $("img[name=cloudIcon]").attr('src').match(/^[^\/]+/) + "/";

	switch(cloudStatus) {
		case "connecting":
			$("img[name=cloudIcon]").attr('src',src+"cloud_connecting.png");
			$("#cloudStatusText").html(document.webL10n.get('connecting'));
			break;
		case "ok":
			$("img[name=cloudIcon]").attr('src',src+"cloud_ok.png");
			$("#cloudStatusText").html(document.webL10n.get('synchronized'));
			break;
		case "error":
			$("img[name=cloudIcon]").attr('src',src+"cloud_error.png");
			$("#cloudStatusText").html(document.webL10n.get('synchronization-error'));
			break;
		case "offline":
			$("img[name=cloudIcon]").attr('src',src+"cloud_error.png");
			$("#cloudStatusText").html(document.webL10n.get('offline'));
			break;
		default:
			console.log("updateCloudStatus():invalid cloud status:"+cloudStatus);
	}
}

/**
	* Returns the number unchecked item in a checklist note
	* @param noteIndex index in noteList for the checklist note
	*/
function getNumberOfUncheckedItemInNote(note) {
	var uncheckedCount=0;

	for(var i=0;i<note.contentChecklist.length;i++) {
		if(! note.contentChecklist[i].checked) uncheckedCount++;
	}

	return uncheckedCount;
}
