function bind(opts) {
    return new function () {
        var el = $('*[data-display="' + opts.name + '"]');
        var model = opts.model;
        var property = opts.property;
        var source = opts.source;

        this.update = function (d) {
            var index = model[property];

            if (index + d >= 0 && index + d < source.length) {
                model[property] += d;

                el.text(source[model[property]].label);
            }
        }

        this.prev = function () {
            this.update(-1);
        }

        this.next = function () {
            this.update(1);
        }

        this.update(0);
    };
}

function listen(attr, event, callback) {
    $('*[' + attr + ']').on(event, function (ev) {
        ev.preventDefault();

        callback(ev, $(this).attr(attr));
    });
}

var Loader = new function () {
    var wrapper = '<div class="loader"><p class="title"></p><p class="description"></p><div class="progress"><span class="bar"></span></div></div>';
    var element = null;

    this.show = function (title, text, callback) {
        if (!element) {
            element = $(wrapper);
            element.hide().appendTo($('body'));
        }

        element.find('.title').text(title);
        element.find('.description').text(text);
        element.find('.bar').css('right', '100%');

        element.delay(100).fadeIn(400, callback);

        return this;
    };

    this.update = function (percentage, title, text) {
        //console.log(percentage);

        if (title)
            element.find('.title').text(title);
        if (text)
            element.find('.description').text(text);
        if (percentage)
            element.find('.bar').css('right', (100 - percentage) + '%');

        return this;

        //console.log(element.find('.bar').css('right'));
    };

    this.hide = function (callback) {
        element.delay(100).fadeOut(400, callback);

        return this;
    };
}();

/**
 * Created by Sud on 2014.09.28..
 */


/*
 !!!

 próbálkozások (szavak) rögzítése
 ->
 lehetőség játék után: próbálkozások listázása ebből kiválaszthatóak szavak: megjelölhetőek mint nem létező vagy hibás szavak
 ezeket el lehet küldeni szerverre
 */

function Game() {
    this.level = 1;
    this.time = 1;

    this.levels = [
        {label: "2x2", size: 2},
        {label: "3x3", size: 3},
        {label: "4x4", size: 4}
        //{label: "5x5", size: 5},
        //{label: "6x6", size: 6}
    ];

    this.times = [
        {label: "1:00", seconds: 60},
        {label: "2:00", seconds: 60 * 2},
        {label: "3:00", seconds: 60 * 3},
        {label: "4:00", seconds: 60 * 4},
        {label: "∞", seconds: false}
    ];
}

Game.prototype.load = function () {
    if (localStorage.game) {
        var game = JSON.parse(localStorage.game);

        this.level = game.level;
        this.time = game.time;
    }
};

Game.prototype.save = function () {
    localStorage.game = JSON.stringify({
        level: this.level,
        time: this.time
    });
};


Game.prototype.createSession = function (puzzle, time, seconds) {
    return new Game.prototype.createSessionClass(puzzle, time, seconds);
};

Game.prototype.createSessionClass = function (puzzle, time, seconds) {
    this.puzzle = puzzle;
    this.found = [];
    this.countTable = {};
    this.time = time;
    this.seconds = seconds;

    if (this.seconds) {
        this.endTime = new Date();
        this.endTime.setSeconds(this.endTime.getSeconds() + seconds);
    }
    else {
        this.endTime = false;
    }

    for (var i in this.puzzle.words) {
        var len = this.puzzle.words[i].length;
        if (len >= 8)
            len = '8p';

        if (typeof this.countTable[len] == 'undefined')
            this.countTable[len] = 0;

        this.countTable[len]++;
    }
};

Game.prototype.createSessionClass.prototype.checkWord = function (word, info) {
    var hasIt = this.puzzle.words.indexOf(word) >= 0;

    info.isNew = false;
    info.isLast = false;

    if (hasIt && this.found.indexOf(word) < 0) {
        this.found.push(word);

        this.countTable[word.length]--;

        info.isNew = true;

        if (this.found.length == this.puzzle.words.length)
            info.isLast = true;
    }

    return hasIt;
};

Game.prototype.createSessionClass.prototype.getSecondsLeft = function () {
    var secs = false;
    if (this.endTime) {
        secs = Math.ceil((this.endTime.getTime() - (new Date().getTime())) / 1000);

        if (secs < 0)
            secs = 0;
    }

    return secs;
};

Game.prototype.createSessionClass.prototype.getTimeLeft = function () {
    var seconds = this.getSecondsLeft();

    if (seconds === false)
        return "--:--";

    var minutes = ("00" + Math.floor(seconds / 60)).slice(-2);
    var seconds = ("00" + seconds % 60).slice(-2);

    return minutes + ':' + seconds;
};

console.log = console.log || console.info || function () {
};

// worker inicializálása
var generator = new Worker('js/worker.js');
var game = new Game();

// játék inicializálása

function initialize() {
    $('.screen').hide();

    $('.screen.menu').fadeIn();

    // játék inicializálása
    game.load();

    // szótár betöltése

    Loader.show('Szótár betöltése', 'Fájl keresése...', function () {
        generator.onmessage = function (event) {
            if (event.data.name == 'loadDictionaryProgress') {
                Loader.update(event.data.progress, null, event.data.message);
            }
            else {
                Loader.update(100).hide();
            }
        };

        generator.postMessage({
            name: 'loadDictionary',
            url: '../data/hu-0.2.json'
        });
    });

    // felületi elemek kötése eseményekhez

    var model = {
        level: bind({
            name: 'level',
            model: game,
            property: "level",
            source: game.levels
        }),

        time: bind({
            name: 'time',
            model: game,
            property: "time",
            source: game.times
        })
    };

    listen('data-decrease', 'click', function (event, value) {
        model[value].prev();

        game.save();
    });

    listen('data-increase', 'click', function (event, value) {
        model[value].next();

        game.save();
    });

    listen('data-call', 'click', function (event, value) {
        if (typeof window[value] == 'function') {
            window[value]();
        }
    });

    $('.screen.menu .help').on('click', function (event) {
        event.preventDefault();

        $('.screen.menu').fadeOut(300, function() {
            $('.screen.help').fadeIn(300);
        });
    });

    $('.screen.help .back').on('click', function (event) {
        event.preventDefault();

        $('.screen.help').fadeOut(300, function() {
            $('.screen.menu').fadeIn(300);
        });
    });

}

// új játék indítása

function startGame() {
    Loader.show('Generálás', 'Előkészítés...', function () {
        generator.onmessage = function (event) {
            if (event.data.name == 'generatePuzzleProgress') {
                Loader.update(event.data.progress, null, event.data.message);
            }
            else {
                $('.screen.menu').fadeOut();

                Loader.update(100).hide(function () {
                    setupLevel(event.data.puzzle);
                });
            }
        };

        generator.postMessage({
            name: 'generatePuzzle',
            size: game.levels[game.level].size
        });
    });
}

// pálya felkonfigurálása

function setupLevel(puzzle) {
    $('.screen.level').fadeIn();

    $('.letters').children().remove();
    $('.level .found-list').text('');

    var session = game.createSession(puzzle, game.time, game.times[game.time].seconds);

    for (var i in puzzle.map) {
        $('.letters').append($('<div class="letter"><span>' + puzzle.map[i] + '</span></div>'));
    }

    $(window).off().on('resize', function () {
        resizeLetters(session)
    }).trigger('resize');

    var dragging = false;
    var selection = [];
    var completed = false;

    $('.letters').off('mousedown touchstart').on('mousedown touchstart', function (event) {
        event.preventDefault();

        dragging = true;
    });

    $('.letters').off('mouseup touchend').on('mouseup touchend', function (event) {
        event.preventDefault();

        if (selection.length > 1) {

            var word = "";
            for (var i in selection) {
                var char = $(selection[i]).text();

                word += char;
            }

            var result = {};
            var hasIt = session.checkWord(word, result);

            if (!hasIt) {
                $('.message span').stop(true).hide().removeClass('success').text(word).fadeIn(100, function () {
                    $(this).fadeOut(3000);
                });
            }
            else {
                $('.message span').stop(true).hide().addClass('success').text(word).fadeIn(100, function () {
                    $(this).fadeOut(3000);
                });

                if (result.isNew) {
                    var text = $('.level .found-list').text();

                    if (!text)
                        $('.level .found-list').text(word)
                    else
                        $('.level .found-list').text(text + ', ' + word);

                    window.navigator.vibrate(200);
                }

                if (result.isLast) {
                    showScore(session);
                    completed = true;
                }
            }

            updateWords(session);
        }

        dragging = false;
        $('.letters .letter span').removeClass('active');
        selection.length = 0;
    });


    $('.letters').off('mouseleave touchcancel').on('mouseleave touchcancel', function (event) {
        event.preventDefault();

        dragging = false;
        $('.letters .letter').removeClass('active');
        selection.length = 0;
    });


    var target = null;
    $('.letters').on('touchmove', function (event) {
        event.preventDefault();
        event.stopPropagation();

        var potentialTarget = $(document.elementFromPoint(event.originalEvent.touches[0].clientX, event.originalEvent.touches[0].clientY));

        if (target != potentialTarget) {
            if (potentialTarget.parent().hasClass('letter')) {
                target = potentialTarget;
                target.trigger('touchenter');
            }
            else {
                target = null;
            }
        }
    });

    $('.letters .letter span').on('mousedown touchstart', function () {
        $(this).toggleClass('active');
        dragging = true;
        selection.push(this);
    });

    $('.letters .letter span').on('mouseenter touchenter', function () {
        if (dragging) {
            var index = selection.indexOf(this);
            if (index >= 0) {
                if (index == selection.length - 2) {
                    $(selection[index + 1]).removeClass('active');
                    selection.splice(index + 1, 1);
                }
            }
            else {
                // csak szomszédosak lehetnek!
                var found = selection.length == 0 ? true : false;

                if (!found) {
                    var index = $(selection[selection.length - 1]).parent().index();
                    var cX = index % puzzle.size;
                    var cY = Math.floor(index / puzzle.size);

                    for (var y = -1; y <= 1; y++) {
                        for (var x = -1; x <= 1; x++) {
                            if (cX + x < 0 || cX + x >= puzzle.size || cY + y < 0 || cY + y >= puzzle.size)
                                continue;

                            var nIndex = (cY + y) * puzzle.size + cX + x;

                            if (nIndex == $(this).parent().index())
                                found = true;
                        }
                    }
                }

                if (found) {
                    $(this).addClass('active');
                    selection.push(this);
                }
            }
        }
    });

    $('.quit').off('click').on('click', function (event) {
        event.preventDefault();

        if (confirm('Biztosan feladod?')) {
            completed = true;
            showScore(session);
        }
    });

    $('table.words td[class*="len-"]').text('-');
    updateWords(session);


    var timer = function () {
        $('.level .timer').text(session.getTimeLeft());

        if (session.getSecondsLeft() === 0) {
            completed = true;
            showScore(session);
        }

        if (!completed) {
            setTimeout(function () {
                timer();
            }, 500);
        }
    };

    setTimeout(function () {
        timer();
    }, 0);
}

function updateWords(session) {
    for (var i in session.countTable) {
        $('table.words td.len-' + i).text(session.countTable[i]);
    }
}

function resizeLetters(session) {
    var allWidth = Math.min(500, Math.min($(window).innerWidth(), $(window).innerHeight() - 180));

    $('.level .letters').css('max-width', allWidth + 'px');

    $('.level .letters .letter').each(function (i, e) {
        $(e).width(allWidth / session.puzzle.size);
        $(e).height(allWidth / session.puzzle.size);

        var span = $(e).children('span');
        span.css('line-height', span.height() + 'px');
    });
}

// pontszám

function showScore(session) {
    $('.screen.level').fadeOut(300, function() {
        $('.screen.score').fadeIn(300);
    });


    var score = {
        id: new Date().getTime(),
        found: session.found.length,
        all: session.puzzle.words.length,
        points: 0
    };

    for (var i in session.found) {
        score.points += session.found[i].length
    }

    var words = $('.score .words').text('');

    var prefix = "";
    for (var i in session.puzzle.words) {
        words.append(prefix);
        words.append('<span class="' + (session.found.indexOf(session.puzzle.words[i]) >= 0 ? 'found' : '') + '">' + session.puzzle.words[i] + '</span>');
        prefix = ", ";
    }


    var scoresDb = null;

    if (localStorage.scores)
        scoresDb = JSON.parse(localStorage.scores);

    if (!scoresDb)
        scoresDb = {};

    var key = session.seconds + '_' + session.puzzle.size;

    if (!scoresDb[key])
        scoresDb[key] = [];

    scoresDb[key].push(score);
    scoresDb[key].sort(function (a, b) {
        return b.points - a.points;
    });
    scoresDb[key] = scoresDb[key].slice(0, 15);

    localStorage.scores = JSON.stringify(scoresDb);

    var table = $('.score table tbody');
    table.children().remove();

    var n = 1;
    for (var i in scoresDb[key]) {
        table.append('<tr class="' + (scoresDb[key][i].id == score.id ? 'current' : '') + '">' +
        '<td>' + (n++) + '.</td>' +
        '<td>' + scoresDb[key][i].points + '</td>' +
        '<td>Játékos</td>' +
        '<td>' + scoresDb[key][i].found + ' / ' + scoresDb[key][i].all + '</td>' +
        '<td>' + ((scoresDb[key][i].found / scoresDb[key][i].all) * 100).toFixed(2) + '</td>' +
        '</tr>');
    }

    $('.score .back').one('click', function (event) {
        event.preventDefault();

        $('.screen.score').fadeOut(300, function() {
            $('.screen.menu').fadeIn(300);
        });
    });
}