/*
 *  TOTP authenticator generator
 *
 *  Local storage schema:
 *
 *    - Key: internal unique ID (not shown to user)
 *    - Value: profileName + '\u0000' + authenticatorBase32
 */

var LS;  // localStorage

var editedAuthKey = null;
var editedAuthPrevSecret = null;

var authIdCounter = 0;
function createUniqueAuthenticatorId() {
	return 'auth-' + Date.now() + '-' + (++authIdCounter);
}

function createDummyLocalStorage() {
	var storage = [];
	storage.push({ key: 'totpSecret', val: 'foobaret' });
	var ret = {
		getItem: function (key) {
			var i, n;
			for (i = 0, n = storage.length; i < n; i++) {
				if (storage[i].key === key) {
					//console.log('dummy local storage: get key ' + key + ' (found)');
					return storage[i].val;
				}
			}
			//console.log('dummy local storage: get key ' + key + ' (not found)');
			return undefined;
		},
		setItem: function (key, val) {
			var i, n;
			for (i = 0, n = storage.length; i < n; i++) {
				if (storage[i].key === key) {
					//console.log('dummy local storage: set key ' + key + ' (found)');
					storage[i].val = val;
					return;
				}
			}
			//console.log('dummy local storage: set key ' + key + ' (created new key)');
			storage.push({ key: key, val: val });
		},
		removeItem: function (key) {
			var i, n;
			for (i = 0, n = storage.length; i < n; i++) {
				if (storage[i].key === key) {
					console.log('dummy local storage: remove key ' + key + ' (found)');
					storage.splice(i, 1);
					return;
				}
			}
			console.log('dummy local storage: remove key ' + key + ' (not found)');
		},
		key: function (idx) {
			if (idx < 0 || idx >= storage.length) {
				return undefined;
			}
			return storage[idx].key;
		}
	};
	Object.defineProperty(ret, 'length', {
		configurable: false, enumerable: false,
		get: function () { return storage.length; },
		set: function () { throw new TypeError('length not writable'); }
	});
	return ret;
}

var viewIds = [
	'totp-authenticators-view',
	'totp-authenticator-edit-view',
	'about-view'
];
function showView(showId) {
	var elem;
	viewIds.forEach(function (id) {
		document.getElementById(id).classList.add('hidden');
	});
	document.getElementById(showId).classList.remove('hidden');

	elem = document.getElementById('auth-edit-buttons');
	if (showId === 'totp-authenticator-edit-view') {
		elem.classList.add('slide');
	} else {
		elem.classList.remove('slide');
	}

	elem = document.getElementById('about-buttons');
	if (showId === 'about-view') {
		elem.classList.add('slide');
	} else {
		elem.classList.remove('slide');
	}
}

// Get a sorted list of internal authenticator IDs.  Sorting is by profile
// name (lowercased), and for identically named profile, by internal identifier.
function getSortedAuthenticatorIds() {
	var list = [];
	var i, n;
	var key, val, tmp;

	for (i = 0, n = LS.length; i < n; i++) {
		key = LS.key(i);
		val = LS.getItem(key);
		if (!val) { continue; }
		tmp = val.split('\u0000');
		if (!tmp) { continue; }
		list.push({ key: key, name: tmp[0].toLowerCase() });
	}

	list.sort(function (a,b) {
		if (a.name < b.name) {
			return -1;
		} else if (a.name > b.name) {
			return 1;
		}
		if (a.key < b.key) {
			return -1;
		} else if (a.key > b.key) {
			return 1;
		}
		return 0;
	});

	//console.log(JSON.stringify(list));

	return list.map(function (ent) { return ent.key; });
}

function recreateAuthenticatorList() {
	var authlist, authids;
	var li, a, p, span;
	var i, n, key, val, tmp;

	//console.log('recreate authenticators, LS.length: ' + LS.length);

	// Delete existing entries
	while (elem = document.querySelector('#authenticator-list li')) {
		elem.parentNode.removeChild(elem);
	}

	// Create a sorted rendering order
	authids = getSortedAuthenticatorIds();

	// Create entries
	authlist = document.querySelector('#authenticator-list');
	for (i = 0, n = authids.length; i < n; i++) {
		key = authids[i];
		//console.log('recreate loop, key: ' + key);
		if (!/^auth-.*$/.test(key)) {
			console.log('skip key ' + key);
			continue;
		}
		val = LS.getItem(key);
		//console.log('recreate loop, val: ' + val);  // dangerous, logs authenticator secret!
		if (!val) {
			console.log('no value ' + key);
			continue;
		}
		tmp = val.split('\u0000');
		if (tmp.length !== 2) {
			console.log('invalid val for key ' + key);
			continue;
		}

		//console.log('setup entry for key ' + key + ', account name "' + tmp[0] + '"');

		li = document.createElement('li');
		li.setAttribute('id', key);
		a = document.createElement('a');
		a.setAttribute('href', '#');
		li.appendChild(a);
		p = document.createElement('p');
		p.appendChild(document.createTextNode(tmp[0]));
		a.appendChild(p);
		p = document.createElement('p');
		span = document.createElement('span');
		span.className = 'auth-curr';
		p.appendChild(span);
		a.appendChild(p);
		authlist.appendChild(li);

		function makeClickHandler(key) {
			return function (event) {
				var elem;
				var val = LS.getItem(key);
				var tmp = val.split('\u0000');

				event.stopPropagation();
				//alert('clicked: ' + key);

				showView('totp-authenticator-edit-view');
				elem = document.getElementById('auth-edit-name');
				elem.value = tmp[0];
				elem = document.getElementById('auth-edit-secret');
				elem.value = '';
				elem = document.getElementById('auth-edit-cancel');
				elem.innerHTML = 'Delete';
				elem = document.getElementById('auth-secret-note');
				elem.classList.remove('hidden');

				editedAuthKey = key;
				editedAuthPrevSecret = tmp[1];
			}
		}
		li.addEventListener('click', makeClickHandler(key));
	}
}

function recomputeAuthenticator(elem, key, profileName, totpSecret, time) {
	var authElem = elem.getElementsByClassName('auth-curr')[0];
	var auth;
	var digits = 6;
	var timeStep = 30e3;

	//console.log('recompute authenticator: ' + elem + ', id: ' + elem.getAttribute('id'));

	try {
		auth = TOTP.TOTP(TOTP.base32Decode(totpSecret), time, 6);
		authElem.innerHTML = auth;
	} catch (e) {
		authElem.innerHTML = '';
		authElem.appendChild(document.createTextNode(String(e)));
	}
}

function clearAuthenticator(elem) {
	var elem = elem.getElementsByClassName('auth-curr')[0];
	elem.innerHTML = '';
}

function recomputeAuthenticatorList() {
	var elems;
	var time = Date.now();
	var i;
	var key, val, tmp;

	// Clear authenticator values first (just in case)
	elems = document.querySelectorAll('#authenticator-list li');
	for (i = 0; i < elems.length; i++) {
		elem = elems.item(i);
		clearAuthenticator(elem);
	}

	// Recompute authenticators, use 'id' property of <li> to find proper
	// authenticator
	elems = document.querySelectorAll('#authenticator-list li');
	for (i = 0; i < elems.length; i++) {
		elem = elems.item(i);
		key = elem.getAttribute('id');
		if (!key || !/^auth-.*$/.test(key)) {
			console.log('unexpected element id: ' + key + ', ignoring');
			continue;
		}
		val = LS.getItem(key);
		if (!val) {
			console.log('value missing for id: ' + key + ', ignoring');
			continue;
		}
		tmp = val.split('\u0000');
		if (tmp.length !== 2) {
			console.log('invalid value for id: ' + key + ', ignoring');
			continue;
		}

		recomputeAuthenticator(elem, key, tmp[0], tmp[1], time);
	}
}

function upgradeOldTotpSecret() {
	var val;
	val = LS.getItem('totpSecret');
	if (!val) { return; }
	LS.removeItem('totpSecret');
	LS.setItem(createUniqueAuthenticatorId(), 'Unnamed\u0000' + val);
	console.log('upgraded old totpSecret into an account');
}

function initRaw() {
	var e;
	var forceDummyStorage;

	if (typeof localStorage !== 'undefined' && !forceDummyStorage) {
		LS = localStorage;
        } else {
		alert('Your browser does not support localStorage, using a dummy replacement.' +
		      'Settings WILL NOT be saved permanently.');
		LS = createDummyLocalStorage();
        }

	upgradeOldTotpSecret();

	e = document.getElementById('about-button');
	e.addEventListener('click', function (event) {
		event.stopPropagation();
		//alert('about clicked');
		showView('about-view');
	});

	e = document.getElementById('about-close');
		e.addEventListener('click', function (event) {
		event.stopPropagation();
		//alert('about-close clicked');
		showView('totp-authenticators-view');
	});

	e = document.getElementById('add-button');
	e.addEventListener('click', function (event) {
		event.stopPropagation();
		//alert('add clicked');
		editedAuthKey = createUniqueAuthenticatorId();
		editedAuthPrevSecret = null;
		showView('totp-authenticator-edit-view');
		elem = document.getElementById('auth-edit-name');
		elem.value = 'Unnamed';
		elem = document.getElementById('auth-edit-secret');
		elem.value = '';
		elem = document.getElementById('auth-edit-cancel');
		elem.innerHTML = 'Cancel';
		elem = document.getElementById('auth-secret-note');
		elem.classList.add('hidden');

		recreateAuthenticatorList();
		recomputeAuthenticatorList();
	});

	e = document.getElementById('auth-edit-cancel');
	e.addEventListener('click', function (event) {
		var elem;
		event.stopPropagation();
		//alert('auth-edit-cancel clicked');
		if (editedAuthKey) {
			LS.removeItem(editedAuthKey);
		}
		editedAuthKey = null;
		editedAuthPrevSecret = null;
		showView('totp-authenticators-view');
		recreateAuthenticatorList();
		recomputeAuthenticatorList();
	});

	e = document.getElementById('auth-edit-save');
	e.addEventListener('click', function (event) {
		var elem, profileName, secret;
		event.stopPropagation();
		//alert('auth-edit-save clicked');
		if (editedAuthKey) {
			profileName = document.getElementById('auth-edit-name').value;
			secret = document.getElementById('auth-edit-secret').value;
			if (secret === '' && typeof editedAuthPrevSecret === 'string') {
				secret = editedAuthPrevSecret;
			}
			LS.setItem(editedAuthKey, profileName + '\u0000' + secret);
		}
		editedAuthKey = null;
		editedAuthPrevSecret = null;
		showView('totp-authenticators-view');
		recreateAuthenticatorList();
		recomputeAuthenticatorList();
	});

	recreateAuthenticatorList();
	recomputeAuthenticatorList();
	setInterval(recomputeAuthenticatorList, 3000);
}

window.onload = function () {
	try {
		initRaw();
	} catch (e) {
		alert(e + '\n' + (e.stack || ''));
	}
}
