// Copyright (c) 2014-2016, Masahiko Imanaka. All rights reserved.
/* global KeyEvent, CodeChecker */
/* jshint moz:true, bitwise:false */

(function() {
'use strict';

var inputContext = null,
    mgmt = null;

var gKeyboardElement = document.getElementById('keyboard'),
    gKeyContainer = document.getElementById('keyContainer'),
    charCodeNumberElement = document.getElementById('codeNumber'),
    charBlockNameElement = document.getElementById('blockName'),
    charPanelElement = document.getElementById('charPanel'),
    charItemElement = document.getElementById('charItem'),
    cachedWindowHeight = screen.availHeight,
    cachedWindowWidth = screen.availWidth,
    lookupCharElement = document.getElementById('lookupChar');


function uninputInit() {
  gKeyboardElement.addEventListener('mousedown', function onMouseDown(ev) {
    ev.preventDefault();
  }, false);

  //renderKeys(null);
  gKeyContainer.addEventListener('touchstart', startKeyHandle, false);
  gKeyContainer.addEventListener('touchend', endKeyHandle, false);

  cachedWindowWidth = gKeyboardElement.clientWidth;
  cachedWindowHeight = gKeyboardElement.clientHeight;
  window.navigator.mozInputMethod.oninputcontextchange = function() {
    resizeWindow();
  };
  window.addEventListener('resize', resizeWindow, false);
  getInputContext();

  charPanelElement.addEventListener('click', sendKeyHandler, false);

  var switchElement = document.getElementById('switchLayout');
  switchElement.addEventListener('click', function switchHandler() {
    mgmt.next();
  });
  // Long press to trigger IME menu
  var menuTimeout = 0;
  switchElement.addEventListener('touchstart', function longHandler() {
    menuTimeout = window.setTimeout(function menuTimeout() {
      mgmt.showAll();
    }, 700);
  }, false);
  switchElement.addEventListener('touchend', function longHandler() {
    clearTimeout(menuTimeout);
  }, false);

  lookupCharElement.addEventListener('click', charLookup, false);

  StackManager.init();
}

function uninputQuit() {
  gKeyContainer.removeEventListener('touchstart', startKeyHandle, false);
  gKeyContainer.removeEventListener('touchend', endKeyHandle, false);
  window.removeEventListener('resize', resizeWindow, false);
  charPanelElement.removeEventListener('click', sendKeyHandler, false);
  // switchElement.removeEventListener('click', switchHandler, false);
  // switchElement.removeEventListener('touchstart', longHandler, false);
  // switchElement.removeEventListener('touchend', longHandler, false);
  lookupCharElement.removeEventListener('click', charLookup, false);
  StackManager = null;
  window.removeEventListener('load', uninputInit);
}

function getInputContext() {
  inputContext = navigator.mozInputMethod.inputcontext;
  mgmt = window.navigator.mozInputMethod.mgmt;
}

function resizeWindow() {
  getInputContext();
  window.resizeTo(cachedWindowWidth, cachedWindowHeight);
}


function sendKeyCode(aKeyCode, isSpecial) {
  if (isSpecial) {
    inputContext.sendKey(aKeyCode, 0, 0);
  } else {
    inputContext.sendKey(0, aKeyCode, 0);
  }
}

function sendKeyHandler(ev) {
  ev.preventDefault();
  if (StackManager.ligature.length > 0) {
    var cpArray = [];
    cpArray = StackManager.ligature.map(function (c) {
      // fromCodePoint() and codePointAt() are compatible with Gecko 29.
      if (c !== '') {
        return String.fromCodePoint(parseInt(c, 16)).codePointAt(0);
      }
    });
    if (cpArray.length > 0) {
      cpArray.forEach(function (c) {
        if (c) {
          // Encoding surrogate pair.
          var [c1, c2] = StackManager.encodeSurrogatePair(c);
          sendKeyCode(c1, false);
          if (c2 !== null) {
            sendKeyCode(c2, false);
          }
        }
      });
      clearCandidate();
      //console.log("sendKeyCode: cpArray = "+cpArray);
    }
  }
}

function toggleKeyStyle(aKey, isTouch) {
  if (isTouch) {
    aKey.style.backgroundColor = '#88d';
  } else {
    aKey.style.backgroundColor = '#333';
  }
}

function startKeyHandle(ev) {
  ev.preventDefault();
  // Set backgound-color.
  toggleKeyStyle(ev.target, true);
  // Handle key touch event.
  keyHandle(ev);
}

function endKeyHandle(ev) {
  ev.preventDefault();
  // Recover backgound-color.
  toggleKeyStyle(ev.target, false);
}

function keyHandle(ev) {
  var currentkey = ev.target,
      val = '';
  
  // Get char from class name.
  if (currentkey.classList.contains('key')) {
    val = currentkey.getAttribute('code');
    //console.log("keyHandle: currentkey.code="+val);
    switch (val) {
      case 'CLR':
        if (StackManager.getCharCodeArrLength() > 0) {
          StackManager.remove_stack();
        } else {
          // Backspace
          sendKeyCode(KeyEvent.DOM_VK_BACK_SPACE, true);
        }
        break;
      case 'NXT':
        StackManager.setNextCharCodeArr();
        break;
      default:
        StackManager.appendNumber(val);
    }
  } else {
    return;
  }

  stack_redraw();
}


function stack_redraw() {
  var charCode_str = StackManager.makeCharCode(),
      blockName = '',
      isPrintable = true;

  // Check character code block.
  // isValidCode() and codeRange() are form 'code_check.js'
  if (CodeChecker.isValidCode(charCode_str)) {
    [isPrintable, blockName] = CodeChecker.codeRange(charCode_str);
    StackManager.setCode2Stack(charCode_str, isPrintable);

    updateCodeAndBlockNameElement(blockName, isPrintable);
    updateCharItemElement(charCode_str, isPrintable);
  } else {
    StackManager.clearCharCodeArr();
  }
}


function updateCodeAndBlockNameElement(aBlockName, aIsPrintable) {
  charCodeNumberElement.textContent = StackManager.getPaddingCode();
  if (StackManager.getCharCodeArrLength() >= 2) {
    charBlockNameElement.textContent = aBlockName;
    // Set font color.
    if (aIsPrintable) {
      charBlockNameElement.style.color = '#eea';
    } else {
      charBlockNameElement.style.color = '#e99';
    }
  } else {
    charBlockNameElement.textContent = '';
  }
}


function updateCharItemElement(aCode, aIsPrintable) {
  var charEntity = '';
  // Show character face.
  if (aCode !== '') {
    var code = parseInt(aCode, 16);
    // For not printable characters such as control code.
    if (code <= 0x20) {
      code += 0x2400;  // Graphic pictures for control codes.
    } else if (!aIsPrintable){
      switch(code) {
        case 0x7f:
          code = 0x2421;  // Symbol for 'Delete'.
          break;
        default:
          code = -1;
      }
    }

    // Set char entity to show.
    if (code > 0) {
      charEntity = '&#x' + code.toString(16) + ';';
    } else {
      charEntity = '&#xfffd;';  // REPLACEMENT CHARACTER
    }
  }
  charItemElement.innerHTML = charEntity;
}


function clearCandidate() {
  StackManager.init();
  charCodeNumberElement.textContent = '';
  charBlockNameElement.textContent = '';
  charItemElement.textContent = '';
}


var StackManager = {
  ligature: [],
  stackPos: 0,
  _charCodeArr: [],
  arrMaxLength: 5,
  arrMinLength: 2,

  init: function stackManager_init() {
    this.ligature = [];
    this.stackPos = this.ligature.length;
    this._charCodeArr = [];
  },

  appendNumber: function stackManager_appendNumber(aVal) {
    if (this._charCodeArr.length < this.arrMaxLength) {
      this._charCodeArr.push(aVal);
    } else {
      this._charCodeArr = [aVal];
    }
  },

  setNextCharCodeArr: function stackManager_setNextCharCodeArr() {
    var currentCode = this.makeCharCode(),
        nextCodeArr = [],
        code = -1;
    if (currentCode.length > 0) {
      this.clearCharCodeArr();
      code = parseInt(currentCode, 16);
      code++;
      nextCodeArr = code.toString(16).toUpperCase().split('');
      if (nextCodeArr.length < 2) {
        nextCodeArr.unshift('0');
      }
      nextCodeArr.forEach(function (c) {
        StackManager._charCodeArr.push(c);
      });
    }
  },

  clearCharCodeArr: function stackManager_clearCharCodeArr() {
    this._charCodeArr = [];
  },

  getCharCodeArrLength: function stackManager_getCharCodeArrLength() {
    return this._charCodeArr.length;
  },

  makeCharCode: function stackManager_makeCharCode() {
    // Threshold of minimum length.
    if (this._charCodeArr.length >= this.arrMinLength) {
      return this._charCodeArr.join('');
    } else {
      return '';
    }
  },

  getPaddingCode: function stackManager_getPaddingCode() {
    var n = this.arrMaxLength - this._charCodeArr.length,
        padding = [];
    while (n > 0) {
      padding.push('_');
      n--;
    }
    return padding.concat(this._charCodeArr).join('');
  },

  setCode2Stack: function stackManager_setCode2Stack(aCode, aIsPrintable) {
    if (aCode.length > 0 && aIsPrintable) {
      this.ligature[this.stackPos] = aCode;
    } else {
      this.ligature[this.stackPos] = '';
    }
  },

  remove_stack: function stackManager_removeLigatureStack() {
    if (this._charCodeArr.length > 0) {
      // Clear current charCodeArr.
      this._charCodeArr = [];
      delete this.ligature[this.stackPos];
    } else {
      // Remove current empty stack item.
      if (this.stackPos > 0) {
        this.ligature.pop();
        this.stackPos--;
        // Set previous stack item to charCodeArr.
        this._charCodeArr = this.ligature[this.stackPos].split('');
      } else {
        this.stackPos = 0;
        this._charCodeArr = [];
      }
    }
  },

  decodeSurrogatePair: function stackManager_decodeSurrogatePair(aHigh, aLow) {
    if (aHigh < 0xd800 || 0xbfff < aHigh || aLow < 0xdc00 || 0xdfff < aLow) {
      return null;
    }
    //console.log('surrogatePair: ' + aHigh + ' + ' + aLow);
    var code = 0x10000 + (aHigh - 0xd800) * 0x400 + (aLow - 0xdc00);
    return code;
  },

  encodeSurrogatePair: function stackManager_encodeSurrogatePair(aCode) {
    if (aCode < 0x10000 || aCode > 0x10ffff) {
      return [aCode, null];
    }
    var hs = (aCode - 0x10000) / 0x400 + 0xd800,
        ls = (aCode - 0x10000) % 0x400 + 0xdc00;
    return [hs, ls];
  }
};


function charLookup() {
  var code = inputContext.textAfterCursor.codePointAt(0),
      numberArr = [];
  if (code > -1) {
    StackManager.clearCharCodeArr();
    numberArr = code.toString(16).toUpperCase().split('');
    if (numberArr.length === 1) {
      numberArr.unshift('0');
    }
    numberArr.forEach(function (val) {
      StackManager.appendNumber(val);
    });
    stack_redraw();
  }
}


window.addEventListener('load', uninputInit);
window.addEventListener('unload', uninputQuit);

})(window);
