/*jshint onevar: false, quotmark: double*/
/*global webserver, Blob*/
(function() {
	"use strict";

	webserver.toArrayBuffer = function(base64) {
		/*jshint bitwise: false*/
		var b = atob(base64);
		var arrary = new Uint8Array(b.length);
		for (var i = 0; i < b.length; i++) {
			arrary[i] = b.charCodeAt(i) & 0xFF;
		}
		return arrary.buffer;
	};

	webserver.favicon = webserver.toArrayBuffer(webserver.favicon);

	webserver.Socket = function(id, tcpSocket, documentRoot, persistentConnection) {
		this.tcpSocket = tcpSocket;
		this.id = id;
		this.documentRoot = documentRoot;
		this.persistentConnection = persistentConnection;
		this.header = null;
		this.data = "";
		this.bufferState = "ready";
		this.messages = [];
		var self = this;
		tcpSocket.ondata = function (e) {
			self.ondata(e.data || e);
		};
		tcpSocket.ondrain = function () {
			self.ondrain();
		};
		tcpSocket.onclose = function () {
			webserver.debug(id, "closed");
			self.tcpSocket = null;
		};
		tcpSocket.onerror = function (e) {
			webserver.error(id, e.data || e.message || e);
			self.close();
		};
	};

	webserver.Socket.prototype.parseHeader = function(data) {
		var lines = data.split(/\r?\n/);
		var requestLine = lines[0];
		var res = /^(\S+)\s+(\S+)\s+HTTP\/(\d+\.\d+)$/.exec(requestLine || "");
		if (!res) {
			throw "Invalid request line: " + requestLine;
		}
		webserver.debug(this.id, requestLine);
		var message = {
			method: res[1],
			url: res[2],
			httpVersion: res[3],
			headers: {}
		};
		for (var i = 1; i < lines.length; i++) {
			var line = lines[i];
			var index = line.indexOf(":");
			if (index < 0) {
				throw "Invalid header: " + line;
			}
			var name = line.substring(0, index).trim().toLowerCase();
			var value = line.substring(index + 1).trim();
			message.headers[name] = value;
		}
		return message;
	};

	function getContentLength(content) {
		if (!content) {
			return 0;
		}
		if (content instanceof ArrayBuffer) {
			return content.byteLength;
		}
		if (content instanceof Blob) {
			return content.size;
		}
		if (typeof content.length === "number") {
			return content.length;
		}
		throw "Invalid content: " + content;
	}

	webserver.Socket.prototype.ondata = function (data) {
		if (!this.tcpSocket) {
			webserver.warn(this.id, "Socket closed: Discard " + getContentLength(data) + " bytes read");
		}
		webserver.debug(this.id, "Received " + getContentLength(data) + " bytes");
		this.data += this.tcpSocket.binaryType === "arraybuffer" ? String.fromCharCode.apply(null, new Uint8Array(data)) : data;
		while (this.data.length) {
			if (!this.header) {
				var endOfHeader = this.data.indexOf("\r\n\r\n");
				if (endOfHeader < 0) {
					webserver.debug(this.id, "Waiting for header: " + this.data.length);
					break;
				}
				try {
					var head = this.data.substring(0, endOfHeader);
					this.data = this.data.substring(endOfHeader + 4);
					this.header = this.parseHeader(head);
				} catch (e) {
					webserver.error(this.id, "Data: " + (e.message || e));
					this.close();
					return;
				}
			}
			var contentLength = Number(this.header["content-length"] || "");
			if (contentLength > this.data.length) {
				webserver.debug(this.id, "Waiting for content: " + this.data.length + " of " + contentLength);
				break;
			} else {
				var message = this.header;
				this.header = null;
				message.body = this.data.substring(0, contentLength);
				this.data = this.data.substring(contentLength);
				this.messages.push(message);
			}
		}
		if (this.messages.length) {
			this.onrequest();
		}
	};

	var unknownExtension = {};

	var lastZipFileName = "";
	var zipFile;

	webserver.Socket.prototype.onrequest = function () {
		var self = this;
		function checkLocal(error) {
			switch (path) {
				case "/index.html":
					self.response(200, webserver.index, {
						"Cache-Control": "no-cache, no-store, must-revalidate",
						"Pragma": "no-cache",
						"Expires": 0,
						"Content-Type": "text/html"
					});
					break;
				case "/favicon.ico":
					self.response(200, webserver.favicon, "image/x-icon");
					break;
				default:
					if (error === "NotFoundError") {
						webserver.warn(self.id, "File not found: " + file);
						self.response(404);
					} else {
						webserver.warn(self.id, error + ": " + request.url);
						self.response(400);
					}
			}
		}
		try {
			var request = this.messages.shift();
			webserver.incRequestCount();
			var path = decodeURI(request.url.split("#")[0].split("?")[0]);
			if (path.charAt(0) !== "/" || path.indexOf("/../") >= 0) {
				webserver.warn(this.id, "Invalid Request URL: " + request.url);
				this.response(400);
				return;
			}
			if (path.charAt(path.length - 1) === "/") {
				path += "index.html";
			}
			var file = this.documentRoot + path;
			var type;
			var res = /\.([^.]*)$/.exec(path);
			if (res) {
				var ext = res[1].toLowerCase();
				type = webserver.mimeTypes[ext];
				if (!type) {
					if (!unknownExtension[ext]) {
						unknownExtension[ext] = true;
						webserver.debug(this.id, "Unknown extension: " + ext);
					}
				}
			}
			if (request.method === "GET") {
				var sdcard = webserver.sdcards[file.split("/")[1]];
				if (sdcard) {
					res = /\.zip\/(.*)$/.exec(file);
					var fileRequest;
					if (res) {
						var zipFileName = file.substring(0, res.index + 4);
						if (zipFileName !== lastZipFileName) {
							fileRequest = sdcard.get(zipFileName);
							fileRequest.onsuccess = function () {
								webserver.readZip(this.result, function(zip, error) {
									if (error) {
										self.response(500, error);
										return;
									}
									zipFile = zip;
									lastZipFileName = zipFileName;
									zipFile.get(res[1], type, function(code, content, type) {
										self.response(code, content, type);
									});
								});
							};
							fileRequest.onerror = function () {
								checkLocal(this.error.name);
							};
						} else {
							zipFile.get(res[1], type, function(code, content, type) {
								self.response(code, content, type);
							});
						}
					} else {
						fileRequest = sdcard.get(file);
						fileRequest.onsuccess = function () {
							self.response(200, this.result, type);
						};
						fileRequest.onerror = function () {
							checkLocal(this.error.name);
						};
					}
				} else {
					checkLocal("NotFoundError");
				}
			} else {
				webserver.warn(this.id, "Method Not Allowed: " + request.method);
				this.response(405);
			}
		} catch (e) {
			webserver.error(this.id, e.message || e);
			this.response(500);
		}
	};

	webserver.Socket.prototype.response = function (code, content, type) {
		try {
			if (!this.tcpSocket) {
				webserver.warn(this.id, "Socket closed: Discard response");
				return;
			}
			var phrase = webserver.reasonPhrases[code] || "Error";
			var head = "HTTP/1.1 " + code + " " + phrase + "\r\n";
			if (code !== 200) {
				content = "ERROR #" + code + ": " + phrase;
				type = "text/plain";
			}
			head += "Server: FFOS-WebServer\r\n";
			if (typeof type === "string") {
				head += "Content-Type: " + type + "\r\n";
			} else if (typeof type === "object") {
				for (var p in type) {
					head += p + ": " + type[p] + "\r\n";
				}
			}
			var contentLength = getContentLength(content);
			head += "Content-Length: " + contentLength + "\r\n";
			if (!this.persistentConnection) {
				head += "Connection: close\r\n";
			} else if (this.bufferState !== "ready" || (this.tcpSocket.bufferedAmount || 0) + contentLength > 0xFFFF ) {
				webserver.debug(this.id, "Request connection close");
				head += "Connection: close\r\n";
			}

			head += "\r\n";
			var message = new Blob([head, content]);
			var reader = new FileReader();
			var self = this;
			reader.onloadend = function () {
				if (this.error) {
					webserver.error(self.id, "FileReader: " + this.error.name);
					self.close();
				} else {
					self.send(this.result);
				}
			};
			if (this.tcpSocket.binaryType === "arraybuffer") {
				reader.readAsArrayBuffer(message);
			} else {
				if (typeof head === "string" && typeof content === "string") {
					this.send(head + content);
				} else {
					reader.readAsBinaryString(message);
				}
			}
		} catch (e) {
			webserver.error(this.id, "Response: " + (e.message || e));
			this.close();
		}
	};

	webserver.Socket.prototype.ondrain = function () {
		webserver.debug(this.id, "drain");
	};

	webserver.Socket.prototype.send = function (data) {
		try {
			if (!this.tcpSocket) {
				webserver.warn(this.id, "Socket closed: Discard " + getContentLength(data) + " bytes to send");
			} else {
				this.bufferState = this.tcpSocket.send(data) ? "ready" : "busy";
				webserver.debug(this.id, "Sent " + getContentLength(data) + "/" + this.tcpSocket.bufferedAmount + " bytes, " + this.bufferState);
				if (this.messages.length) {
					this.onrequest();
				}
			}
		} catch (e) {
			webserver.error(this.id, "Send: " + (e.message || e));
			this.close();
		}
	};

	webserver.Socket.prototype.close = function () {
		try {
			if (this.tcpSocket) {
				this.tcpSocket.close();
			}
			this.tcpSocket = null;
		} catch (e) {
			webserver.warn(this.id, e.message || e);
		}
	};

	webserver.Socket.prototype.dumpState = function () {
		if (this.tcpSocket) {
			var message;
			if (this.tcpSocket.readyState === "open") {
				message = "buffer=" + this.bufferState;
				if (this.tcpSocket.bufferedAmount) {
					message += ", buffered=" + this.tcpSocket.bufferedAmount;
				}
				if (this.header) {
					message += ", header=" + this.header.url;
				}
				if (this.data.length) {
					message += ", data=" + this.data.length;
				}
				if (this.messages.length) {
					message += ", messages=" + this.messages.length;
				}
			} else {
				message = this.tcpSocket.readyState;
			}
			webserver.debug(this.id, message);
		}
	};

})();

