/**
 * XRate currency converter application
 *
 * The application converts two currencies given by the user. It returns with the actual exchange rate, ask, and bid values.
 * Used modules:
 * - ngRoute: for navigation.
 * - mobileAngularUi: Mobile framework for AngularJS.
 *
 * @author Peter Szrnka
 * @version 1.0
 * 
 */
var app = angular.module('XRate', ["ngRoute", "mobile-angular-ui", "ngAdsense"]);

app.config(function($routeProvider) {
    $routeProvider.when('/', {
		controller: 'XRateController',
		templateUrl: 'js/xrate/template/main.html'
    });
});
/**
 * Basic service for handling storage operations.
 */
app.service("StorageService", ["$window", function($window) {
	/**
	 * The variable returns true if local storage functionality supported by the users device.
	 */
	this.storageSupported = typeof(Storage)!=="undefined";
	
	/**
	 * Returns with an item by the given id.
	 * @param id
	 */
	this.getItem = function(id) {
		return $window.localStorage.getItem(id) || null;
	};

	/**
	 * Sets an item by the given id and new value.
	 * @param id
	 * @param value
	 */
	this.setItem = function(id, value) {
		$window.localStorage.setItem(id, value);
	};

	/**
	 * Deletes an item from the local storage by the given id.
	 * @param id
	 */
	this.delete = function(id) {
		delete $window.localStorage[id];
	};
}]);
/**
 * The currency exchange service.
 */
app.service("ExchangeService", ["$http", "BASE_URL", function($http, BASE_URL) {
	/**
	 * Returns with the string replaced, correct URL to be invoked.
	 */
	var generateUrl = function(fromCurrency, toCurrency) {
		return BASE_URL.replace("{1}", fromCurrency).replace("{2}", toCurrency);
	}
	/**
	 * The function returns with a $http.post() promise, and with the retrieved information about the exchange.
	 */
	this.changeRate = function(fromCurrency, toCurrency, unit) {
		return $http.post(generateUrl(fromCurrency, toCurrency));
	};
}]);
/**
 * Exchange history management service.
 */
app.service("HistoryService", ["$window", "StorageService", "STORAGE_ID", function($window, StorageService, STORAGE_ID) {
	/**
	 * Saves the current currency exchange invocation result into the localstorage.
	 * @param Object item
	 */
	this.saveExchange = function(item) {
		if(StorageService.storageSupported) {
			if(StorageService.getItem(STORAGE_ID)) {
				var parseObj = JSON.parse(StorageService.getItem(STORAGE_ID));
				parseObj.push(item);
				StorageService.setItem(STORAGE_ID, JSON.stringify(parseObj));
			} else {
				var serObj = [item];
				StorageService.setItem(STORAGE_ID, JSON.stringify(serObj));
			}
		}
	};

	/**
	 * Clears the entire local storage of the application.
	 */
	this.clearHistory = function() {
		StorageService.delete(STORAGE_ID);
	};

	/**
	 * The function deletes an item from the history by the given timestamp
	 * @param number changeDate
	 */
	this.deleteHistoryItem = function(changeDate) {
		if(StorageService.storageSupported) {
			if(StorageService.getItem(STORAGE_ID)) {
				var parseObj = JSON.parse(StorageService.getItem(STORAGE_ID));
				
				for(var i = 0; i < parseObj.length; i++) {
					if(parseObj[i]["changeDate"] == changeDate) {
						parseObj.splice(i, 1);
					}
				}
				if(JSON.stringify(parseObj) != "[]") {
					StorageService.setItem(STORAGE_ID, JSON.stringify(parseObj));
				} else {
					this.clearHistory();
				}
			}
		}
	};

	/**
	 * Preload function for retrieving the former exchange invocations.
	 */
	this.preloadHistory = function() {
		var historyItems = [];
		if(StorageService.storageSupported && StorageService.getItem(STORAGE_ID)) {
			var parseObj = JSON.parse(StorageService.getItem(STORAGE_ID));
			for(var i = 0; i < parseObj.length; i++) {
				historyItems.push(parseObj[i]);
			}
		}
	
		return historyItems;
	};
}]);
app.controller("XRateController", ["$scope", "$rootScope", "$window", "ExchangeService", "HistoryService", "StorageService", "DEFAULT_LANGUAGE", "LANGUAGES", "CURRENCIES", "AD_ENABLED",
function($scope, $rootScope, $window, ExchangeService, HistoryService, StorageService, DEFAULT_LANGUAGE, LANGUAGES, CURRENCIES, AD_ENABLED) {
	// Init language
	if(StorageService.getItem("defaultLanguage") === null) {
		StorageService.setItem("defaultLanguage", DEFAULT_LANGUAGE);
	}
	var defaultLanguage = StorageService.getItem("defaultLanguage");
	// Init main variables
	var loadMaskOn = false;
	var dataModel = {
		fromCurrency : "",
		toCurrency : "",
		unit : 1
	};
	var filterModel = {
		fromFilterText : undefined,
		toFilterText : undefined
	};
	var fromCurrencies = angular.copy(CURRENCIES);
	var toCurrencies = angular.copy(CURRENCIES);
	var settingsModel = {
		usedLanguage : defaultLanguage
	};
	var result = {
		from: "",
		to: "",
		unit: 1,
		value: 0
	};
	var changeRate = function(fromCurrency, toCurrency, unit) {
		$rootScope.toggle('loadOverlay', 'on');
		ExchangeService.changeRate(fromCurrency, toCurrency, unit)
		.then(function(result) {
			$rootScope.toggle('loadOverlay', 'off');
			HistoryService.saveExchange({
				"from" : result.data.query.results.rate.Name.split(" to ")[0],
				"to" : result.data.query.results.rate.Name.split(" to ")[1],
				"unit" : unit,
				"ask" : result.data.query.results.rate.Ask,
				"bid" : result.data.query.results.rate.Bid,
				"value" : result.data.query.results.rate.Rate,
				"changeDate" : new Date().getTime()
			});

			$scope.result = {
				from: result.data.query.results.rate.Name.split(" to ")[0],
				to: result.data.query.results.rate.Name.split(" to ")[1],
				ask: result.data.query.results.rate.Ask,
				bid: result.data.query.results.rate.Bid,
				unit: dataModel.unit,
				value: parseInt(dataModel.unit) * result.data.query.results.rate.Rate
			};
			$rootScope.toggle('resultOverlay', 'on');
			$scope.items = HistoryService.preloadHistory();
		}, function(result) {
			console.info("result fail", result);
		});
	};
	var validateForm = function() {
		return (dataModel.fromCurrency.length > 0 && dataModel.toCurrency.length > 0 && angular.isDefined(dataModel.unit) && dataModel.unit > 0);
	};
	var clearHistory = function() {
		HistoryService.clearHistory();
		$scope.items = HistoryService.preloadHistory();
		$rootScope.toggle('removeAllOverlay', 'off');
	};
	
	$scope.repeatSearch = function(fromCurrency, toCurrency, unit) {
		$rootScope.toggle('detailOverlay', 'off');
		changeRate(fromCurrency, toCurrency, unit);
	};
	$scope.openItem = function(item) {
		$scope.itemDetails = item;
		$rootScope.toggle('detailOverlay', 'on');
	};
	$scope.closeDetails = function() {
		$rootScope.toggle('detailOverlay', 'off');
	};
	$scope.clearHistory = function() {
		$rootScope.toggle('removeAllOverlay', 'on');
	};
	$scope.submitClearHistory = function() {
		clearHistory();
	};
	$scope.dismissClearHistoryDialog = function() {
		$rootScope.toggle('removeAllOverlay', 'off');
	};
	$scope.deleteItem = function(item) {
		HistoryService.deleteHistoryItem(item.changeDate);
		$scope.items = HistoryService.preloadHistory();
	};
	$scope.switchItems = function() {
		var tmp = dataModel.fromCurrency;
		$scope.dataModel.fromCurrency = dataModel.toCurrency;
		$scope.dataModel.toCurrency = tmp;
		$window.NativeWindow.toast.show("Showing you a toast", "short"); 
	};
	$scope.clearForm = function() {
		$scope.dataModel.fromCurrency = "";
		$scope.dataModel.toCurrency = "";
	};
	$scope.currencyFieldsValidation = function() {
		return (dataModel.fromCurrency === "" && dataModel.toCurrency === "");
	};
	$scope.saveSettings = function() {
		StorageService.setItem("defaultLanguage", settingsModel.usedLanguage);
	};
	/**
	 *
	 * @since 1.1
	 */
	$scope.filterCurrency = function(direction) {
		if(direction === 'from') {
			$rootScope.toggle('fromFilterOverlay', 'on');
		}
		else {
			$rootScope.toggle('toFilterOverlay', 'on');
		}
	};
	$scope.doFilter = function(direction) {
		if(direction === 'from') {
			$scope.fromCurrencies = [];
			for(var i = 0; i < CURRENCIES.length; i++) {
				if(
					CURRENCIES[i].currencyId.toLowerCase().indexOf(filterModel.fromFilterText.toLowerCase()) !== -1 ||
					CURRENCIES[i].currencyName.toLowerCase().indexOf(filterModel.fromFilterText.toLowerCase()) !== -1
				) {
					$scope.fromCurrencies.push(CURRENCIES[i]);
				}
			}
			$rootScope.toggle('fromFilterOverlay', 'off');
		}
		else {
			$scope.toCurrencies = [];
			for(var i = 0; i < CURRENCIES.length; i++) {
				if(
					CURRENCIES[i].currencyId.toLowerCase().indexOf(filterModel.toFilterText.toLowerCase()) !== -1 ||
					CURRENCIES[i].currencyName.toLowerCase().indexOf(filterModel.toFilterText.toLowerCase()) !== -1
				) {
					$scope.toCurrencies.push(CURRENCIES[i]);
				}
			}
			$rootScope.toggle('toFilterOverlay', 'off');
		}
	};

	// Scope variables
	$scope.filterModel = filterModel;
	$scope.defaultLanguage = defaultLanguage;
	$scope.languages = LANGUAGES;
	$scope.changeRate = changeRate;
	$scope.validateForm = validateForm;
	$scope.result = result;
	$scope.settingsModel = settingsModel;
	$scope.dataModel = dataModel;
	$scope.items = HistoryService.preloadHistory();
	$scope.fromCurrencies = fromCurrencies;
	$scope.toCurrencies = toCurrencies;
	$scope.adEnabled = (AD_ENABLED === true);
}]);
/**
 * Separated controller for managing ads.
 * @since 1.1
 */
app.controller('AdController', ["$scope", "AD_ENABLED", "ADSENSE_PARAMETERS", function($scope, AD_ENABLED, ADSENSE_PARAMETERS) {
	$scope.adEnabled = (AD_ENABLED === true);
	$scope.adsenseParameters = ADSENSE_PARAMETERS;
}]);
/**
 * Basic filter for handling translations.
 */
app.filter('resolve', ["StorageService", "LANGUAGE_RESOLVATIONS", "DEFAULT_LANGUAGE", 
function(StorageService, LANGUAGE_RESOLVATIONS, DEFAULT_LANGUAGE) {
	return function(input) {
		//return input ? '\u2713' : '\u2718';
		try {
			return LANGUAGE_RESOLVATIONS[input][StorageService.getItem("defaultLanguage")];
		}
		catch(e) {
			return "Resource not resolved['"+input+"']";
		}
	};
}]);