
/*
 Class: js.util.Text
 Useful text functions
 */

(function(JS, js){
    'use strict';
    var jTrimLeft = /^\s+/,
        jTrimRight = /\s+$/,
        escapeRegExp = /[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g;
    if( String.prototype.trim === undefined )
        String.prototype.trim = function(){
            return this.replace( jTrimLeft, '' ).replace( jTrimRight, '' );
        };
    if( String.prototype.capitalize === undefined )
        String.prototype.capitalize = function(){
            return this.charAt( 0 ).toUpperCase() + this.substr( 1 );
        };

    if (String.prototype.endsWith === void 0) {
        String.prototype.endsWith = function(suffix) {
            return this.indexOf(suffix, this.length - suffix.length) !== -1;
        };
    }

    js.util.Text = {
        trim: function( text ){
            return text.trim();
        },
        capitalize: function( text ){
            return text.substr( 0, 1 ).toUpperCase() + text.substr(1);
        },

        getTemplated: function( text, replacements ){
            var out = text, i;
            for( i in replacements )
                replacements.hasOwnProperty( i ) &&
                    ( out = out.replace( new RegExp('\\{'+ i +'\\}','gi'), replacements[ i ] ) );

            return out;
        },
        escapeRegExp: function( text ){
            return text.replace( escapeRegExp, "\\$&" );
        },
        splitByWords: function( text, inArray ){
            var i, _i, p,
                txt = text,
                txtL = text.toLowerCase(),
                out = [],
                nearest,
                pos,
                finded,
                arr = [];

            for( i = 0, _i = inArray.length; i < _i; i++ ){
                arr[i] = inArray[ i ].toLowerCase();
            }

            do {
                nearest = txt.length;
                for( i = 0, _i = arr.length; i < _i; i++ ){
                    pos = txtL.indexOf( arr[ i ] );
                    if( pos !== -1 && pos < nearest ){
                        nearest = pos;
                        finded = i;
                    }
                }
                if( nearest === txt.length)
                    break;

                out.push( txt.substr( 0, nearest ) );
                out.push( inArray[ finded ] );
                p = nearest + arr[ finded ].length;
                txt = txt.substr( p );
                txtL = txtL.substr( p );
            }while( true );
            out.push( txt );

            return out;
        },
        escapeHtml: function escapeHtml(text) {
            return text
                .replace(/&/g, "&amp;")
                .replace(/</g, "&lt;")
                .replace(/>/g, "&gt;")
                .replace(/"/g, "&quot;")
                .replace(/'/g, "&#039;");
        },
        /*
            Function: splitWithStripping
                Splits text with stripping feature (\)

            Parameters:
                text - text to split
                splitter - splitter
         */
        splitWithStripping: function( text, splitter ){

            var tokens = text.split( splitter ),
                token,
                i, tmp;
            for( i = tokens.length - 1; i ;){
                token = tokens[--i];
                if( token.substr(token.length - 1) === '\\' ){
                    tmp = tokens[i].substr(0, tokens[i].length - 1) + splitter + tokens[i + 1];
                    tokens.splice(i,1);
                    tokens[i] = tmp;
                }
            }
            return tokens;
        }/*cut*/,
        /*
        Function:  tokenizeQuotes
            tokenize text. get quotes (single and double) with stripping

        Parameters:
            text - text to tokenize
        */
        tokenizeQuotes: function( text ){
            var tokens = [];
            text.replace(/"(?:[^"\\]|\\.)*"|'(?:[^'\\]|\\.)*'/g,function(match, from){
                tokens.push([from, match.length, 1]);
            });
            return this.splitByRanges( text, tokens );
        },
        splitByRanges: function( text, ranges ){
            var el, elOffset, offset = 0, list = [];
            while( el = ranges.shift()){

                elOffset = el[0]-offset;
                if(elOffset>0)
                    list.push( [0,ranges.substr(0,elOffset)] );

                list.push( [el[2],ranges.substr(elOffset,el[1])] );

                ranges = ranges.substr( elOffset+el[1] );
                offset+=elOffset+el[1];
            }
            ranges.length && list.push([0,ranges]);

            return list;
        },
        /*
            Function: tokenizeBracesWithStripping
            Parameters:
                format - string with braces like 'ololo {mimimi} zhazha {pish}'

            Return:
                Array of tokens like
                |[
                |    [ false, "ololo "],
                |    [ true, "mimimi"],
                |    [ false, " zhazha "],
                |    [ true, "pish"]
                |]
         */
        tokenizeBracesWithStripping:function ( format ){
            var tpl = format,
                i, _i, chr, tmp, tmpInt,
                stripped = false,
                strip = '\\',
                beginPos = 0,
                tokens = [],
                openBrace = '{',
                closeBrace = '}';


            for( i = 0, _i = tpl.length; i < _i; i++ ){
                chr = tpl.substr( i, 1 );
                if( stripped )
                    stripped = false;else if( chr === strip ){
                    stripped = true;
                }else if( chr === openBrace ){
                    beginPos = i;
                }else if( chr === closeBrace ){
                    tmp = tpl.substr( 0, beginPos );
                    tmpInt = tmp.length;
                    if( tmpInt > 0 ){
                        tokens.push(
                            [
                                false,
                                tmp
                            ] );
                    }
                    tmp = tpl.substr( beginPos + 1, i - beginPos - 1 );
                    tokens.push(
                        [
                            true,
                            tmp
                        ] );
                    tpl = tpl.substr( i + 1 );
                    _i -= i;
                    i = -1;

                }
            }
            if( tpl.length > 0 )
                tokens.push(
                    [
                        false,
                        tpl
                    ] );

            return tokens;

        },

        /*
        Function: addSlashes
            Function is from PHP.js, it adds slashes

            Parameters:
                text - text

            Return:
                text with more slashes
         */
        addSlashes:function ( str ){ // from php.js
            return (str + '').replace( /[\\"']/g, '\\$&' ).replace( /\u0000/g, '\\0' ).replace( /\n/g, '\\n' );
        },

        /*
        Function: bonusParams
            Extract [bonus: params] [from: string]

            Example:
                (code)
                    js.util.Text.bonusParams( 'cut[b:1]ted[c:2][d] parts', 'out' )
                    // Output:
                    {
                        "b": "1",
                        "c": "2",
                        "d": true,
                        "out": "cutted parts"
                    }
                (end code)

            Parameters:
                text - text
                holdTo - where to put cutted string in output

            Return:
                Formed object
         */
        bonusParams: function ( text, holdTo ){

            var spec =
                [
                    text.indexOf( '[' ),
                    text.indexOf( ']' )
                ],
                obj = {},
                piece;
            while( spec[ 0 ] !== -1 && spec[ 1 ] !== -1 ){
                spec =
                [
                    text.indexOf( '[' ),
                    text.indexOf( ']' )
                ];
                piece = text.substr( spec[ 0 ] + 1, spec[ 1 ] - spec[ 0 ] - 1 ).split( ':', 2 );
                text = text.substr( 0, spec[ 0 ] ) + text.substr( spec[ 1 ] + 1 );
                piece[ 0 ] = $.trim( piece[ 0 ] );
                if( piece[ 0 ] !== '' ){
                    obj[ piece[ 0 ] ] = piece[ 1 ] || true;
                    obj[ piece[ 0 ].toLowerCase() ] = obj[ piece[ 0 ] ];
                }
            }
            obj[ holdTo ] = text;
            return obj;
        }/*end cut*/,
        arrayReplace: function( text, search, replace ){
            var i, j;
            if( replace === undefined)
                for( i in search )
                    search.hasOwnProperty( i ) &&
                        ( text = text.replace( i, search[ i ] ) );
            else
                for( i = 0, j = search.length; i < j; i++ )
                    text = text.replace( search[ i ], replace[ i ] );
            return text;
        },
        stripSlashes: function( text ){ // from php.js
            return (text  + '').replace(/\\(.?)/g, function (tmp, token) {
                switch (token) {
                    case '\\':
                        return '\\';
                    case '0':
                        return '\u0000';
                    case '':
                        return '';
                    default:
                        return token;
                }
            });
        },

        urlToLink: function (text) {
            var replacePattern = /(\b(https?|ftp):\/\/[-A-Z0-9+&@#\/%?=~_|!:,.;]*[-A-Z0-9+&@#\/%=~_|])/gim;
            var replacedText = text.replace(replacePattern, '<a target="_blank" href="$1" >$1</a>');

            replacePattern = /(^|[^\/])(www\.[^\s<>]+(\b|$))/gim;
            replacedText = replacedText.replace(replacePattern, '$1<a target="_blank" href="http://$2" >$2</a>');

            replacePattern = /(\w+@[a-zA-Z_]+?\.[a-zA-Z]{2,6})/gim;
            replacedText = replacedText.replace(replacePattern, '<a target="_blank" href="mailto:$1">$1</a>');

            return replacedText;
        },


        /*
        * Usage example:
            var a = 'Welcome <b g=2>to</i></i></b> <i>your</i> first task in Comindware! <br /><br />Working with Comindware Tracker is easy! Our online how-to video tutorials are a good starting point for touring Comindware Tracker. Just click the Menu tab, then click How to Videos and choose a video. <br />In addition to videos, the following resources are available to help you get started with Comindware: <br /><a href="http://dl2.comindware.com/pdf/Comindware_Tracker_User_Guide_en_A4.pdf?utm_source=tracker&utm_medium=NA&utm_term=NA&utm_content=firsttask&utm_campaign=product" target="_blank">Comindware Tracker User’s Guide</a><br /><a href="http://dl2.comindware.com/pdf/Comindware_Tracker_Administrators_Guide_en_A4.pdf?utm_source=tracker&utm_medium=NA&utm_term=NA&utm_content=firsttask&utm_campaign=product" target="_blank">Comindware Tracker Administrator’s Guide</a><br /><a href="http://kb.comindware.com/#utm_source=tracker&utm_medium=NA&utm_term=NA&utm_content=firsttask&utm_campaign=product" target="_blank">Knowledgebase at www.kb.comindware.com</a><br /><br />We hope you will enjoy working with Comindware! <br /><br /><br />Best wishes,<br /><br />The Comindware Team.<br />';


            var htmlParser = js.util.Text.HtmlParser({
                allow: {
                    tag: ['br', 'a', 'b'],
                    attr: {
                        a: [ 'href', 'target' ]
                    }
                },
                addAttr: {
                    a: {
                        lol: '77'
                    }
                }

            });
            htmlParser.sanitize(a)
        * */
        HtmlParser: (function(){


            var HtmlParser = function( cfg ){
                JS.apply(this, cfg);

                var tags = this.allow.tag || [],
                    attr = this.allow.attr || {},
                    addAttr = this.addAttr || {},
                    convert = this.convert || {};
                tags.forEach(function(el, i){ tags[i] = el.toUpperCase(); });

                this.allow.tag = JS.a2o( (this.allow.tag || []).map( JS.mapFn.toUpperCase ) );

                JS.each( attr, function( key, val ){
                    var upperKey = key.toUpperCase();
                    if( upperKey !== key ){
                        attr[ upperKey ] = JS.a2o( val );
                        delete attr[ key ];
                    }
                });
                JS.each( addAttr, function( key, val ){
                    var upperKey = key.toUpperCase();
                    if( upperKey !== key ){
                        addAttr[ upperKey ] = val;
                        delete addAttr[ key ];
                    }
                });
                JS.each( convert, function( key, val ){
                    var upperKey = key.toUpperCase();
                    if( upperKey !== key ){
                        convert[ upperKey ] = val.toUpperCase();
                        delete convert[ key ];
                    }
                });

            };
            HtmlParser.prototype = {
                allow: {
                    tag: ['br', 'a', 'b'],
                    attr: {
                        a: [ 'href', 'target' ]
                    }
                },
                singleTags: JS.a2o(['IMG', 'BR', 'HR', 'INPUT', '?XML']),
                quotes: JS.a2o(['"', '\'']),
                spaces: JS.a2o([ ' ', '\t', '\n', '\r' ]),
                _rebuildText: function( el ){
                    if( el.type !== 2 )
                        return el.text;
                    return el.data.type + el.text + el.data.type ;
                },
                tokenize: function( text ){
                    var tree = [],

                        token = '', i, l, _i,
                        tokenData = {},
                        lastTokenData = tokenData,

                        lastToken,

                        quotes = this.quotes,
                        spaces = this.spaces,

                        INTAG = false,
                        INQUOTE = false,
                        QUOTETYPE = null,
                        ESCAPED = false,

                        EXTRACT = false,

                        TOKENTYPE = 0,// 0 - TEXT, 1 - TAG, 2 - quote, 3 - space, 4 - word, 5 - equal, 6 - close tag(/), 7 - end

                        letter, lastTokenType = 0,

                        rebuildText = this._rebuildText;

                    for( i = 0, l = _i = text.length, _i = l + 1; i < _i; i++ ){

                        i === l && ( TOKENTYPE = 7 );

                        letter = text.charAt( i );

                        if( INQUOTE ){

                            if( ESCAPED ){
                                ESCAPED = false;
                            }else if( letter === '\\' ){
                                ESCAPED = true;
                            }else if( letter === QUOTETYPE && !ESCAPED ){
                                INQUOTE = false;
                                TOKENTYPE = 4;
                                letter = '';
                            }

                        }else if( INTAG ){
                            if( letter in spaces ){
                                TOKENTYPE = 3;
                            }else if( letter in quotes ){
                                TOKENTYPE = 2;
                                INQUOTE = true;
                                QUOTETYPE = letter;
                                tokenData.type = letter;
                                letter = '';
                            }else if( letter === '<' ){
                                EXTRACT = true;
                                TOKENTYPE = 1;
                            }else if( letter === '>' ){
                                TOKENTYPE = 0;
                                INTAG = false;
                                letter = '';
                            }else if( letter === '=' ){
                                TOKENTYPE = 5;
                            }else if( letter === '/' ){
                                TOKENTYPE = 6;
                            }else{
                                TOKENTYPE = 4;
                            }

                        }else if( letter === '<' ){
                            INTAG = true;
                            TOKENTYPE = 1;
                            letter = '';
                        }

                        if(lastTokenType !== TOKENTYPE || EXTRACT){
                            if( lastTokenType < 2 ){
                                if( EXTRACT ){
                                    EXTRACT = false;
                                    lastToken.type = 0;
                                    //lastToken.text = ('<' + (lastToken.sub || []).map( rebuildText ).join( '' ));
                                    token = '';
                                }
                                if( !(token === '' && lastTokenType === 0) )
                                    tree.push( lastToken = {text: token, data: lastTokenData, type: lastTokenType } );

                                if( lastTokenType === 1 ){
                                    lastToken.sub = [];
                                }

                            }else{
                                if( !(lastTokenType === 4 && token === '' ) && !(token === '' && lastTokenType === 0) )
                                    lastToken.sub.push( {text: token, data: lastTokenData, type: lastTokenType } );
                            }
                            lastTokenData = tokenData;
                            tokenData = {};

                            token = '';
                            lastTokenType = TOKENTYPE;
                        }

                        token += letter;

                    }
                    return tree;

                },
                buildDOM: function( plain ){

                    var tree = {childNodes: [], root: true},
                        pointer = tree,
                        parent,
                        parentStack = [tree],

                        singleTags = this.singleTags,
                        el,

                        attrs, tagName, closeTag, endCloseTag, properties,
                        SINGLETAG,
                        attrName,
                        equal,
                        i, _i,
                        subs,
                        sub,
                        j, _j, attr,
                        attrsLength;

                    for( i = 0, _i = plain.length; i < _i; i++ ){
                        SINGLETAG = false;
                        equal = false;
                        attrs = [];
                        el = plain[i];
                        properties = {};

                        if( el.type === 0 ){
                            pointer.childNodes.push( el.text );
                        }else{
                            subs = el.sub;

                            for( j = 0, _j = subs.length; j < _j; j++ ){

                                sub = subs[j];

                                if( sub.type !== 3 )
                                    attrs.push( sub );
                                
                            }
                            
                            if( attrsLength = attrs.length ){
                                closeTag = attrs[ 0 ].type === 6;
                                endCloseTag = attrs[ attrsLength - 1 ].type === 6;
                                tagName = closeTag ? attrs[ 1 ] : attrs[ 0 ];
                                if( tagName ){
                                    tagName = ( tagName.text || '' ).toUpperCase();

                                    if( tagName in singleTags )
                                        SINGLETAG = true;

                                    el.tagName = tagName;
                                }
                            }

                            if( closeTag ){
                                if( tagName === pointer.tagName ){
                                    pointer = parentStack.pop();
                                    parent = parentStack[ parentStack.length - 1 ];
                                }
                            }else if( !SINGLETAG && !endCloseTag ){
                                pointer.childNodes.push( el );
                                el.childNodes = [];
                                parentStack.push( parent = pointer );

                                pointer = el;

                                
                                for( j = 1, _j = attrsLength; j < _j; j++ ){

                                    attr = attrs[j];
                                    if( attr.type === 5 )
                                        equal = true;
                                    else{

                                        if( equal ){
                                            properties[attrName] = attr.text;
                                            attrName = void 0;
                                        }else{
                                            if( attrName )
                                                properties[attrName] = '';
                                            attrName = attr.text;
                                        }

                                        equal = false;
                                    }
                                }
                                
                                if( attrName )
                                    properties[attrName] = '';
                                attrName = void 0;

                                el.attr = properties;
                            }else{
                                pointer.childNodes.push( el );
                            }
                            delete el.sub;
                            delete el.text;
                            delete el.type;
                            delete el.data;
                            el.parent = parent;
                        }
                    }

                    return tree;
                },
                buildTree: function( plain ){

                    var tree = {childNodes: [], text: 'root'},
                        pointer = tree,
                        parent,
                        parentStack = [tree],

                        singleTags = this.singleTags;

                    plain.forEach(function( el ){

                        var attrs = [], tagName, closeTag, endCloseTag,

                            SINGLETAG = false;

                        if( el.type === 0 ){
                            pointer.childNodes.push( el );
                        }else{

                            el.sub.forEach(function( el ){
                                if( el.type !== 3 ){
                                    attrs.push( el );
                                }
                            });
                            if( attrs.length ){
                                closeTag = attrs[ 0 ].type === 6;
                                endCloseTag = attrs[ attrs.length - 1 ].type === 6;
                                tagName = closeTag ? attrs[ 1 ] : attrs[ 0 ];
                                if( tagName ){
                                    tagName = ( tagName.text || '' ).toUpperCase();

                                    if( tagName in singleTags )
                                        SINGLETAG = true;

                                    el.tagName = tagName;
                                }
                            }
                            if( closeTag ){
                                if( tagName === pointer.tagName ){
                                    pointer = parentStack.pop();
                                    parent = parentStack[ parentStack.length - 1 ];
                                }
                            }else if( !SINGLETAG && !endCloseTag ){
                                pointer.childNodes.push( el );
                                el.childNodes = [];
                                parentStack.push( parent = pointer );

                                pointer = el;

                            }else{
                                pointer.childNodes.push( el );
                            }
                        }
                    });

                    return tree;
                },
                sanitize: function( text ){
                    return this.flat( this.buildTree( this.tokenize( text ) ) );

                },
                buildTag: function( tag ){
                    var tagHTML = '<',
                        tagName = tag.tagName,
                        first = true,

                        allowAttr = this.allow.attr[ tagName ] || {},

                        addAttr = (this.addAttr || {})[tagName] || {},

                        convert = this.convert,

                        stateMachine = 0,
                        allowThis = true; // 0 = param, 1 = equal, 2 = value

                    // 0 - TEXT, 1 - TAG, 2 - quote, 3 - space, 4 - word, 5 - equal, 6 - close tag(/)
                    tag.sub.forEach(function( el ){
                        if( el.type === 4 ){
                            if( first ){
                                tagHTML += convert[ tagName ] || tagName;
                                first = false;
                                JS.each( addAttr, function( key, val ){
                                    var quote = val.indexOf('"') !== -1 ? '\'' : '"';
                                    tagHTML += ' ' + key + '=' + quote + val + quote;
                                });
                            }else{
                                stateMachine === 1 && (stateMachine = 2);
                                if( stateMachine === 0 )
                                    allowThis = el.text in allowAttr;

                                if( allowThis )
                                    tagHTML += el.text;

                                (stateMachine === 2) && (stateMachine = 0);
                            }
                        }else if( el.type === 2 ){
                            stateMachine === 1 && (stateMachine = 2);
                            if( stateMachine === 0 )
                                allowThis = el.text in allowAttr;

                            if( allowThis )
                                tagHTML += el.data.type + el.text + el.data.type;

                            (stateMachine === 2) && (stateMachine = 0);
                        }else if( el.type === 5 ){
                            (stateMachine === 0) && (stateMachine = 1);
                            if( allowThis )
                                tagHTML += el.text;
                        }else if( el.type === 3 ){
                            tagHTML += ' ';
                        }else if( el.type !== 6 ){
                            tagHTML += el.text;
                        }
                    });

                    tagName in this.singleTags && ( tagHTML += '/' );
                    tagHTML = tagHTML.trim();
                    tagHTML += '>';

                    return tagHTML;
                },
                flat: function( tree, wrap ){
                    var tagName = tree.tagName,
                        allowTag = this.allow.tag[ tagName ],
                        text = wrap && allowTag ? this.buildTag(tree) : '', i, _i,

                        childNodes = tree.childNodes,

                        child,
                        convert = this.convert;

                    if(childNodes){
                        for( i = 0, _i = childNodes.length; i < _i; i++ ){
                            child = childNodes[ i ];
                            if( child.type === 0 ){
                                text += child.text
                                    .replace( /</g, '&lt;' )
                                    .replace( />/g, '&gt;' );
                            }else{
                                text += this.flat( child, true );
                            }
                        }
                        wrap && allowTag && ( text += '</'+ (convert[ tagName ] || tagName) +'>' );
                    }

                    return text;
                }
            };
            js.util.H = HtmlParser;
            return function( cfg ){
                return new HtmlParser( cfg );
            };
        })()
    };
    js.util.Text.sanitizeHTML = (function(){
        /*var d = document.createElement('div');
        return function( text ){
            var child = document.createTextNode(text), out;
            d.appendChild(child);
            out = d.innerHTML;
            d.removeChild(child);
            return out;
        };*/
        var parser = js.util.Text.HtmlParser({
            allow: {
                tag: ['br', 'a', 'b', 'i', 'u', 'img', 'hr', 'irony'],
                attr: {
                    a: [ 'href' ],
                    img: [ 'src' ]
                }
            },
            addAttr: {
                a: {
                    target: '_blank'
                },
                img: {
                    'class': 'user-image'
                },
                irony: {
                    'class': 'irony'
                }
            },
            convert: {
                IRONY: 'SPAN'
            }

        });

        return JS.bind( parser, 'sanitize' );
    })();
})(Z,Z);
/* jQuery caret plugin rewritten to our framework */
(function( k, js, e, i, j, z, y ){
    js.util.Text.caret = function( f, b, l ){
        var a, c, d = k.browser.browser == 'msie';
        if( typeof b === "object" && typeof b.start === z && typeof b.end === z ){
            a = b.start;
            c = b.end
        }else if( typeof b === z && typeof l === z ){
            a = b;
            c = l
        }else if( typeof b === "string" )if( (a = f.value.indexOf( b )) > -1 )c = a + b[e];else a = null;else if( Object.prototype.toString.call( b ) === "[object RegExp]" ){
            b = b.exec( f.value );
            if( b != null ){
                a = b.index;
                c = a + b[0][e]
            }
        }
        if( typeof a != "undefined" ){
            if( d ){
                try{
                    d = f.createTextRange();
                    d.collapse( true );
                    d.moveStart( y, a );
                    d.moveEnd( y, c - a );
                    d.select()
                }catch(e){
                    //ok, ie
                }
            }else{
                f.selectionStart = a;
                f.selectionEnd = c
            }
            f.focus();
            return f;
        }else{
            if( d ){
                c = document.selection;
                if( f.tagName.toLowerCase() != "textarea" ){
                    d = f.value;
                    a = c[i]()[j]();
                    a.moveEnd( y, d[e] );
                    var g = a.text == "" ? d[e] : d.lastIndexOf( a.text );
                    a = c[i]()[j]();
                    a.moveStart( y, -d[e] );
                    var h = a.text[e]
                }else{
                    a = c[i]();
                    c = a[j]();
                    c.moveToElementText( f );
                    c.setEndPoint( "EndToEnd", a );
                    g = c.text[e] - a.text[e];
                    h = g + a.text[e]
                }
            }else{
                g =
                    f.selectionStart;
                h = f.selectionEnd
            }
            a = f.value.substring( g, h );
            return{start: g, end: h, text: a, replace: function( m ){
                return f.value.substring( 0, g ) + m + f.value.substring( h, f.value[e] )
            }}
        }
    }
})( Z, Z, "length", "createRange", "duplicate", "number", "character" );


